package fuku.eb4j.io;

import java.io.*;

import fuku.eb4j.EBException;

/**
 * ファイルおよびディレクトリの抽象表現クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class EBFile {

    /** PLAIN形式 */
    public static final int FORMAT_PLAIN = 0;
    /** EBZIP形式 */
    public static final int FORMAT_EBZIP = 1;
    /** EPWING V4/V5形式 */
    public static final int FORMAT_EPWING = 2;
    /** EPWING V6形式 */
    public static final int FORMAT_EPWING6 = 3;
    /** S-EBXA形式 */
    public static final int FORMAT_SEBXA = 4;

    /** ファイル情報 */
    private FileInfo _info = null;
    /** オリジナルファイル名 */
    private String _name = null;


    /**
     * コンストラクタ。
     *
     * @param parent 親ディレクトリ
     * @param name ファイル名
     * @param defaultFormat デフォルトのフォーマット形式
     * @exception EBException ファイルが存在しない場合
     * @see EBFile#FORMAT_PLAIN
     * @see EBFile#FORMAT_EBZIP
     * @see EBFile#FORMAT_EPWING
     * @see EBFile#FORMAT_EPWING6
     */
    public EBFile(File parent, String name,
                  int defaultFormat) throws EBException {
        super();
        _info = new FileInfo();

        String ebzName = name + ".ebz";
        String orgName = name + ".org";
        String[] list = parent.list();
        if (list != null) {
            int len = list.length;
            for (int i=0; i<len; i++) {
                File f = new File(parent, list[i]);
                if (f.isDirectory()) {
                    continue;
                }
                if (list[i].compareToIgnoreCase(name) == 0) {
                    _info.file = f;
                    _info.format = defaultFormat;
                    _name = list[i];
                    break;
                } else if (list[i].compareToIgnoreCase(orgName) == 0) {
                    _info.file = f;
                    _info.format = FORMAT_PLAIN;
                    _name = list[i].substring(0, list[i].length() - 4);
                    break;
                } else if (list[i].compareToIgnoreCase(ebzName) == 0) {
                    _info.file = f;
                    _info.format = FORMAT_EBZIP;
                    _name = list[i].substring(0, list[i].length() - 4);
                    break;
                }
            }
        }
        if (_info.file == null) {
            throw new EBException(EBException.FILE_NOT_FOUND, parent.getPath(), name);
        }
        if (!_info.file.canRead()) {
            throw new EBException(EBException.CANT_READ_FILE, _info.file.getPath());
        }
        getInputStream().initFileInfo();
    }


    /**
     * 指定された親ディレクトリ内から指定されたディレクトリを
     * 大文字/小文字の区別なく検索します。
     *
     * @param parent 親ディレクトリ
     * @param name ディレクトリ名
     * @exception EBException ファイルが存在しない場合
     */
    public static File searchDirectory(String parent, String name) throws EBException {
        return searchDirectory(new File(parent), name);
    }

    /**
     * 指定された親ディレクトリ内から指定されたディレクトリを
     * 大文字/小文字の区別なく検索します。
     *
     * @param parent 親ディレクトリ
     * @param name ディレクトリ名
     * @exception EBException ファイルが存在しない場合
     */
    public static File searchDirectory(File parent, String name) throws EBException {
        String[] list = parent.list();
        File dir = null;
        if (list != null) {
            int len = list.length;
            for (int i=0; i<len; i++) {
                File f = new File(parent, list[i]);
                if (!f.isDirectory()) {
                    continue;
                }
                if (list[i].compareToIgnoreCase(name) == 0) {
                    dir = f;
                    break;
                }
            }
        }
        if (dir == null) {
            throw new EBException(EBException.DIR_NOT_FOUND, parent.getPath(), name);
        }
        if (!dir.canRead()) {
            throw new EBException(EBException.CANT_READ_DIR, dir.getPath());
        }
        return dir;
    }

    /**
     * このオブジェクトのファイルを返します。
     *
     * @return ファイル
     */
    public File getFile() {
        return _info.file;
    }

    /**
     * このファイルのパス名を返します。
     *
     * @return ファイルのパス名
     */
    public String getPath() {
        return _info.file.getPath();
    }

    /**
     * このファイルのオリジナル名を返します。
     *
     * @return オリジナルファイル名
     */
    public String getName() {
        return _name;
    }

    /**
     * このファイルのフォーマット形式を返します。
     *
     * @return フォーマット形式
     * @see EBFile#FORMAT_PLAIN
     * @see EBFile#FORMAT_EBZIP
     * @see EBFile#FORMAT_EPWING
     * @see EBFile#FORMAT_EPWING6
     */
    public int getFormat() {
        return _info.format;
    }

    /**
     * S-EBXAの圧縮情報を設定します。
     *
     * @param index 圧縮本文データインデックス開始位置
     * @param base 圧縮本文データ開始位置
     * @param start 本文開始位置
     * @param end 本文終了位置
     */
    public void setSEBXAInfo(long index, long base, long start, long end) {
        _info.sebxaIndexPos = index;
        _info.sebxaBasePos = base;
        _info.sebxaStartPos = start;
        _info.sebxaEndPos = end;
        _info.fileSize = end;
        _info.format = FORMAT_SEBXA;
    }

    /**
     * このファイルのデータ読み込みストリームを返します。
     *
     * @return データ読み込みストリーム (ディレクトリの場合はnull)
     * @exception EBException 入出力エラーが発生した場合
     */
    public BookInputStream getInputStream() throws EBException {
        BookInputStream bis = null;
        switch (_info.format) {
            case FORMAT_EBZIP:
                bis = new EBZipInputStream(_info);
                break;
            case FORMAT_EPWING:
            case FORMAT_EPWING6:
                bis = new EPWINGInputStream(_info);
                break;
            case FORMAT_SEBXA:
                bis = new SEBXAInputStream(_info);
                break;
            case FORMAT_PLAIN:
            default:
                bis = new PlainInputStream(_info);
        }
        return bis;
    }
}

// end of EBFile.java
