package fuku.eb4j.io;

import java.io.*;
import java.util.*;

import fuku.eb4j.EBException;
import fuku.eb4j.util.ByteUtil;

/**
 * EPWING形式の書籍入力ストリームクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class EPWINGInputStream extends BookInputStream {

    /**
     * コンストラクタ。
     *
     * @param info ファイル情報
     * @exception EBException 入出力エラーが発生した場合
     */
    EPWINGInputStream(FileInfo info) throws EBException {
        super(info);
        open();
        _cache = new byte[PAGE_SIZE];
    }


    /**
     * EPWING形式のファイル情報を初期化します。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    protected void initFileInfo() throws EBException {
        try {
            _info.realFileSize = _stream.length();
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        }

        byte[] b = new byte[512];

        // ヘッダの読み込み
        int len = 32;
        if (_info.format == EBFile.FORMAT_EPWING6) {
            len += 16;
        }
        readRawFully(b, 0, len);

        _info.epwingIndexPos = ByteUtil.getLong4(b, 0);
        _info.epwingIndexSize = ByteUtil.getLong4(b, 4);
        _info.epwingFreqPos = ByteUtil.getLong4(b, 8);
        _info.epwingFreqSize = ByteUtil.getLong4(b, 12);
        if (_info.epwingIndexSize < 36 || _info.epwingFreqSize < 512) {
            throw new EBException(EBException.UNEXP_FILE, _info.file.getPath());
        }

        // ファイルサイズの取得
        long pos = _info.epwingIndexPos + (_info.epwingIndexSize - 36) / 36 * 36;
        try {
            _stream.seek(pos);
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
        }
        readRawFully(b, 0, 36);
        _info.fileSize = (_info.epwingIndexSize / 36) * (PAGE_SIZE * 16);
        for (int i=1; i<16; i++) {
            int p = i * 2 + 4;
            if (ByteUtil.getInt2(b, p) == 0) {
                _info.fileSize = _info.fileSize - PAGE_SIZE * (16 - i);
                break;
            }
        }

        int leaf32 = 0;
        int leaf16 = 0;
        if (_info.format == EBFile.FORMAT_EPWING) {
            leaf16 = (int)((_info.epwingFreqSize - (256 * 2)) / 4);
        } else {
            leaf16 = 0x400;
            leaf32 = (int)((_info.epwingFreqSize - (leaf16 * 4) - (256 * 2)) / 6);
        }

        ArrayList list = null;
        // 32bitデータのハフマンノード作成
        if (_info.format == EBFile.FORMAT_EPWING6) {
            list = new ArrayList(leaf32 + leaf16 + 256 + 1);
            len = b.length - (b.length % 6);
            try {
                _stream.seek(_info.epwingFreqPos);
            } catch (IOException e) {
                throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
            }
            readRawFully(b, 0, len);
            for (int i=0, off=0; i<leaf32; i++, off+=6) {
                if (off >= len) {
                    readRawFully(b, 0, len);
                    off = 0;
                }
                long value = ByteUtil.getLong4(b, off);
                int freq = ByteUtil.getInt2(b, off+4);
                list.add(new HuffmanNode(value, freq, HuffmanNode.LEAF_32));
            }
        } else {
            list = new ArrayList(leaf16 + 256 + 1);
        }

        // 16bitデータのハフマンノード作成
        len = b.length - (b.length % 4);
        try {
            _stream.seek(_info.epwingFreqPos + leaf32 * 6);
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
        }
        readRawFully(b, 0, len);
        for (int i=0, off=0; i<leaf16; i++, off+=4) {
            if (off >= b.length) {
                readRawFully(b, 0, len);
                off = 0;
            }
            long value = ByteUtil.getInt2(b, off);
            int freq = ByteUtil.getInt2(b, off+2);
            list.add(new HuffmanNode(value, freq, HuffmanNode.LEAF_16));
        }

        // 8bitデータのハフマンノード作成
        try {
            _stream.seek(_info.epwingFreqPos + leaf32 * 6 + leaf16 * 4);
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
        }
        readRawFully(b, 0, b.length);
        for (int i=0, off=0; i<256; i++, off+=2) {
            int freq = ByteUtil.getInt2(b, off);
            list.add(new HuffmanNode(i, freq, HuffmanNode.LEAF_8));
        }

        // EOFデータのハフマンノード作成
        list.add(new HuffmanNode(256, 1, HuffmanNode.LEAF_EOF));

        // ハフマンツリーの作成
        _info.epwingRootNode = HuffmanNode.makeTree(list);

        super.initFileInfo();
    }

    /**
     * EPWING形式のファイルから最大b.lengthバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    public int read(byte[] b, int off, int len) throws EBException {
        int rlen = 0;
        while (rlen < len) {
            if (_info.fileSize <= _filePos) {
                if (rlen == 0) {
                    return -1;
                } else {
                    return rlen;
                }
            }
            // キャッシュの作成
            if (_cachePos < 0
                || _filePos < _cachePos
                || _cachePos + PAGE_SIZE <= _filePos) {
                _cachePos = _filePos - (_filePos % PAGE_SIZE);

                // インデックスの読み込み
                long pos = _info.epwingIndexPos + _filePos / (PAGE_SIZE * 16) * 36;
                try {
                    _stream.seek(pos);
                } catch (IOException e) {
                    throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
                }
                byte[] buf = new byte[36];
                readRawFully(buf, 0, buf.length);

                // ページ位置の取得
                int offset = (int)(4 + (_filePos / PAGE_SIZE % 16) * 2);
                long pagePos = (ByteUtil.getLong4(buf, 0)
                                + ByteUtil.getInt2(buf, offset));

                // 圧縮ページをデコードしてキャッシュに読み込む
                try {
                    _stream.seek(pagePos);
                } catch (IOException e) {
                    throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
                }
                _decode();
            }

            // キャッシュからデータの取得
            int n = (int)(PAGE_SIZE - (_filePos % PAGE_SIZE));
            if (len - rlen < n) {
                n = len - rlen;
            }
            if (_info.fileSize - _filePos < n) {
                n = (int)(_info.fileSize - _filePos);
            }
            int p = (int)(_filePos - _cachePos);
            System.arraycopy(_cache, p, b, off+rlen, n);
            rlen += n;
            _filePos += n;
        }
        return rlen;
    }

    /**
     * 復号化します。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    private void _decode() throws EBException {
        byte[] b = new byte[PAGE_SIZE];
        int inPos = 0;
        int inLen = 0;
        int outPos = 0;
        int outLen = 0;
        int bitIndex = 7;

        if (_info.format == EBFile.FORMAT_EPWING6) {
            // 圧縮形式の取得
            readRawFully(b, 0, 1);
            if ((b[0] & 0xff) != 0) {
                // 無圧縮なのでそのまま読み込む
                readRawFully(_cache, 0, PAGE_SIZE);
                return;
            }
        }

        while (outLen < PAGE_SIZE) {
            HuffmanNode node = _info.epwingRootNode;
            while (!node.isLeaf()) {
                // データがなければ次を読み込む
                if (inLen <= inPos) {
                    inLen = readRaw(b, 0, b.length);
                    if (inLen <=0) {
                        throw new EBException(EBException.UNEXP_FILE, _info.file.getPath());
                    }
                    inPos = 0;
                }
                int bit = (b[inPos] >>> bitIndex) & 0x01;
                if (bit == 1) {
                    node = node.getLeft();
                } else {
                    node = node.getRight();
                }
                if (node == null) {
                    throw new EBException(EBException.UNEXP_FILE, _info.file.getPath());
                }

                if (bitIndex > 0) {
                    bitIndex--;
                } else {
                    bitIndex = 7;
                    inPos++;
                }
            }

            if (node.getLeafType() == HuffmanNode.LEAF_EOF) {
                // 残りを埋める
                if (outLen < PAGE_SIZE) {
                    Arrays.fill(_cache, outPos, _cache.length, (byte)'\0');
                    outLen = PAGE_SIZE;
                }
                break;
            } else if (node.getLeafType() == HuffmanNode.LEAF_32) {
                if (outLen >= PAGE_SIZE - 1) {
                    _cache[outPos] = (byte)((node.getValue() >>> 24) & 0xff);
                    outPos++;
                    outLen++;
                } else if (outLen >= PAGE_SIZE - 2) {
                    _cache[outPos] = (byte)((node.getValue() >>> 24) & 0xff);
                    _cache[outPos+1] = (byte)((node.getValue() >>> 16) & 0xff);
                    outPos += 2;
                    outLen += 2;
                } else if (outLen >= PAGE_SIZE - 3) {
                    _cache[outPos] = (byte)((node.getValue() >>> 24) & 0xff);
                    _cache[outPos+1] = (byte)((node.getValue() >>> 16) & 0xff);
                    _cache[outPos+2] = (byte)((node.getValue() >>> 8) & 0xff);
                    outPos += 3;
                    outLen += 3;
                } else {
                    _cache[outPos] = (byte)((node.getValue() >>> 24) & 0xff);
                    _cache[outPos+1] = (byte)((node.getValue() >>> 16) & 0xff);
                    _cache[outPos+2] = (byte)((node.getValue() >>> 8) & 0xff);
                    _cache[outPos+3] = (byte)(node.getValue() & 0xff);
                    outPos += 4;
                    outLen += 4;
                }
            } else if (node.getLeafType() == HuffmanNode.LEAF_16) {
                if (outLen >= PAGE_SIZE - 1) {
                    _cache[outPos] = (byte)((node.getValue() >>> 8) & 0xff);
                    outPos++;
                    outLen++;
                } else {
                    _cache[outPos] = (byte)((node.getValue() >>> 8) & 0xff);
                    _cache[outPos+1] = (byte)(node.getValue() & 0xff);
                    outPos += 2;
                    outLen += 2;
                }
            } else {
                _cache[outPos] = (byte)(node.getValue() & 0xff);
                outPos++;
                outLen++;
            }
        }
    }
}

// end of EPWINGInputStream.java
