package fuku.eb4j.io;

import java.io.*;

import fuku.eb4j.EBException;

/**
 * 無圧縮形式の書籍入力ストリームクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class PlainInputStream extends BookInputStream {

    /**
     * コンストラクタ。
     *
     * @param info ファイル情報
     * @exception EBException 入出力エラーが発生した場合
     */
    PlainInputStream(FileInfo info) throws EBException {
        super(info);
        open();
        _cache = new byte[PAGE_SIZE];
    }


    /**
     * EPWING形式のファイル情報を初期化します。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    protected void initFileInfo() throws EBException {
        try {
            _info.realFileSize = _stream.length();
            _info.fileSize = _info.realFileSize;
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        }
        super.initFileInfo();
    }

    /**
     * このファイルから最大lenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    public int read(byte[] b, int off, int len) throws EBException {
        int rlen = 0;
        while (rlen < len) {
            if (_info.fileSize <= _filePos) {
                // ストリームの終わり
                if (rlen == 0) {
                    // データを読み込んでいなければ-1
                    return -1;
                } else {
                    // データを読み込んでいればバイト数を返す
                    return rlen;
                }
            }
            // キャッシュの作成
            if (_cachePos < 0
                || _filePos < _cachePos
                || _cachePos + PAGE_SIZE <= _filePos) {
                // キャッシュのデータ位置
                // _filePosの位置が含まれるページの先頭位置
                _cachePos = _filePos - (_filePos % PAGE_SIZE);

                // ページのデータをキャッシュに読み込む
                try {
                    _stream.seek(_cachePos);
                } catch (IOException e) {
                    throw new EBException(EBException.FAILED_SEEK_FILE, _info.file.getPath(), e);
                }
                int readLen = PAGE_SIZE;
                if (_info.fileSize < _cachePos + PAGE_SIZE) {
                    readLen = (int)(_info.fileSize - _cachePos);
                }
                readRawFully(_cache, 0, readLen);
            }

            // キャッシュからデータの取得
            int cacheLen = PAGE_SIZE;
            if (_info.fileSize < _cachePos + PAGE_SIZE) {
                cacheLen = (int)(_info.fileSize - _cachePos);
            }
            int rest = (int)(cacheLen - (_filePos % PAGE_SIZE));
            if (len - rlen < rest) {
                rest = len - rlen;
            }
            if (_info.fileSize - _filePos < rest) {
                rest = (int)(_info.fileSize - _filePos);
            }
            int p = (int)(_filePos % PAGE_SIZE);
            System.arraycopy(_cache, p, b, off+rlen, rest);
            rlen += rest;
            _filePos += rest;
        }
        return rlen;
    }
}

// end of PlainInputStream.java
