package fuku.eb4j.tool;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

import fuku.eb4j.Book;
import fuku.eb4j.SubBook;
import fuku.eb4j.Version;
import fuku.eb4j.EBException;
import fuku.eb4j.io.BookInputStream;
import fuku.eb4j.util.ByteUtil;

/**
 * 書籍データダンププログラム。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class EBDump {

    /** プロブラム名 */
    private static final String _PROGRAM = "fuku.eb4j.tool.EBDump";

    /** デフォルト読み込みディレクトリ */
    private static final String DEFAULT_BOOK_DIR = ".";

    /** コマンドラインオプション */
    private static final LongOpt[] LONGOPT = {
        new LongOpt("subbook", LongOpt.REQUIRED_ARGUMENT, null, 's'),
        new LongOpt("page", LongOpt.REQUIRED_ARGUMENT, null, 'p'),
        new LongOpt("offset", LongOpt.REQUIRED_ARGUMENT, null, 'o'),
        new LongOpt("position", LongOpt.REQUIRED_ARGUMENT, null, 'P'),
        new LongOpt("dump", LongOpt.REQUIRED_ARGUMENT, null, 'd'),
        new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
        new LongOpt("version", LongOpt.NO_ARGUMENT, null, 'v'),
    };

    /** 書籍 */
    private Book _book = null;


    /**
     * メインメソッド。
     *
     * @param args コマンド行引数
     */
    public static void main(String[] args) {
        Getopt g = new Getopt(_PROGRAM, args, "s:p:o:P:d:hv", LONGOPT);
        int c;
        int subindex = 0;
        long page = 1L;
        int off = 0;
        long pos = -1L;
        int size = 0;
        String arg = null;
        while ((c=g.getopt()) != -1) {
            switch (c) {
                case 's':
                    arg = g.getOptarg();
                    try {
                        subindex = Integer.parseInt(arg);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid subbook index `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    if (subindex <= 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid subbook index `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    subindex--;
                    break;
                case 'p':
                    arg = g.getOptarg();
                    try {
                        page = Long.parseLong(arg, 16);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid page number `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    if (page <= 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid page number `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    break;
                case 'o':
                    arg = g.getOptarg();
                    try {
                        off = Integer.parseInt(arg, 16);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid offset number `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    if (off < 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid offset number `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    break;
                case 'P':
                    arg = g.getOptarg();
                    try {
                        pos = Long.parseLong(arg, 16);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid position `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    if (pos < 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid position `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    break;
                case 'd':
                    arg = g.getOptarg();
                    try {
                        size = Integer.parseInt(arg, 16);
                    } catch (NumberFormatException e) {
                        System.err.println(_PROGRAM
                                           + ": invalid dump size `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    if (size <= 0) {
                        System.err.println(_PROGRAM
                                           + ": invalid dump size `"
                                           + arg + "'");
                        System.exit(1);
                    }
                    break;
                case 'h':
                    _usage(0);
                    break;
                case 'v':
                    _version();
                    break;
                default:
                    _usage(1);
            }
        }

        String path = null;
        int idx = g.getOptind();
        if (idx+1 == args.length) {
            path = args[idx];
        } else if (idx+1 > args.length) {
            path = DEFAULT_BOOK_DIR;
        } else {
            System.err.println(_PROGRAM + ": too many arguments");
            _usage(1);
        }

        try {
            EBDump ebdump = new EBDump(path);
            if (pos < 0) {
                pos = BookInputStream.getPosition(page, off);
            }
            ebdump._dump(subindex, pos, size);
        } catch (EBException e) {
            System.err.println(_PROGRAM + ": " + e.getMessage());
        }
    }


    /**
     * 使用方法を表示します。
     *
     * @param status 終了ステータス
     */
    private static void _usage(int status) {
        if (status != 0) {
            System.out.println("Try `java " + _PROGRAM + " --help' for more information");
        } else {
            System.out.println("Usage: java " + _PROGRAM + " [option...] [book-directory]");
            System.out.println("");
            System.out.println("Options:");
            System.out.println("  -s INTEGER  --subbook INTEGER");
            System.out.println("                             subbook index number");
            System.out.println("  -p LONG  --page LONG");
            System.out.println("                             page number (hex)");
            System.out.println("  -o INTEGER  --offset INTEGER");
            System.out.println("                             offset number (hex)");
            System.out.println("  -P LONG  --position LONG");
            System.out.println("                             position (hex)");
            System.out.println("  -d INTEGER  --dump INTEGER");
            System.out.println("                             dump size (hex)");
            System.out.println("  -h  --help                 display this help and exit");
            System.out.println("  -v  --version              output version information and exit");
            System.out.println("");
            System.out.println("Argument:");
            System.out.println("  book-directory             top directory of a book");
            System.out.println("                             (default: " + DEFAULT_BOOK_DIR + ")");
            System.out.println("");
            System.out.println("Report bugs to <" + Version.EMAIL + ">.");
        }
        System.exit(status);
    }

    /**
     * バージョンを表示します。
     *
     */
    private static void _version() {
        System.out.println(_PROGRAM + " " + Version.VERSION);
        System.out.println(Version.COPYRIGHT);
        System.out.println("All right reserved.");
        System.exit(0);
    }


    /**
     * コンストラクタ。
     *
     * @param path 書籍のパス
     * @exception EBException 書籍の初期化中に例外が発生した場合
     */
    private EBDump(String path) throws EBException {
        super();
        _book = new Book(path);
    }


    /**
     * 書籍のデータを出力します。
     *
     * @param subindex 副本のインデックス
     * @param pos データ位置
     * @param size ダンプサイズ
     * @exception EBException ファイル読み込み中にエラーが発生した場合
     */
    private void _dump(int subindex, long pos, int size) throws EBException {
        SubBook sub = _book.getSubBook(subindex);
        if (sub == null) {
            return;
        }
        if (size <= 0) {
            size = BookInputStream.PAGE_SIZE;
        }

        BookInputStream bis = sub.getTextFile().getInputStream();
        byte[] b = new byte[size];
        try {
            bis.seek(pos);
            bis.readFully(b, 0, b.length);
        } finally {
            bis.close();
        }

        long page = BookInputStream.getPage(pos);
        long pos2 = pos + size;
        long start = pos - (pos & 0x0f);
        long end = pos2;
        if ((end & 0x0f) > 0) {
            end = end + (16 - (end % 16));
        }

        StringBuffer buf = new StringBuffer();
        int idx = 0;
        long i = 0L;
        int j, k;
        int offset, high, low;
        for (i=start; i<end; i+=16) {
            if (pos + idx >= page * BookInputStream.PAGE_SIZE) {
                page++;
            }
            buf.append(_toHexString(page)).append(':');
            offset = (int)(i%BookInputStream.PAGE_SIZE);
            buf.append(_toHexString(offset)).append(' ');
            k = 0;
            for (j=0; j<16; j++) {
                if (j == 8) {
                    buf.append(' ');
                }
                buf.append(' ');
                if (i+j >= pos && i+j < pos2) {
                    buf.append(_toHexString(b[idx+k]));
                    k++;
                } else {
                    buf.append("  ");
                }
            }
            buf.append("  ");
            for (j=0; j<16; j+=2) {
                if (i+j >= pos && i+j < pos2) {
                    high = b[idx++] & 0xff;
                    if (i+j+1 >= pos && i+j+1 < pos2) {
                        low = b[idx++] & 0xff;
                        if (high > 0x20 && high < 0x7f
                            && low > 0x20 && low < 0x7f) {
                            // JIS X 0208
                            buf.append(ByteUtil.jisx0208ToString(b, idx-2, 2));
                        } else if (high > 0x20 && high > 0x7f
                                   && low > 0xa0 && low < 0xff) {
                            // GB 2312
                            buf.append("??");
                        } else if (high > 0xa0 && high < 0xff
                                   && low > 0x20 && low < 0x7f) {
                            // 外字
                            buf.append("??");
                        } else {
                            buf.append("..");
                        }
                    } else {
                        buf.append(". ");
                    }
                } else {
                    buf.append(' ');
                    if (i+j+1 >= pos && i+j+1 < pos2) {
                        idx++;
                        buf.append('.');
                    } else {
                        buf.append(' ');
                    }
                }
            }
            System.out.println(buf.toString());
            System.out.flush();
            buf.delete(0, buf.length());
        }
    }

    /**
     * 指定されたbyte値を16進数表現に変換ます。
     *
     * @param hex byte値
     * @return 変換後の文字列
     */
    private String _toHexString(byte hex) {
        return _toHexString(Integer.toHexString(hex&0xff), 2);
    }

    /**
     * 指定されたint値を16進数表現に変換ます。
     *
     * @param hex int値
     * @return 変換後の文字列
     */
    private String _toHexString(int hex) {
        return _toHexString(Integer.toHexString(hex), 3);
    }

    /**
     * 指定されたlong値を16進数表現に変換ます。
     *
     * @param hex long値
     * @return 変換後の文字列
     */
    private String _toHexString(long hex) {
        return _toHexString(Long.toHexString(hex), 5);
    }

    /**
     * 指定された文字列を16進数表現に変換ます。
     *
     * @param str 文字列
     * @param length 桁数
     * @return 変換後の文字列
     */
    private String _toHexString(String str, int length) {
        StringBuffer buf = new StringBuffer(str.toUpperCase());
        int len = str.length();
        if (len < length) {
            len = length - len;
        } else {
            len = 0;
        }
        for (int i=0; i<len; i++) {
            buf.insert(0, '0');
        }
        return buf.toString();
    }
}

// end of EBDump.java
