package fuku.webbook;

import java.awt.*;
import java.io.*;
import java.util.*;

import fuku.eb4j.SubBook;
import fuku.eb4j.hook.HookAdapter;
import fuku.eb4j.util.ByteUtil;

/**
 * HTML用エスケープシーケンス加工クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class HTMLHook extends HookAdapter {

    /** オーディオプレイヤアプレット */
    private static final String AUDIO_PLAYER = "fuku.player.AudioPlayerApplet";
    /** ビデオプレイヤアプレット */
    private static final String VIDEO_PLAYER = "fuku.player.VideoPlayerApplet";
    /** アプレットのJARファイル */
    private static final String ARCHIVE = "player.jar";
    /** アプレットの幅 (オーディオ) */
    private static final String AUDIO_WIDTH = "32";
    /** アプレットの高さ (オーディオ) */
    private static final String AUDIO_HEIGHT = "16";

    /** 最大入力行数 */
    private static final int MAX_LINE = 500;

    /** 半角表示が開始されているかどうかを示すフラグ */
    private boolean _narrow = false;
    /** インデント量 */
    private int _indent = 0;
    /** ネスト用スタック */
    private Stack _stack = new Stack();

    /** 行数 */
    private int _line = 0;

    /** デフォルトの前景色 */
    private Color _default = Color.BLACK;
    /** キーワード表示の前景色 */
    private Color _keyword = Color.RED;
    /** アンカー表示の前景色 */
    private Color _anchor = Color.BLUE;

    /** 現在の前景色 */
    private Color _foreground = _default;
    /** 現在の背景色 */
    private Color _background = Color.WHITE;

    /** 副本 */
    private SubBook _sub = null;

    /** HTMLデータ */
    private StringBuffer _html = new StringBuffer(2048);
    /** 文字列バッファ */
    private StringBuffer _buf = new StringBuffer(256);
    /** 出力バッファ */
    private StringBuffer _output = _html;

    /** HTMLバッファ内の半角開始位置 */
    private int _position = 0;

    /** 参照時のURL */
    private String _href = null;
    /** バイナリデータファイル出力先のURL */
    private String _base = "./";
    /** URNリダイレクタのURL */
    private String _urn = null;

    /** 画像のインライン表示 */
    private boolean _inline = false;
    /** アプレットの使用 */
    private boolean _applet = false;

    /** モノクロ画像/動画の幅 */
    private int _width = -1;
    /** モノクロ画像/動画の高さ */
    private int _height = -1;
    /** カラー画像の位置 */
    private long _imgPos = -1L;
    /** カラー画像形式 */
    private int _imgFormat = -1;
    /** 文字修飾種別 */
    private int _decoration = 0;

    /** アプレットパラメータ */
    private StringBuffer _file = new StringBuffer(128);

    /** 見出しフラグ */
    private boolean _heading = false;


    /**
     * コンストラクタ。
     *
     * @param sub 副本
     * @param href 参照時のURL
     * @param base バイナリデータのベースURL
     * @param urn URNリダイレクタのURL
     */
    public HTMLHook(SubBook sub, String href, String base, String  urn) {
        super();
        _sub = sub;
        _href = href;
        _urn = urn;

        if (base == null) {
            _base = "./";
        } else {
            if (!base.endsWith("/")) {
                _base = base + "/";
            } else {
                _base = base;
            }
        }
    }

    /**
     * 画像のインライン表示を設定します。
     *
     * @param inline 画像をインラインで表示する場合はture、そうでない場合はfalse
     */
    public void setInline(boolean inline) {
        _inline = inline;
    }

    /**
     * アプレットの使用を設定します。
     *
     * @param applet アプレットを使用する場合はture、そうでない場合はfalse
     */
    public void setApplet(boolean applet) {
        _applet = applet;
    }

    /**
     * 見出しフラグを設定します。
     *
     * @param heading 見出しの場合はture、そうでない場合はfalse
     */
    public void setHeading(boolean heading) {
        _heading = heading;
    }

    /**
     * デフォルトの前景色を設定します。
     *
     * @param color 設定する色
     */
    public void setForegroundColor(Color color) {
        if (color != null) {
            if (_foreground.equals(_default)) {
                _foreground = color;
            }
            _default = color;
        }
    }

    /**
     * 背景色を設定します。
     *
     * @param color 設定する色
     */
    public void setBackgroundColor(Color color) {
        if (color != null) {
            _background = color;
        }
    }

    /**
     * キーワード開始時の前景色を設定します。
     *
     * @param color 設定する色
     */
    public void setKeywordColor(Color color) {
        if (color != null) {
            if (_foreground.equals(_keyword)) {
                _foreground = color;
            }
            _keyword = color;
        }
    }

    /**
     * 参照開始時の前景色を設定します。
     *
     * @param color 設定する色
     */
    public void setAnchorColor(Color color) {
        if (color != null) {
            if (_foreground.equals(_anchor)) {
                _foreground = color;
            }
            _anchor = color;
        }
    }

    /**
     * すべての入力をクリアし、初期化します。
     *
     */
    public void clear() {
        _html.delete(0, _html.length());
        _buf.delete(0, _buf.length());
        _narrow = false;
        _indent = 0;
        _line = 0;
    }

    /**
     * フックによって加工されたオブジェクトを返します。
     *
     * @return HTML文字列
     */
    public Object getObject() {
        if (!_stack.empty()) {
            _html.append("</DD></DL>");
            _indent = ((Integer)_stack.pop()).intValue();
            while (!_stack.empty()) {
                _html.append("</DD></DL>");
                _indent = ((Integer)_stack.pop()).intValue();
            }
        }
        return _html.toString();
    }

    /**
     * 次の入力が可能かどうかを返します。
     *
     * @return まだ入力を受けつける場合はtrue、そうでない場合はfalse
     */
    public boolean isMoreInput() {
        if (_line >= MAX_LINE) {
            return false;
        }
        return true;
    }

    /**
     * 文字列を追加します。
     *
     * @param str 文字列
     */
    public void append(String str) {
        if (_narrow) {
            str = ByteUtil.wideToNarrow(str);
        }
        _output.append(HTMLUtil.encode(str));
    }

    /**
     * 外字を追加します。
     *
     * @param code 外字の文字コード
     */
    public void append(int code) {
        // 出力ファイル名
        int fore = _foreground.getRGB() | 0xff000000;
        int back = _background.getRGB() | 0xff000000;
        String height = Integer.toString(_sub.getFont().getFontHeight());
        String width = null;
        String prefix = null;
        if (_narrow) {
            prefix = "N";
            width = Integer.toString(_sub.getFont().getNarrowFontWidth());
        } else {
            prefix = "W";
            width = Integer.toString(_sub.getFont().getWideFontWidth());
        }
        String alt = prefix + height
            + "-" + Integer.toHexString(code).toUpperCase();

        // HTMLタグの追加
        _output.append("<IMG class=\"gaiji\" src=\"");
        _output.append(_base).append(alt);
        _output.append("_F-");
        _output.append(Integer.toHexString(fore).substring(2).toUpperCase());
        _output.append("_B-");
        _output.append(Integer.toHexString(back).substring(2).toUpperCase());
        _output.append(".png\"");
        _output.append(" width=\"").append(width).append("\"");
        _output.append(" height=\"").append(height).append("\"");
        _output.append(" alt=\"[").append(alt).append("]\">");
    }

    /**
     * 半角表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginNarrow() {
        _narrow = true;
        _position = _html.length();
    }

    /**
     * 半角表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endNarrow() {
        _narrow = false;
        _addLink();
    }

    /**
     * 下付き表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginSubscript() {
        _output.append("<SUB>");
    }

    /**
     * 下付き表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endSubscript() {
        _output.append("</SUB>");
    }

    /**
     * 上付き表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginSuperscript() {
        _output.append("<SUP>");
    }

    /**
     * 上付き表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endSuperscript() {
        _output.append("</SUP>");
    }

    /**
     * 行頭の字下げ指定を表すエスケープシーケンスに対するフックです。
     *
     * @param indent 字下げ量
     */
    public void setIndent(int indent) {
        if (indent < 0) {
            return;
        }
        if (_output.length() <= 0) {
            _indent = indent;
            return;
        }
        if (indent > _indent) {
            _stack.push(new Integer(_indent));
            _indent = indent;
            _output.append("<DL><DT></DT><DD>");
        } else {
            while (indent < _indent) {
                if (_stack.empty()) {
                    break;
                }
                _output.append("</DD></DL>");
                _indent = ((Integer)_stack.pop()).intValue();
            }
        }
    }

    /**
     * 改行を表すエスケープシーケンスに対するフックです。
     *
     */
    public void newLine() {
        _output.append("<BR>");
        _line++;
    }

    /**
     * 改行禁止の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginNoNewLine() {
        _output.append("<NOBR>");
    }

    /**
     * 改行禁止の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endNoNewLine() {
        _output.append("</NOBR>");
    }

    /**
     * 強調表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginEmphasis() {
        _output.append("<EM>");
    }

    /**
     * 強調表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endEmphasis() {
        _output.append("</EM>");
    }

    /**
     * 文字修飾の開始を表すエスケープシーケンスに対するフックです。
     *
     * @param type 文字修飾種別
     * @see #BOLD
     * @see #ITALIC
     */
    public void beginDecoration(int type) {
        _decoration = type;
        switch (_decoration) {
            case BOLD:
                _output.append("<I>");
                break;
            case ITALIC:
                _output.append("<B>");
                break;
            default:
                _output.append("<U>");
                break;
        }
    }

    /**
     * 文字修飾の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endDecoration() {
        switch (_decoration) {
            case 1:
                _output.append("</I>");
                break;
            case 3:
                _output.append("</B>");
                break;
            default:
                _output.append("</U>");
                break;
        }
    }

    /**
     * 複合検索の候補となる語の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginCandidate() {
        _buf.delete(0, _buf.length());
        _output = _buf;
    }

    /**
     * 複合検索の候補となる語の終了を表すエスケープシーケンスに対するフックです。<BR>
     * 候補となる語はさらに細かい選択肢に分かれていることを示します。
     *
     * @param pos 次の階層の候補一覧データの位置
     */
    public void endCandidateGroup(long pos) {
        _output = _html;
        if (_href != null) {
            _output.append("<A href=\"").append(_href);
            _output.append("&pos=").append(Long.toHexString(pos));
            _output.append("\">").append(_buf).append("</A>");
        }
    }

    /**
     * 複合検索の候補となる語の終了を表すエスケープシーケンスに対するフックです。<BR>
     * 候補となる語が実際に検索の入力語として使えるものであることを示します。
     *
     */
    public void endCandidateLeaf() {
        _output = _html;
        _output.append(_buf);
    }

    /**
     * 別位置のテキストデータの参照開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginReference() {
        _foreground = _anchor;
        _buf.delete(0, _buf.length());
        _output = _buf;
    }

    /**
     * 別位置のテキストデータの参照終了を表すエスケープシーケンスに対するフックです。
     *
     * @param pos 参照先の位置
     */
    public void endReference(long pos) {
        _output = _html;
        if (_href != null) {
            _output.append("<A href=\"").append(_href);
            _output.append("&pos=").append(Long.toHexString(pos));
            _output.append("\">").append(_buf).append("</A>");
        }
        _foreground = _default;
    }

    /**
     * キーワード表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginKeyword() {
        _foreground = _keyword;
        _output.append("<SPAN class=\"keyword\">");
    }

    /**
     * キーワード表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endKeyword() {
        _foreground = _default;
        _output.append("</SPAN>");
    }

    /**
     * モノクロ画像の参照開始を表すエスケープシーケンスに対するフックです。
     *
     * @param width 画像の幅
     * @param height 画像の高さ
     */
    public void beginMonoGraphic(int width, int height) {
        _width = width;
        _height = height;
        _buf.delete(0, _buf.length());
        _output = _buf;
    }

    /**
     * モノクロ画像の参照終了を表すエスケープシーケンスに対するフックです。
     *
     * @param pos 画像データの位置
     */
    public void endMonoGraphic(long pos) {
        _output = _html;
        if (_width >= 0 && _height >= 0 && !_heading) {
            int len = _output.length();
            if (len >= 4 && _output.lastIndexOf("<BR>") == len-4) {
                _output.append("<BR>");
            }
            if (_inline || _buf.length() <= 0) {
                boolean flag = false;
                if (_buf.length() > 0) {
                    _output.append(_buf).append("<BR>");
                    flag = true;
                }
                int idx = _buf.indexOf("<BR>");
                while (idx >= 0) {
                    _buf.replace(idx, idx+4, " ");
                    idx = _buf.indexOf("<BR>", idx);
                }
                _output.append("<IMG src=\"");
                _output.append(_base).append("M-");
                _output.append(Long.toHexString(pos).toUpperCase());
                _output.append("_W-");
                _output.append(Integer.toHexString(_width).toUpperCase());
                _output.append("_H-");
                _output.append(Integer.toHexString(_height).toUpperCase());
                _output.append(".png\"");
                _output.append(" width=\"").append(Integer.toString(_width)).append("\"");
                _output.append(" height=\"").append(Integer.toString(_height)).append("\"");
                _output.append(" alt=\"").append(_buf).append("\">");
                if (flag) {
                    _output.append("<BR>");
                }
            } else {
                _output.append("<A href=\"");
                _output.append(_base).append("M-");
                _output.append(Long.toHexString(pos).toUpperCase());
                _output.append("_W-");
                _output.append(Integer.toHexString(_width).toUpperCase());
                _output.append("_H-");
                _output.append(Integer.toHexString(_height).toUpperCase());
                _output.append(".png\">").append(_buf).append("</A>");
            }
        }
    }

    /**
     * インラインカラー画像の参照開始を表すエスケープシーケンスに対するフックです。
     *
     * @param format 画像形式
     * @param pos 画像データの位置
     * @see #DIB
     * @see #JPEG
     */
    public void beginInlineColorGraphic(int format, long pos) {
        _imgFormat = format;
        _imgPos = pos;
        _buf.delete(0, _buf.length());
        _output = _buf;
    }

    /**
     * インラインカラー画像の参照終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endInlineColorGraphic() {
        _output = _html;
        if (_imgPos >= 0 && !_heading) {
            String suffix = null;
            if (_imgFormat == JPEG) {
                suffix = ".jpeg";
            } else {
                suffix = ".png";
            }
            int len = _output.length();
            if (len >= 4 && _output.lastIndexOf("<BR>") == len-4) {
                _output.append("<BR>");
            }
            boolean flag = false;
            if (_buf.length() > 0) {
                _output.append(_buf).append("<BR>");
                flag = true;
            }
            int idx = _buf.indexOf("<BR>");
            while (idx >= 0) {
                _buf.replace(idx, idx+4, " ");
                idx = _buf.indexOf("<BR>", idx);
            }
            _output.append("<IMG src=\"");
            _output.append(_base).append("C-");
            _output.append(Long.toHexString(_imgPos).toUpperCase());
            _output.append(suffix).append("\"");
            _output.append(" alt=\"").append(_buf).append("\">");
            if (flag) {
                _output.append("<BR>");
            }
        }
    }

    /**
     * カラー画像の参照開始を表すエスケープシーケンスに対するフックです。
     *
     * @param format 画像形式
     * @param pos 画像データの位置
     */
    public void beginColorGraphic(int format, long pos) {
        _imgFormat = format;
        _imgPos = pos;
        _buf.delete(0, _buf.length());
        _output = _buf;
    }

    /**
     * カラー画像の参照終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endColorGraphic() {
        _output = _html;
        if (_imgPos >= 0 && !_heading) {
            String suffix = null;
            if (_imgFormat == JPEG) {
                suffix = ".jpeg";
            } else {
                suffix = ".png";
            }
            int len = _output.length();
            if (len >= 4 && _output.lastIndexOf("<BR>") == len-4) {
                _output.append("<BR>");
            }
            if (_inline || _buf.length() <= 0) {
                boolean flag = false;
                if (_buf.length() > 0) {
                    _output.append(_buf).append("<BR>");
                    flag = true;
                }
                int idx = _buf.indexOf("<BR>");
                while (idx >= 0) {
                    _buf.replace(idx, idx+4, " ");
                    idx = _buf.indexOf("<BR>", idx);
                }
                _output.append("<IMG src=\"");
                _output.append(_base).append("C-");
                _output.append(Long.toHexString(_imgPos).toUpperCase());
                _output.append(suffix).append("\"");
                _output.append(" alt=\"").append(_buf).append("\">");
                if (flag) {
                    _output.append("<BR>");
                }
            } else {
                _output.append("<A href=\"");
                _output.append(_base).append("C-");
                _output.append(Long.toHexString(_imgPos).toUpperCase());
                _output.append(suffix).append("\">").append(_buf).append("</A>");
            }
        }
    }

    /**
     * 音声の開始を表すエスケープシーケンスに対するフックです。
     *
     * @param format 音声形式
     * @param start 音声データの開始位置
     * @param end 音声データの終了位置
     * @see #WAVE
     * @see #MIDI
     */
    public void beginSound(int format, long start, long end) {
        _foreground = _anchor;
        _file.delete(0, _file.length());
        _file.append(_base).append("S-");
        _file.append(Long.toHexString(start).toUpperCase());
        _file.append("_E-");
        _file.append(Long.toHexString(end).toUpperCase());
        if (format == MIDI) {
            _file.append(".mid");
        } else {
            _file.append(".wav");
        }

        // HTMLタグの追加
        _output.append("<A href=\"").append(_file).append("\">");
    }

    /**
     * 音声の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endSound() {
        _output.append("</A>");
        if (_applet) {
            _output.append("<APPLET");
            _output.append(" code=\"").append(AUDIO_PLAYER).append("\"");
            _output.append(" archive=\"").append(ARCHIVE).append("\"");
            _output.append(" align=\"absbottom\"");
            _output.append(" width=\"").append(AUDIO_WIDTH).append("\"");
            _output.append(" height=\"").append(AUDIO_HEIGHT).append("\">");
            _output.append("<PARAM name=\"file\"");
            _output.append(" value=\"").append(_file).append("\">");
            _output.append("</APPLET>");
        }
        _foreground = _default;
    }

    /**
     * 動画の開始を表すエスケープシーケンスに対するフックです。
     *
     * @param format 動画形式
     * @param width 動画の幅
     * @param height 動画の高さ
     * @param filename 動画ファイル名
     */
    public void beginMovie(int format, int width, int height, String filename) {
        File mpeg = _sub.getMovieFile(filename);
        if (mpeg == null) {
            return;
        }
        _width = width;
        if (_width <= 0) {
            _width = 320;
        }
        _height = height;
        if (_height <= 0) {
            _height = 240;
        }
        _foreground = _anchor;

        _file.delete(0, _file.length());
        _file.append(_base).append(mpeg.getName()).append(".mpeg");

        // HTMLタグの追加
        _output.append("<A href=\"").append(_file).append("\">");
    }

    /**
     * 動画の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endMovie() {
        if (_foreground.equals(_anchor)) {
            _output.append("</A>");
            if (_applet) {
                _output.append("<BR><APPLET");
                _output.append(" code=\"").append(VIDEO_PLAYER).append("\"");
                _output.append(" archive=\"").append(ARCHIVE).append("\"");
                _output.append(" width=\"").append(Integer.toString(_width)).append("\"");
                _output.append(" height=\"").append(Integer.toString(_height+20)).append("\">");
                _output.append("<PARAM name=\"file\"");
                _output.append(" value=\"").append(_file).append("\">");
                _output.append("</APPLET><BR>");
            }
            _foreground = _default;
        }
    }

    /**
     * ハイパーリンクを追加します。
     *
     */
    private void _addLink() {
        /*
         *   name@host        ->  mailto:name@host
         *   method://string  ->  method://string
         *   www.host.name    ->  http://www.host.name
         *   ftp.host.name    ->  ftp://ftp.host.name
         *   urn:string       ->  _urn + urn:string
         */
        int[] pos = new int[5];
        pos[0] = _html.indexOf("@", _position+1);
        pos[1] = _html.indexOf("://", _position+1);
        pos[2] = _html.indexOf("www.", _position);
        pos[3] = _html.indexOf("ftp.", _position);
        pos[4] = _html.indexOf("urn:", _position);
        int index = -1;
        for (int i=0; i<pos.length; i++) {
            if (pos[i] >= 0 && (index < 0 || pos[i] < pos[index])) {
                index = i;
            }
        }
        int len, idx1, idx2, off1, off2, mark;
        char ch;
        String anchor = null;
        StringBuffer href = new StringBuffer();
        while (index >= 0) {
            href.delete(0, href.length());
            idx1 = -1;
            idx2 = -1;
            off1 = 0;
            off2 = 0;
            switch (index) {
                case 0: // mailto
                    href.append("@");
                    off1 = 1;
                    off2 = 1;
                    idx1 = pos[index];
                    while (idx1 > 0) { // 前方
                        ch = _html.charAt(idx1-1);
                        if ((ch >= 'a' && ch <= 'z')
                            || (ch >= 'A' && ch <= 'Z')
                            || (ch >= '0' && ch <= '9')
                            || ch == '!' || ch == '%' || ch == '+'
                            || ch == '-' || ch == '.' || ch == '_') {
                            idx1--;
                            href.insert(0, ch);
                        } else {
                            break;
                        }
                    }
                    idx2 = pos[index] + off2;
                    len = _html.length();
                    mark = 0;
                    while (idx2 < len) { // 後方
                        ch = _html.charAt(idx2);
                        if ((ch >= 'a' && ch <= 'z')
                            || (ch >= 'A' && ch <= 'Z')
                            || (ch >= '0' && ch <= '9')
                            || ch == '+' || ch == '-' || ch == '.' || ch == '_') {
                            idx2++;
                            href.append(ch);
                            if (ch == '.') {
                                mark++;
                            }
                        } else {
                            break;
                        }
                    }
                    if (mark > 0) {
                        href.insert(0, "mailto:");
                    } else {
                        idx1 = -1;
                    }
                    break;
                case 1: // url
                    href.append("://");
                    off1 = 1;
                    off2 = 3;
                    idx1 = pos[index];
                    while (idx1 > 0) { // 前方
                        ch = _html.charAt(idx1-1);
                        if (ch < 'a' || ch > 'z') {
                            break;
                        }
                        idx1--;
                        href.insert(0, ch);
                    }
                    idx2 = pos[index] + off2;
                    len = _html.length();
                    while (idx2 < len) { // 後方
                        ch = _html.charAt(idx2);
                        if ((ch >= 'a' && ch <= 'z')
                            || (ch >= 'A' && ch <= 'Z')
                            || (ch >= '0' && ch <= '9')
                            || ch == '#' || ch == '%' || ch == '(' || ch == ')'
                            || ch == '+' || ch == ',' || ch == '-' || ch == '.'
                            || ch == '/' || ch == ':' || ch == '=' || ch == '?'
                            || ch == '@' || ch == '_' || ch == '~') {
                            idx2++;
                            href.append(ch);
                        } else if (ch == '&'
                                   && _html.indexOf("&amp;", idx2) == idx2) {
                            idx2 += 5;
                            href.append(ch);
                        } else {
                            break;
                        }
                    }
                    break;
                case 2: // www
                case 3: // ftp
                case 4: // urn
                    if (index == 2) {
                        href.append("http://www.");
                    } else if (index == 3) {
                        href.append("ftp://ftp.");
                    } else {
                        href.append(_urn).append("urn:");
                    }
                    off1 = 0;
                    off2 = 4;
                    idx1 = pos[index];
                    if (idx1 > 0) {
                        ch = _html.charAt(idx1-1);
                        if ((ch >= 'a' && ch <= 'z')
                            || (ch >= 'A' && ch <= 'Z')
                            || (ch >= '0' && ch <= '9')
                            || ch == '#' || ch == '%' || ch == '(' || ch == ')'
                            || ch == '+' || ch == ',' || ch == '-' || ch == '.'
                            || ch == '/' || ch == ':' || ch == '=' || ch == '?'
                            || ch == '@' || ch == '_' || ch == '~') {
                            idx1 = -1;
                        }
                    }
                    idx2 = pos[index] + off2;
                    len = _html.length();
                    mark = 0;
                    while (idx2 < len) { // 後方
                        ch = _html.charAt(idx2);
                        if ((ch >= 'a' && ch <= 'z')
                            || (ch >= 'A' && ch <= 'Z')
                            || (ch >= '0' && ch <= '9')
                            || ch == '#' || ch == '%' || ch == '(' || ch == ')'
                            || ch == '+' || ch == ',' || ch == '-' || ch == '.'
                            || ch == '/' || ch == ':' || ch == '=' || ch == '?'
                            || ch == '@' || ch == '_' || ch == '~') {
                            idx2++;
                            href.append(ch);
                            if (ch == ':') {
                                mark++;
                            }
                        } else if (ch == '&'
                                   && _html.indexOf("&amp;", idx2) == idx2) {
                            idx2 += 5;
                            href.append(ch);
                        } else {
                            break;
                        }
                    }
                    if (index == 4 && mark == 0) {
                        // urn:xxx:xxx の形式でないので無視
                        idx1 = -1;
                    }
                    break;
                default:
                    break;
            }
            if (idx1 >= 0 && idx2 >= 0
                && idx1 <= pos[index]-off1 && idx2 > pos[index]+off2) {
                if (index > 0) {
                    anchor = "<A href=\"" + href.toString() + "\" target=\"_top\">";
                } else {
                    anchor = "<A href=\"" + href.toString() + "\">";
                }
                _html.insert(idx1, anchor);
                idx2 += anchor.length();
                _html.insert(idx2, "</A>");
                idx2 += 4;
            }
            pos[0] = _html.indexOf("@", idx2+1);
            pos[1] = _html.indexOf("://", idx2+1);
            pos[2] = _html.indexOf("www.", idx2);
            pos[3] = _html.indexOf("ftp.", idx2);
            pos[4] = _html.indexOf("urn:", idx2);
            index = -1;
            for (int i=0; i<pos.length; i++) {
                if (pos[i] >= 0 && (index < 0 || pos[i] < pos[index])) {
                    index = i;
                }
            }
        }
    }
}

// end of HTMLHook.java
