package fuku.webbook;

import fuku.eb4j.SubBook;
import fuku.eb4j.EBException;
import fuku.eb4j.hook.HookAdapter;
import fuku.eb4j.util.ByteUtil;

/**
 * 候補一覧リストアップ用エスケープシーケンス加工クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class ListupHook extends HookAdapter {

    /** 半角表示が開始されているかどうかを示すフラグ */
    private boolean _narrow = false;
    /** 文字の追加を無視するフラグ */
    private boolean _ignore = true;

    /** HTMLデータ */
    private StringBuffer _html = new StringBuffer(512);
    /** 文字列バッファ */
    private StringBuffer _buf = new StringBuffer(32);

    /** 副本 */
    private SubBook _sub = null;


    /**
     * コンストラクタ。
     *
     * @param sub 副本
     */
    public ListupHook(SubBook sub) {
        super();
        _sub = sub;
    }


    /**
     * すべての入力をクリアし、初期化します。
     *
     */
    public void clear() {
        _html.delete(0, _html.length());
        _buf.delete(0, _buf.length());
        _narrow = false;
        _ignore = true;
    }

    /**
     * フックによって加工されたオブジェクトを返します。
     *
     * @return 文字列オブジェクト
     */
    public Object getObject() {
        return _html.toString();
    }

    /**
     * 次の入力が可能かどうかを返します。
     *
     * @return まだ入力を受けつける場合はtrue、そうでない場合はfalse
     */
    public boolean isMoreInput() {
        return true;
    }

    /**
     * 文字列を追加します。
     *
     * @param str 文字列
     */
    public void append(String str) {
        if (_ignore) {
            return;
        }
        if (_narrow) {
            str = ByteUtil.wideToNarrow(str);
        }
        _buf.append(HTMLUtil.encode(str));
    }

    /**
     * 外字を追加します。<BR>
     * "[N-####]", "[W-####]"に変換して追加します。
     *
     * @param code 外字の文字コード
     */
    public void append(int code) {
        String str = null;
        if (_narrow) {
            str = "[N-" + Integer.toHexString(code).toUpperCase() + "]";
        } else {
            str = "[W-" + Integer.toHexString(code).toUpperCase() + "]";
        }
        _buf.append(str);
    }

    /**
     * 半角表示の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginNarrow() {
        _narrow = true;
    }

    /**
     * 半角表示の終了を表すエスケープシーケンスに対するフックです。
     *
     */
    public void endNarrow() {
        _narrow = false;
    }

    /**
     * 複合検索の候補となる語の開始を表すエスケープシーケンスに対するフックです。
     *
     */
    public void beginCandidate() {
        _ignore = false;
    }

    /**
     * 複合検索の候補となる語の終了を表すエスケープシーケンスに対するフックです。<BR>
     * 候補となる語はさらに細かい選択肢に分かれていることを示します。
     *
     * @param pos 次の階層の候補一覧データの位置
     */
    public void endCandidateGroup(long pos) {
        String str = null;
        try {
            str = (String)_sub.getText(pos, new ListupHook(_sub));
        } catch (EBException e) {
            System.err.println("WebBook: " + e.getMessage());
        }
        if (str != null) {
            _html.append("<OPTGROUP label=\"").append(_buf).append("\">");
            _html.append(str).append("</OPTGROUP>");
        }
        _buf.delete(0, _buf.length());
        _ignore = true;
    }

    /**
     * 複合検索の候補となる語の終了を表すエスケープシーケンスに対するフックです。<BR>
     * 候補となる語が実際に検索の入力語として使えるものであることを示します。
     *
     */
    public void endCandidateLeaf() {
        _html.append("<OPTION value=\"").append(_buf).append("\">");
        _html.append(_buf).append("</OPTION>");
        _buf.delete(0, _buf.length());
        _ignore = true;
    }
}

// end of ListupHook.java
