package fuku.webbook;

import java.awt.Color;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.servlet.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import org.xml.sax.helpers.*;

/**
 * WebBook設定ファイル読み込みクラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public final class WebBookConfig {

    /** 外字の前景色のインデックス */
    private static final int FOREGROUND = 0;
    /** 外字の背景色のインデックス */
    private static final int BACKGROUND = 1;
    /** 外字のキーワード表示色のインデックス */
    private static final int KEYWORD = 2;
    /** 外字の参照表示色のインデックス */
    private static final int ANCHOR = 3;

    /** 外字のデフォルト表示色定義 */
    private static final Color[] DEF_COLOR = {
        Color.BLACK, Color.WHITE, Color.GREEN, Color.BLUE
    };

    /** 外字の表示色リスト */
    private Color[] _colorList = null;

    /** 書籍リスト */
    private String[] _bookList = null;
    /** 付録リスト */
    private String[] _appendixList = null;
    /** ホストリスト */
    private Hosts[] _hostsList = null;

    /** 外字キャッシュ */
    private static final int CACHE_GAIJI = 0;
    /** 画像キャッシュ */
    private static final int CACHE_IMAGE = 1;
    /** 音声キャッシュ */
    private static final int CACHE_SOUND = 2;
    /** キャッシュの有無 */
    private boolean[] _cache = new boolean[3];
    /** キャッシュディレクトリ */
    private File _cacheDir = null;
    /** URNリダイレクタのURL */
    private String _urn = null;


    /**
     * コンストラクタ。
     *
     * @param url 設定ファイルのURL
     * @param context サーブレットコンテキスト
     * @exception FactoryConfigurationError パーサファクトリの実装が使用できないかインスタンス化できない場合
     * @exception IllegalArgumentException パーサファクトリが"http://java.sun.com/xml/jaxp/properties/schemaLanguage"属性を認識しない場合
     * @exception ParserConfigurationException DocumentBuilderを生成できない場合
     * @exception SAXException 構文解析エラーが発生した場合
     * @exception IOException 入出力エターが発生した場合
     */
    WebBookConfig(URL url, ServletContext context)
        throws ParserConfigurationException, SAXException, IOException {
        super();

        ArrayList list = new ArrayList(10);
        InputStream stream = null;
        try {
            stream = url.openStream();
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setIgnoringComments(true);
            factory.setIgnoringElementContentWhitespace(true);
            factory.setValidating(true);
            factory.setNamespaceAware(true);
            factory.setAttribute("http://java.sun.com/xml/jaxp/properties/schemaLanguage",
                                 "http://www.w3.org/2001/XMLSchema");
            DocumentBuilder builder = factory.newDocumentBuilder();
            builder.setErrorHandler(new NullHandler(context));
            String systemID = url.toString();
            String file = new File(url.getFile()).getName();
            int idx = systemID.indexOf(file);
            if (idx > 0) {
                systemID = systemID.substring(0, idx);
            }
            Document doc = builder.parse(stream, systemID);
            Node root = doc.getFirstChild();

            NamedNodeMap nmap;
            NodeList nlist;
            Node node1, node2;
            String tag;
            int i, j, num;

            NodeList nodes = root.getChildNodes();
            int len = nodes.getLength();
            for (i=0; i<len; i++) {
                node1 = nodes.item(i);
                if (node1.getNodeType() != Node.ELEMENT_NODE) {
                    continue;
                }
                if (node1.getNodeName().equals("book")) {
                    // 書籍定義の読み込み
                    nlist = node1.getChildNodes();
                    num = nlist.getLength();
                    String[] val = new String[4];
                    for (j=0; j<num; j++) {
                        node2 = nlist.item(j);
                        if (node2.getNodeType() != Node.ELEMENT_NODE) {
                            continue;
                        }
                        tag = node2.getNodeName();
                        idx = -1;
                        if (tag.equals("data")) {
                            if (node2.hasAttributes()) {
                                nmap = node2.getAttributes();
                                node2 = nmap.getNamedItem("path");
                                val[0] = node2.getNodeValue();
                            }
                        } else if (tag.equals("appendix")) {
                            if (node2.hasAttributes()) {
                                nmap = node2.getAttributes();
                                node2 = nmap.getNamedItem("path");
                                val[1] = node2.getNodeValue();
                            }
                        } else if (tag.equals("hosts")) {
                            if (node2.hasAttributes()) {
                                nmap = node2.getAttributes();
                                node2 = nmap.getNamedItem("type");
                                val[2] = node2.getNodeValue();
                                node2 = nmap.getNamedItem("list");
                                val[3] = node2.getNodeValue();
                            }
                        }
                    }
                    if (val[0] != null) {
                        list.add(val);
                    }
                } else if (node1.getNodeName().equals("gaiji")) {
                    // 外字の表示色の読み込み
                    _colorList = new Color[DEF_COLOR.length];
                    nlist = node1.getChildNodes();
                    num = nlist.getLength();
                    for (j=0; j<num; j++) {
                        node2 = nlist.item(j);
                        if (node2.getNodeType() != Node.ELEMENT_NODE) {
                            continue;
                        }
                        tag = node2.getNodeName();
                        idx = -1;
                        if (tag.equals("foreground")) {
                            idx = FOREGROUND;
                        } else if (tag.equals("background")) {
                            idx = BACKGROUND;
                        } else if (tag.equals("keyword")) {
                            idx = KEYWORD;
                        } else if (tag.equals("anchor")) {
                            idx = ANCHOR;
                        }
                        if (idx >= 0 && node2.hasAttributes()) {
                            nmap = node2.getAttributes();
                            node2 = nmap.getNamedItem("color");
                            try {
                                String val = node2.getNodeValue();
                                _colorList[idx] = Color.decode(val);
                            } catch (NumberFormatException e) {
                            }
                        }
                    }
                } else if (node1.getNodeName().equals("cache")) {
                    // キャッシュの設定
                    if (node1.hasAttributes()) {
                        nmap = node1.getAttributes();
                        num = nmap.getLength();
                        for (j=0; j<num; j++) {
                            node2 = nmap.item(j);
                            tag = node2.getNodeName();
                            idx = -1;
                            if (tag.equals("gaiji")) {
                                idx = CACHE_GAIJI;
                            } else if (tag.equals("image")) {
                                idx = CACHE_IMAGE;
                            } else if (tag.equals("sound")) {
                                idx = CACHE_SOUND;
                            }
                            if (idx >= 0) {
                                String val = node2.getNodeValue();
                                _cache[idx] = StringUtil.toBoolean(val);
                            }
                        }
                    }
                    nlist = node1.getChildNodes();
                    num = nlist.getLength();
                    for (j=0; j<num; j++) {
                        node2 = nlist.item(j);
                        if (node2.getNodeType() != Node.ELEMENT_NODE) {
                            continue;
                        }
                        tag = node2.getNodeName();
                        if (tag.equals("dir") && node2.hasAttributes()) {
                            nmap = node2.getAttributes();
                            node2 = nmap.getNamedItem("path");
                            String val = node2.getNodeValue();
                            _cacheDir = new File(val, "webbook");
                        }
                    }
                } else if (node1.getNodeName().equals("redirect")) {
                    nlist = node1.getChildNodes();
                    num = nlist.getLength();
                    for (j=0; j<num; j++) {
                        node2 = nlist.item(j);
                        if (node2.getNodeType() != Node.ELEMENT_NODE) {
                            continue;
                        }
                        tag = node2.getNodeName();
                        if (tag.equals("urn") && node2.hasAttributes()) {
                            nmap = node2.getAttributes();
                            node2 = nmap.getNamedItem("url");
                            _urn = node2.getNodeValue();
                        }
                    }
                }
            }
        } finally {
            if (stream != null) {
                stream.close();
                stream = null;
            }
        }

        if (!list.isEmpty()) {
            // 書籍リストの初期化
            int len = list.size();
            _bookList = new String[len];
            _appendixList = new String[len];
            _hostsList = new Hosts[len];
            for (int i=0; i<len; i++) {
                String[] item = (String[])list.get(i);
                _bookList[i] = item[0];
                _appendixList[i] = item[1];
                if (item[2] != null && item[2].equals("allow")) {
                    _hostsList[i] = new Hosts(true, item[3]);
                } else {
                    _hostsList[i] = new Hosts(false, item[3]);
                }
            }
        }

        if (_cacheDir == null) {
            _cacheDir =
                new File(System.getProperty("java.io.tmpdir"), "webbook");
        }
    }


    /**
     * 書籍リストを返します。
     *
     * @return 書籍リスト (なければnull)
     */
    public String[] getBookList() {
        return _bookList;
    }

    /**
     * 付録リストを返します。
     *
     * @return 付録リスト (なければnull)
     */
    public String[] getAppendixList() {
        return _appendixList;
    }

    /**
     * 指定されたパスの書籍へのアクセスが許可されているかどうかを判定します。
     *
     * @param host ホスト名またはIPアドレス
     * @param path 書籍のパス
     * @return 許可されている場合はtrue、そうでない場合はfalse
     */
    public boolean isAllowed(String host, String path) {
        for (int i=0; i<_bookList.length; i++) {
            if (path.equals(_bookList[i])) {
                return _hostsList[i].isAllowed(host);
            }
        }
        return false;
    }

    /**
     * 外字キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isGaijiCache() {
        return _cache[CACHE_GAIJI];
    }

    /**
     * 画像キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isImageCache() {
        return _cache[CACHE_IMAGE];
    }

    /**
     * 音声キャッシュの有無を返します。
     *
     * @return キャッシュの有無
     */
    public boolean isSoundCache() {
        return _cache[CACHE_SOUND];
    }

    /**
     * キャッシュディレクトリを返します。
     *
     * @return キャッシュディレクトリ
     */
    public File getCacheDirectory() {
        return _cacheDir;
    }

    /**
     * 外字の前景色を返します。
     *
     * @return 外字の前景色
     */
    public Color getForegroundColor() {
        if (_colorList == null || _colorList[FOREGROUND] == null) {
            return DEF_COLOR[FOREGROUND];
        }
        return _colorList[FOREGROUND];
    }

    /**
     * 外字の背景色を返します。
     *
     * @return 外字の背景色
     */
    public Color getBackgroundColor() {
        if (_colorList == null || _colorList[BACKGROUND] == null) {
            return DEF_COLOR[BACKGROUND];
        }
        return _colorList[BACKGROUND];
    }

    /**
     * 外字のキーワード表示色を返します。
     *
     * @return 外字のキーワード表示色
     */
    public Color getKeywordColor() {
        if (_colorList == null || _colorList[KEYWORD] == null) {
            return DEF_COLOR[KEYWORD];
        }
        return _colorList[KEYWORD];
    }

    /**
     * 外字のアンカー表示色を返します。
     *
     * @return 外字のアンカー表示色
     */
    public Color getAnchorColor() {
        if (_colorList == null || _colorList[ANCHOR] == null) {
            return DEF_COLOR[ANCHOR];
        }
        return _colorList[ANCHOR];
    }

    /**
     * URNリダイレクタのURLを返します。
     *
     * @return URNリダイレクタのURL
     */
    public String getURNRedirectURL() {
        if (_urn == null) {
            return "";
        }
        return _urn;
    }


    /**
     * エラーハンドラ。
     *
     */
    private class NullHandler extends DefaultHandler {

        /** サーブレットコンテキスト */
        private ServletContext _context = null;


        /**
         * コンストラクタ。
         *
         * @param context サーブレットコンテキスト
         */
        public NullHandler(ServletContext context) {
            super();
            _context = context;
        }


        /**
         * 回復可能なエラーの通知を受け取ります。
         *
         * @param exp SAX構文解析例外にカプセル化されたエラー情報
         * @exception SAXException SAX例外
         */
        public void error(SAXParseException exp) throws SAXException {
            _context.log("[ERROR " + exp.getLineNumber()
                         + ":" + exp.getColumnNumber() + "] "
                         + exp.getMessage());
        }

        /**
         * 警告の通知を受け取ります。
         *
         * @param exp SAX構文解析例外にカプセル化されたエラー情報
         * @exception SAXException SAX例外
         */
        public void warning(SAXParseException exp) throws SAXException {
            _context.log("[WARNING " + exp.getLineNumber()
                         + ":" + exp.getColumnNumber() + "] "
                         + exp.getMessage());
        }

        /**
         * 回復できないエラーの通知を受け取ります。
         *
         * @param exp SAX構文解析例外にカプセル化されたエラー情報
         * @exception SAXException SAX例外
         */
        public void fatal(SAXParseException exp) throws SAXException {
            _context.log("[FATAL " + exp.getLineNumber()
                         + ":" + exp.getColumnNumber() + "] "
                         + exp.getMessage());
        }
    }
}

// end of WebBookConfig.java
