/*
 * @(#)RestServlet.java
 *
 */
package elazyrest.core.servlet;


import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.net.InetAddress;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.bind.annotation.XmlType;


import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import elazyrest.core.annotation.Aspect;
import elazyrest.core.annotation.RestMethod;
import elazyrest.core.annotation.RestParam;
import elazyrest.core.annotation.SimpleXml;
import elazyrest.core.aspect.AspectBase;
import elazyrest.core.aspect.AspectChain;
import elazyrest.core.exception.AuthenticationErrorException;
import elazyrest.core.exception.AuthorizationRequiredException;
import elazyrest.core.exception.DataEmptyException;
import elazyrest.core.exception.DuplicationException;
import elazyrest.core.exception.ForbiddenException;
import elazyrest.core.exception.InternalServerErrorException;
import elazyrest.core.exception.InvalidParameterException;
import elazyrest.core.exception.NotSupportedMethodException;
import elazyrest.core.exception.NotSupportedOperationException;
import elazyrest.core.exception.ResponseCodeException;
import elazyrest.core.provider.InstanceProvider;
import elazyrest.core.provider.SingletonProvider;
import elazyrest.core.provider.ThreadLocalProvider;
import elazyrest.core.util.AppContext;
import elazyrest.core.util.JaxbUtil;
import elazyrest.core.util.ResponseGen;
import elazyrest.core.validator.RestParamErrors;
import elazyrest.core.validator.RestParamValidator;
import elazyrest.core.validator.RestParamValidatorException;
import generator.DocGen;
import generator.ResourceGen;

/**
 * 
 * @author kaz
 *
 */
public class RestServlet extends HttpServlet {

	@Override
	public void init(ServletConfig sc) throws ServletException {
		if (sc.getInitParameter("requestEncoding") != null) {
			this.requestEncoding = sc.getInitParameter("requestEncoding");
		}
		if  (sc.getInitParameter("responseEncoding") != null) {
			this.responseEncoding = sc.getInitParameter("responseEncoding");
		}
		this.rootPackage = sc.getInitParameter("package");
		//t@X쐬
		String docGenStr = sc.getInitParameter("docgen");
		if  (docGenStr == null || docGenStr.equals("true")) {
			try {
				String docRoot = sc.getServletContext().getRealPath("");
				String contextPath = docRoot.substring(docRoot.lastIndexOf(System.getProperty("file.separator")) +1);
				
				InetAddress inet = InetAddress.getLocalHost();
				String endPoint = "http://" +inet.getHostAddress() +":8080/" +contextPath +"/" +sc.getServletName();
				DocGen docGen = new DocGen(this.rootPackage, docRoot +"/services", endPoint);
				docGen.generate();
				ResourceGen resourceGen = new ResourceGen();
				resourceGen.generate(this.rootPackage, docRoot +"/../src/main/resources/");
			}
			catch (Exception e) {
				e.printStackTrace();
				//
			}
		}
		
		super.init(sc);
	}

	private static Log log = LogFactory.getLog(RestServlet.class);

	// default UTF-8
	private String requestEncoding = "UTF-8";
	// default UTF-8
	private String responseEncoding = "UTF-8";
	private String rootPackage = null;

	protected InstanceProvider instanceProvider = new InstanceProvider();
	protected SingletonProvider singletonProvider = new SingletonProvider();

	@Override
	public void destroy() {
		instanceProvider.clear();
		singletonProvider.clear();
		ThreadLocalProvider.clear();
		rootPackage = null;
		super.destroy();
	}

	/**
	 * 
	 */
	private static final long serialVersionUID = 6807245574600933798L;

	protected void service(HttpServletRequest req, HttpServletResponse res) throws ServletException, IOException {
		// request charset
        req.setCharacterEncoding(requestEncoding);
		// response charset
		res.setCharacterEncoding(responseEncoding);
		//res.setContentType("text/xml");

		String requestMethod = req.getMethod();
		
		AppContext appContext = new AppContext();
		appContext.setRequest(req);
		appContext.setResponse(res);
		AppContext.setCurrentContext(appContext);
		boolean isBinary = false;
		String responseType = null;
		RestParamErrors errors = new RestParamErrors(); 

		Object resultData;
		try {
			responseType = appContext.getParameter("response_type");
    		String method = (String)appContext.getParameter("method");
    		if (method == null || method.length() <= 0) {
    			res.sendRedirect("services/index.html");
    			return;
    		}
			int lastIndex = method.lastIndexOf(".");

			String className = rootPackage != null && rootPackage.length() > 0?
							   rootPackage +"." +method.substring(0, lastIndex):
							   method.substring(0, lastIndex);

			String methodName = method.substring(lastIndex+1);
			
			Class<?> cl = Class.forName(className);
			
			// Is SimpleXml?
			if (!cl.isAnnotationPresent(SimpleXml.class)) {
				//log.error(cl.getName() +"is not SimpleXml Class.");
				throw new NotSupportedMethodException(cl.getName() +"is not SimpleXml Class.");
			}

			Method[] ms = cl.getMethods();
			Method targetMethod = null;
			for(Method m: ms) {
				if (m.getName().equals(methodName)) {
					targetMethod = m;
					break;
				}
			}

			// Is RestMethod?
			if (targetMethod == null || 
				!targetMethod.isAnnotationPresent(RestMethod.class)) {
				//log.debug(methodName +" is not RestMethod.");
				throw new NotSupportedMethodException(methodName +" is not RestMethod.");
			}
			// contentType
			String contentType = targetMethod.getAnnotation(RestMethod.class).contentType();
			res.setContentType(contentType);
			// charset
			String charset = targetMethod.getAnnotation(RestMethod.class).responseEncoding();
			if (charset.length() > 0) {
				res.setCharacterEncoding(charset);
			}

			
			String[] allowMethods = targetMethod.getAnnotation(RestMethod.class).value();
			boolean isAccessible = false;
			// Is Accessible Method?
			for (String allowMethod: allowMethods) {
				if (allowMethod.equals(requestMethod)) {
					isAccessible = true;
					break;
				}
			}
			
			if (!isAccessible) {
				throw new NotSupportedOperationException();
			}
			Class[] paramTypes = targetMethod.getParameterTypes();
			Object[] params = new Object[paramTypes.length];

			Annotation[][] paramAnnotation = targetMethod.getParameterAnnotations();
			
			// request parameter to rest method parameter
			for (int i = 0; i < paramAnnotation.length; i++) {
				for (int j = 0; j < paramAnnotation[i].length; j++) {
					if (paramAnnotation[i][j].annotationType() == RestParam.class) {
						RestParam restParam = (RestParam)paramAnnotation[i][j];
						String key = restParam.name();
						String[] sufixs = restParam.suffix();
						String[] val = null;
						if (sufixs != null && sufixs.length > 0 && sufixs[0] != null && sufixs[0].length() > 0) {
							val = new String[sufixs.length];
							for (int si = 0; si < sufixs.length; si++) {
								val[si] = appContext.getParameter(key +sufixs[si]);
							}
						}
						else {
							val = appContext.getParameters(key);
							if (val == null ||
								(val.length == 1 && (val[0] == null || val[0].length() == 0))) {
								// defaultlݒ
								if (restParam.def().length() > 0) {
									val = new String[1];
									val[0] = restParam.def();
								}
								
							}
						}
						// multipart
						if (appContext.isMultipart() && restParam.binary()) {
							params[i] = appContext.getBytes(key);
						}
						// convert type
						else if (val == null) {
							params[i] = null;
						}
						// for simple
						else if (val.length == 1) {
							params[i] = val[0] != null && val[0].length() > 0?
										ConvertUtils.convert(val[0], paramTypes[i]):
										null;
						}
						// for array
						else if (val.length > 1){
							params[i] = ConvertUtils.convert(val, paramTypes[i]);
						}
						// validate
						RestParamValidator.validate(method, params[i], restParam, errors);
						break;
					}
				}
			}
			if (!errors.isEmpty()) {
				throw new RestParamValidatorException(errors);
			} else {
	    		boolean validateOnly = Boolean.parseBoolean(appContext.getParameter("validate"));
	    		if(validateOnly) {
					// validatê
					resultData = ResponseGen.generateXML(ResponseGen.SUCCESS, null);
	    		} else {
	    			Object restService = singletonProvider.get(cl);
	    			
	    			//// AspectChain setting
	    			AspectChain chain = new AspectChain();
	    			chain.setMainMethod(targetMethod, restService, params);
	    			
	    			// class Annotation
	    			Annotation[] as = cl.getAnnotations();
	    			for (Annotation a: as) {
//	    				if (a.getClass().isAnnotationPresent(Aspect.class)) {
//	    					Class<? extends AspectBase> aspectCl = a.getClass().getAnnotation(Aspect.class).value();
//	    					AspectBase aspect = (AspectBase)instanceProvider.get(aspectCl);
//	    					chain.addAdvice(aspect, a);
//	    				}
	    				Class aType = a.annotationType();
	    				if (aType != null && aType.isAnnotationPresent(Aspect.class)) {
	    					Aspect aspectType = (Aspect)aType.getAnnotation(Aspect.class);
	    					Class<? extends AspectBase> aspectCl = aspectType.value();
	    					AspectBase aspect = (AspectBase)instanceProvider.get(aspectCl);
	    					chain.addAdvice(aspect, a);
	    				}

	    			}
	    			// method Annotation
	    			Annotation[] methodAs = targetMethod.getAnnotations();
	    			for (Annotation methodA: methodAs) {
	    				Class aType = methodA.annotationType();
	    				if (aType != null && aType.isAnnotationPresent(Aspect.class)) {
	    					Aspect aspectType = (Aspect)aType.getAnnotation(Aspect.class);
	    					Class<? extends AspectBase> aspectCl = aspectType.value();
	    					AspectBase aspect = (AspectBase)instanceProvider.get(aspectCl);
	    					chain.addAdvice(aspect, methodA);
	    				}
	    			}

	    			// AdviceChain invoke
	    			resultData = chain.invoke();
	    			if (resultData == null) {
	    				resultData = "";
	    			}
	    			if (!appContext.isBinary() && !resultData.getClass().isAnnotationPresent(XmlType.class) && (res.getContentType() != null && !res.getContentType().equals("text/html")))  {
	    				resultData = ResponseGen.generateXML(ResponseGen.SUCCESS, null, resultData.toString());
	    			}
	    		}
			}

		}
		catch (NotSupportedMethodException e) {
			resultData = ResponseGen.generateXML(ResponseGen.NOT_SUPPORTED_METHOD, e.getMessage(), null, errors);
//			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (NotSupportedOperationException e) {
			resultData = ResponseGen.generateXML(ResponseGen.NOT_SUPPORTED_OPERATION, e.getMessage(), null, errors);
//			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (ResponseCodeException e) {
			log.warn(e.toString());
			res.sendError(e.getResponseCode());
			return ;
		}
		catch (RestParamValidatorException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INVALID_PARAMETER, e.getMessage(), null, errors);
//			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (InvalidParameterException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INVALID_PARAMETER, "p[^G[");
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (InternalServerErrorException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INTERNAL_SERVER_ERROR, "T[oG[");
			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (DataEmptyException e) {
			resultData = ResponseGen.generateXML(ResponseGen.DATA_EMPTY, e.getMessage());
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (AuthorizationRequiredException e) {
			resultData = ResponseGen.generateXML(ResponseGen.AUTHORIZATION_REQUIRED, e.getMessage());
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (AuthenticationErrorException e) {
			resultData = ResponseGen.generateXML(ResponseGen.AUTHENTICATION_ERROR, e.getMessage());
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (DuplicationException e) {
			resultData = ResponseGen.generateXML(ResponseGen.DUPLICATION, e.getMessage());
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (ForbiddenException e) {
			resultData = ResponseGen.generateXML(ResponseGen.FORBIDDEN, e.getMessage());
			log.warn(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (ClassNotFoundException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INVALID_PARAMETER, "p[^G[");
			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (IllegalArgumentException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INVALID_PARAMETER, "p[^G[");
			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (RuntimeException e) {
			resultData = ResponseGen.generateXML(ResponseGen.INTERNAL_SERVER_ERROR, "T[oG[");
			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		catch (Exception e) {
			resultData = ResponseGen.generateXML(ResponseGen.INTERNAL_SERVER_ERROR, "T[oG[");
			log.error(e.getMessage(), e);
//			e.printStackTrace();
		}
		finally {
			isBinary = appContext.isBinary();
			AppContext.setCurrentContext(null);
		}
		
		// binary
		if (isBinary) {
			OutputStream os = res.getOutputStream();
			if (resultData != null) {
				byte[] bytes = (byte[])resultData;
				os.write(bytes, 0, bytes.length);
				os.close();
			}
			// NOT FOUND
			else {
				res.sendError(HttpServletResponse.SC_NOT_FOUND);
			}
		}
		// xml (JAXB)
		else if (resultData.getClass().isAnnotationPresent(XmlType.class))  {
			String resStr = "";
			if (responseType == null || responseType.equals("xml")) { 
				// Object to XML
				resStr = JaxbUtil.object2Xml(resultData, appContext.getResponseEncoding());
			}
			else if (responseType.equals("vars")) {
				res.setContentType("text/plain");
				resStr = JaxbUtil.object2Html(resultData, "resources/xml2vars.xsl", appContext.getResponseEncoding());
			}
			else {
				resStr = JaxbUtil.object2Xml(resultData, appContext.getResponseEncoding());
			}
			PrintWriter pw = res.getWriter();
			
			pw.print(resStr); 
			pw.flush();
			pw.close();
		}
		// html etc
		else {
			PrintWriter pw = res.getWriter();
			
			pw.print(resultData.toString());
			pw.flush();
			pw.close();
		}
	}
}