/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

typedef struct {
  eh_fd_t *child;
  eh_fd_t *child_stderr;
  eh_connection_t *connection_backref;
  eh_method_t method;
  pid_t pid;
  eh_strbuf_t strbuf_from_client;
  eh_strbuf_t strbuf_from_child;
  eh_strbuf_t strbuf_from_child_stderr;
  int reqbody_len;
  char *headers_str;
  int http_version_minor;
  int reply_flag;
  int child_error_flag;
} eh_cgi_extdata_t;

static void
eh_cgi_request_events (eh_cgi_extdata_t *extdata)
{
  short events = 0;
  if (eh_strbuf_writeok (&extdata->strbuf_from_client)) {
    events = POLLOUT;
  } else if (eh_strbuf_readok (&extdata->strbuf_from_child)) {
    events = POLLIN;
  }
  eh_fd_request_events (extdata->child, events);
}

static void
eh_cgi_do_reply (eh_cgi_extdata_t *extdata)
{
  eh_connection_t *ec = extdata->connection_backref;
  char *buf = extdata->strbuf_from_child.buffer;
  size_t buflen = extdata->strbuf_from_child.buffer_len;
  eh_debug ("buf = [%s]", buf);
  if (extdata->child_error_flag) {
    /* child has not exited gracefully */
    char *str, *s;
    s = x_strndup (buf, buflen);
    str = x_strdup_printf ("Child process has exited abnormally\n--------\n%s",
			   s);
    x_free (s);
    eh_cgicommon_response_500 (ec, extdata->method, str);
    x_free (str);
  } else if (extdata->strbuf_from_child_stderr.buffer_len > 0) {
    /* stderr is not empty */
    char *str, *s, *t;
    t = x_strndup (extdata->strbuf_from_child_stderr.buffer,
		   extdata->strbuf_from_child_stderr.buffer_len);
    s = x_strndup (buf, buflen);
    str = x_strdup_printf ("Child process has exited abnormally\n--------\n%s"
			   "\n--------\n%s",
			   t, s);
    x_free (t);
    x_free (s);
    eh_cgicommon_response_500 (ec, extdata->method, str);
    x_free (str);
  } else {
    eh_cgicommon_reply (ec, extdata->method, buf, buflen);
  }
  eh_connection_request_finish (ec);
  eh_connection_set_request_events (ec, 0x447);
}

static void
eh_cgi_on_read_request_body (eh_rhandler_t *eh, const char *buf, size_t buflen)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)eh->extdata;
  eh_debug ("len = %d", buflen);
  eh_strbuf_append (&extdata->strbuf_from_client, buf, buflen);
  eh->body_length_left = extdata->strbuf_from_client.read_limit;
  if (extdata->child) {
    eh_cgi_request_events (extdata);
  } else {
    if (eh->body_length_left == 0) {
      /* client is already exited abnormally */
      eh_cgi_do_reply (extdata);
      return;
    }
  }
  return;
}

static int
eh_cgi_is_finished (eh_cgi_extdata_t *extdata)
{
  return ((!eh_strbuf_readok (&extdata->strbuf_from_client)) &&
	  (!eh_strbuf_writeok (&extdata->strbuf_from_client)) &&
	  (!eh_strbuf_readok (&extdata->strbuf_from_child)) &&
	  (!eh_strbuf_writeok (&extdata->strbuf_from_child)));
}

static void
eh_cgi_on_close_child (eh_fd_t *ef)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)ef->user_data;
  eh_debug ("");
  eh_fd_set_data (ef, NULL, NULL, NULL, NULL);
  eh_strbuf_set_read_limit (&extdata->strbuf_from_child, 0);
  extdata->child = NULL;
  if (extdata->strbuf_from_client.read_limit == 0) {
    eh_cgi_do_reply (extdata);
  } else {
    /* more data from client. don't reply immediately. */
    return;
  }
}

static void
eh_cgi_stderr_on_close_child (eh_fd_t *ef)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)ef->user_data;
  eh_fd_set_data (ef, NULL, NULL, NULL, NULL);
  extdata->child_stderr = NULL;
}

static void
eh_cgi_on_read_auxfd (eh_cgi_extdata_t *extdata)
{
  int r;
  int fd = extdata->child->pfd.fd;
  r = eh_strbuf_read_append (&extdata->strbuf_from_child, fd);
  eh_debug ("read %d bytes from child", r);
  if (r < 0) {
    if (errno == EWOULDBLOCK || errno == EAGAIN) {
      eh_debug ("wouldblock");
      eh_fd_clear_revents (extdata->child, POLLIN);
      eh_cgi_request_events (extdata);
      return;
    } else {
      eh_log_perror (EH_LOG_WARNING, "read");
      eh_fd_delete_request (extdata->child);
      return;
    }
  } else if (r == 0) {
    eh_debug ("EOF");
    eh_fd_delete_request (extdata->child);
    return;
  }
  eh_cgi_request_events (extdata);
}

static int
eh_cgi_reqbody_has_sent (eh_cgi_extdata_t *extdata)
{
  /* returns 1 if request-body has sent to the child process completely */
  return (!(eh_strbuf_readok (&extdata->strbuf_from_client)) &&
	  !(eh_strbuf_writeok (&extdata->strbuf_from_client)));
}

static void
eh_cgi_on_write_auxfd (eh_cgi_extdata_t *extdata)
{
  int fd = extdata->child->pfd.fd;
  int r;
  eh_debug ("%s", extdata->strbuf_from_client.buffer);
  r = eh_strbuf_write_remove (&extdata->strbuf_from_client, fd);
  if (r < 0) {
    if (errno == EWOULDBLOCK || errno == EAGAIN) {
      eh_fd_clear_revents (extdata->child, POLLOUT);
      eh_cgi_request_events (extdata);
      return;
    } else {
      eh_log_perror (EH_LOG_WARNING, "eh_strbuf_write");
      extdata->child_error_flag = 1;
      eh_fd_delete_request (extdata->child);
      return;
    }
  } else if (r == 0) {
    eh_log (EH_LOG_INFO, "unexpected end of file");
    eh_fd_delete_request (extdata->child);
    return;
  }
  eh_debug ("wrote %d bytes", r);
  eh_debug ("more %d bytes to read, %d bytes in buffer",
	    extdata->strbuf_from_client.read_limit,
	    extdata->strbuf_from_client.buffer_len);
  if (eh_cgi_reqbody_has_sent (extdata)) {
    shutdown (fd, SHUT_WR);
    eh_debug ("shutdown");
  }
  eh_cgi_request_events (extdata);
}

static void
eh_cgi_on_event (eh_fd_t *ef)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)ef->user_data;
  eh_debug ("");
  if ((ef->pfd.revents & POLLIN)) {
    eh_cgi_on_read_auxfd (extdata);
  } else if ((ef->pfd.revents & POLLOUT)) {
    eh_cgi_on_write_auxfd (extdata);
  }
}

static void
eh_cgi_stderr_on_event (eh_fd_t *ef)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)ef->user_data;
  int fd = ef->pfd.fd;
  int r;
  r = eh_strbuf_read_append (&extdata->strbuf_from_child_stderr, fd);
  if (r < 0) {
    if (errno == EWOULDBLOCK || errno == EAGAIN) {
      eh_fd_clear_revents (ef, POLLIN);
      return;
    } else {
      eh_log_perror (EH_LOG_WARNING, "read");
      eh_fd_delete_request (ef);
      return;
    }
  } else if (r == 0) {
    eh_fd_delete_request (ef);
    return;
  }
}

static void
eh_cgi_on_timer (eh_fd_t *ef, int graceful_shutdown)
{
  /* do nothing */
}

static void
eh_cgi_stderr_on_timer (eh_fd_t *ef, int graceful_shutdown)
{
  /* do_nothing */
}

static int
eh_cgi_do_timeout (eh_rhandler_t *eh)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)eh->extdata;
  if (extdata->child == NULL) {
    return 1;
  }
  if (eh_cgi_is_finished (extdata)) {
    return 1;
  }
  if (eh_cgi_reqbody_has_sent (extdata)) {
    /* we've sent the request body to the child process, but the child
       process still remains. we don't want to expire timeout. */
    return 0;
  }
  return 1; /* do expire timeout */
}

static void
eh_cgi_on_delete (eh_rhandler_t *eh)
{
  eh_cgi_extdata_t *extdata = (eh_cgi_extdata_t *)eh->extdata;
  eh_debug ("%p", eh);
  eh_strbuf_discard (&extdata->strbuf_from_client);
  eh_strbuf_discard (&extdata->strbuf_from_child);
  eh_strbuf_discard (&extdata->strbuf_from_child_stderr);
  if (extdata->headers_str)
    x_free (extdata->headers_str);
  if (extdata->child) {
    eh_fd_set_data (extdata->child, NULL, NULL, NULL, NULL);
    eh_fd_delete_request (extdata->child);
  }
  if (extdata->child_stderr) {
    eh_fd_set_data (extdata->child_stderr, NULL, NULL, NULL, NULL);
    eh_fd_delete_request (extdata->child_stderr);
  }
  x_free (extdata);
  x_free (eh);
}

static void
eh_cgi_do_execve (const eh_request_t *er, const eh_connection_t *ec)
{
  /* it doesn't matter if we have memory leaks here. this is a
     child process and we'll do exec() soon. */
  char *argv[3], **envp = NULL;
  argv[0] = er->filename;
  argv[1] = NULL;
  argv[2] = NULL;
  if (er->query_string && strchr (er->query_string, '=') == NULL) {
    char *str;
    str = x_strdup (er->query_string);
    /* it's ok even if str contains invalid %xx sequences */
    (void)eh_decode_url (str);
    if (strchr (str, '=') == NULL)
      argv[1] = eh_strdup_escape_shell_chars (str);
    x_free (str);
  }
  envp = eh_cgicommon_make_env (er, ec);
  if (eh_cgicommon_chdir (er->filename)) {
    fprintf (stderr, "Failed to change directory");
    exit (0);
  }
  execve (er->filename, (char **)argv, (char **)envp);
  perror (er->filename);
}

eh_rhandler_t eh_cgi_tmpl = {
  0, NULL,
  eh_cgi_on_read_request_body,
  eh_cgi_do_timeout,
  eh_cgi_on_delete,
};

eh_rhandler_t *
eh_rhandler_cgi_new (eh_connection_t *ec, const eh_request_t *er,
		     void *rhfunc_data)
{
  eh_rhandler_t *eh = NULL;
  eh_cgi_extdata_t *extdata;
  int sv[2] = {-1, -1};
  int sv_stderr[2] = {-1, -1};
  size_t reqbody_len = 0;
  const char *s;

  s = er->headers.predef.content_length;
  if (s) {
    eh_parse_sizestr (s, &reqbody_len);
  }

  extdata = (eh_cgi_extdata_t *)x_malloc (sizeof (*extdata));
  memset (extdata, 0, sizeof (*extdata));
  eh_strbuf_init (&extdata->strbuf_from_client, reqbody_len);
  eh_strbuf_init (&extdata->strbuf_from_child, (size_t)-1);
  eh_strbuf_init (&extdata->strbuf_from_child_stderr, (size_t)-1);
  extdata->connection_backref = ec;
  extdata->method = er->method;
  extdata->headers_str = eh_headers_strdup_getall (&er->headers);
  extdata->http_version_minor = er->http_version_minor;
  extdata->reply_flag = 0;
  
  eh = (eh_rhandler_t *)x_malloc (sizeof (*eh));
  memcpy (eh, &eh_cgi_tmpl, sizeof (*eh));
  eh->extdata = extdata;
  eh->body_length_left = reqbody_len;

  if (access (er->filename, X_OK) < 0) {
    eh_connection_append_wvec_response (ec, extdata->method, "403",
					NULL, NULL, 0);
    eh_connection_set_request_events (ec, 0x447);
    goto failed;
  }
  if (socketpair (AF_UNIX, SOCK_STREAM, 0, sv) < 0) {
    eh_log_perror (EH_LOG_FATAL, "socketpair");
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
    eh_connection_set_request_events (ec, 0x448);
    goto failed;
  }
  if (socketpair (AF_UNIX, SOCK_STREAM, 0, sv_stderr) < 0) {
    eh_log_perror (EH_LOG_FATAL, "socketpair");
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
    eh_connection_set_request_events (ec, 0x448);
    goto failed;
  }
  if (fcntl (sv[0], F_SETFL, O_RDWR | O_NONBLOCK) < 0) {
    eh_log_perror (EH_LOG_FATAL, "fcntl F_SETFL");
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
    eh_connection_set_request_events (ec, 0x449);
    goto failed;
  }
  extdata->pid = fork ();
  if (extdata->pid < 0) {
    eh_log_perror (EH_LOG_FATAL, "fork");
    eh_connection_append_wvec_response (ec, extdata->method, "500",
					NULL, NULL, 0);
    eh_connection_set_request_events (ec, 0x44a);
    goto failed;
  }
  fflush (stdout);
  if (extdata->pid == 0) {
    /* child */
    close (sv[0]);
    close (sv_stderr[0]);
    dup2 (sv[1], 0);
    dup2 (sv[1], 1);
    dup2 (sv_stderr[1], 2);
    eh_cgi_do_execve (er, ec);
    exit (-2);
  }
  close (sv[1]);
  close (sv_stderr[1]);
  extdata->child = eh_fd_new (sv[0],
			      eh_cgi_on_event,
			      eh_cgi_on_timer,
			      eh_cgi_on_close_child,
			      extdata);
  extdata->child_stderr = eh_fd_new (sv_stderr[0],
				     eh_cgi_stderr_on_event,
				     eh_cgi_stderr_on_timer,
				     eh_cgi_stderr_on_close_child,
				     extdata);
  sv[1] = -1;
  sv[0] = -1;
  sv_stderr[1] = -1;
  sv_stderr[1] = -1;
  eh_debug ("child pid=%u, fd=%d", extdata->pid, extdata->child->pfd.fd);
  /* it's possible that reqest body is of zero-length */
  if (eh_cgi_reqbody_has_sent (extdata)) {
    shutdown (extdata->child->pfd.fd, SHUT_WR);
    eh_debug ("shutdown");
  }
  eh_cgi_request_events (extdata);
  eh_fd_request_events (extdata->child_stderr, POLLIN);
  return eh;

 failed:
  if (sv[0] >= 0)
    close (sv[0]);
  if (sv[1] >= 0)
    close (sv[1]);
  if (sv_stderr[0] >= 0)
    close (sv_stderr[0]);
  if (sv_stderr[1] >= 0)
    close (sv_stderr[1]);
  if (eh)
    eh_cgi_on_delete (eh);
  return eh_rhandler_vacuum_new (ec, er);
}

REGISTER_HANDLER ("cgi-script", eh_rhandler_cgi_new, NULL, 0);
