/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

#define EH_INDEX_FILENAME_MAX 32

typedef struct {
  char *fname;
  struct stat sbuf;
} eh_dirent_t;

static int compare_name (const void *x, const void *y)
{
  const eh_dirent_t *a, *b;
  a = (const eh_dirent_t *)x;
  b = (const eh_dirent_t *)y;
  return strcmp (a->fname, b->fname);
}

void
eh_connection_get_dirent (char **strp, const char *dirname,
			  const char *url, DIR *dir)
{
  int i;
  int num_ent = 0;
  struct dirent *dent;
  eh_dirent_t *ent = NULL;
  while ((dent = readdir (dir)) != NULL) {
    char *path;
    if (dent->d_name[0] == '.')
      continue;
    i = num_ent++;
    ent = (eh_dirent_t *)x_realloc (ent, num_ent * sizeof (*ent));
    ent[i].fname = x_strdup (dent->d_name);
    path = x_strdup_printf ("%s%s", dirname, dent->d_name);
    if (stat (path, &ent[i].sbuf)) {
      x_free (path);
      x_free (ent[i].fname);
      num_ent--;
      continue;
    }
    x_free (path);
  }
  qsort (ent, num_ent, sizeof (*ent), &compare_name);
  for (i = 0; i < num_ent; i++) {
    char *slash;
    char trname[EH_INDEX_FILENAME_MAX + 10];
    char space[EH_INDEX_FILENAME_MAX + 10];
    char *filename_q;
    char *trname_q;
    char sizestr[21];
    static char mtimebuf[81];
    int trnamemax = EH_INDEX_FILENAME_MAX;
    int trnamelen, spacelen;
    eh_debug ("%d: fname=%s", i, ent[i].fname);
    trname[trnamemax] = '\0';
    trnamelen = strlen (ent[i].fname);
    strncpy (trname, ent[i].fname, trnamemax);
    if (trnamelen > trnamemax) {
      trname[trnamemax - 2] = trname[trnamemax - 1] = '.';
      trnamelen = trnamemax;
    }
    spacelen = trnamemax - trnamelen;
    if (S_ISDIR (ent[i].sbuf.st_mode)) {
      slash = "/";
      snprintf (sizestr, 20, "         -");
    } else {
      slash = "";
      snprintf (sizestr, 20, "%10lu", ent[i].sbuf.st_size);
      sizestr[20] = '\0';
      spacelen++;
    }
    memset (space, ' ', spacelen);
    space[spacelen] = '\0';
    strftime (mtimebuf, 80, "%d %b %Y %H:%M:%S",
	      gmtime (&ent[i].sbuf.st_mtime));
    trname_q = eh_html_strdup_escape (trname);
    filename_q = eh_strdup_encode_url (ent[i].fname);
    x_append_printf (strp, "<A HREF=\"%s%s\">%s%s</A>%s%s  %s\r\n",
		     filename_q, slash,
		     trname_q, slash,
		     space, mtimebuf, sizestr);
    x_free (trname_q);
    x_free (filename_q);
  }
  for (i = 0; i < num_ent; i++) {
    x_free (ent[i].fname);
  }
  x_free (ent);
}

void
eh_connection_append_wvec_directory (eh_connection_t *ec, eh_request_t *er)
{
  size_t bodystr_len;
  char *bodystr, *tmpstr;
  DIR *dir;
  if (er->method != eh_method_get && er->method != eh_method_head) {
    eh_connection_append_wvec_response (ec, er->method, "405", NULL, NULL, 0);
    return;
  }
  if ((dir = opendir (er->filename)) == NULL) {
    eh_debug ("opendir : %s", er->filename);
    eh_connection_append_wvec_response (ec, er->method, "404", NULL, NULL, 0);
    return;
  }
  tmpstr = eh_html_strdup_escape (er->url);
  bodystr = x_strdup_printf ("%s"
			     "<HTML>"
			     "<HEAD>"
			     "<TITLE>Index of %s</TITLE>"
			     "</HEAD>\r\n"
			     "<BODY>\r\n"
			     "<H1>Index of %s</H1>\r\n"
			     "<PRE>\r\n"
			     "Name                             "
			     "Last Modified               "
			     "Size"
			     "\r\n"
			     "<HR>"
			     ,
			     eh_html_get_doctype_decl (), tmpstr, tmpstr);
  x_free (tmpstr);
  eh_connection_get_dirent (&bodystr, er->filename, er->url, dir);
  closedir (dir);
  x_str_append (&bodystr, "</PRE>\r\n<HR>\r\n");
  tmpstr = eh_html_strdup_server_address (er);
  x_append_printf (&bodystr, "%s\r\n", tmpstr);
  x_free (tmpstr);
  x_str_append (&bodystr, "</BODY>\r\n</HTML>\r\n");
  bodystr_len = strlen (bodystr);
  eh_connection_append_wvec_response (ec, er->method, "200", NULL,
				      bodystr, bodystr_len);
  x_free (bodystr);
}
