/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <crypt.h>
#include <sys/types.h>
#include <sys/stat.h>

static void
show_usage (const char *argv0)
{
  fprintf (stderr, "Usage: %s [-c] [-m] passwordfile username\n", argv0);
  fprintf (stderr, "-c    Create a new file.\n");
  fprintf (stderr, "-m    Use MD5 encryption. Only work with the GNU libc.\n");
  exit (1);
}

static char *
get_passwd (void)
{
  char *r;
  r = strdup (getpass ("New password:"));
  if (strcmp (r, getpass ("Re-type new password:")) != 0) {
    fprintf (stderr, "They don't match, sorry.\n");
    exit (1);
  }
  return r;
}

static char *
read_passwd_file (const char *filename, const char *username, int c_flag,
		  size_t *contents_len_r)
{
  char *pf;
  size_t pflen;
  FILE *fp;
  char line[1024];
  pf = (char *)malloc (1);
  assert (pf);
  pflen = 0;
  *contents_len_r = 0;
  fp = fopen (filename, "r");
  if (fp == NULL) {
    if (c_flag)
      return pf;
    perror (filename);
    exit (1);
  }
  while (fgets (line, 1024, fp)) {
    size_t len = strlen (line);
    char *p;
    if (len < 1 && line[len - 1] != '\n') {
      fprintf (stderr, "invalid line is found in %s.", filename);
      exit (1);
    }
    if (line[0] == '#' || line[0] == ' ') {
      /* don't parse */
    } else {
      p = strchr (line, ':');
      if (p &&
	  (int)strlen (username) == p - line &&
	  strncmp (username, line, p - line) == 0) {
	/* skip this entry */
	continue;
      }
    }
    pf = (char *)realloc (pf, pflen + len);
    memcpy (pf + pflen, line, len);
    pflen += len;
  }
  *contents_len_r = pflen;
  return pf;
}

static void
write_passwd_file (const char *filename, const char *contents,
		   size_t contents_len, int md5_flag,
		   const char *username, const char *passwd)
{
  char salt[13];
  const char saltchars[] =
    "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
  int randval;
  char *cpasswd;
  FILE *fp;
  
  assert (strlen (saltchars) == 64);
  srand ((int) time (NULL));
  if (md5_flag) {
    /* MD5(works with glibc only) */
    int i;
    salt[0] = '$';
    salt[1] = '1';
    salt[2] = '$';
    for (i = 0; i < 4; i++) {
      randval = (int) (4096.0 * rand () / RAND_MAX);
      salt[i * 2 + 3] = saltchars[randval & 63];
      salt[i * 2 + 4] = saltchars[(randval / 64) & 63];
    }
    salt[11] = '$';
    salt[12] = '\0';
  } else {
    /* DES */
    randval = (int) (4096.0 * rand () / RAND_MAX);
    salt[0] = saltchars[randval & 63];
    salt[1] = saltchars[(randval / 64) & 63];
    salt[2] = '\0';
  }
  cpasswd = crypt (passwd, salt);
  if (md5_flag && cpasswd && cpasswd[2] != '$') {
    fprintf (stderr,
	     "The crypt routine seems to have no support for the MD5\n"
	     "encryption. Don't use the -m flag.\n");
    exit (1);
  }
      

  umask (077);
  fp = fopen (filename, "w");
  if (fp == NULL) {
    perror (filename);
    exit (1);
  }
  if (contents_len)
    fwrite (contents, contents_len, 1, fp);
  fprintf (fp, "%s:%s\n", username, cpasswd);
  if (fclose (fp)) {
    perror (filename);
    exit (1);
  }
}

int
main (int argc, char **argv)
{
  int c_flag = 0, md5_flag = 0;
  const char *passwdfile = NULL, *username = NULL;
  char *passwd;
  char *contents;
  size_t contents_len;
  int i;
  for (i = 1; i < argc; i++) {
    if (strcmp (argv[i], "-c") == 0) {
      c_flag = 1;
    } else if (strcmp (argv[i], "-m") == 0) {
      md5_flag = 1;
    } else if (argv[i][0] == '-') {
      show_usage (argv[0]);
    } else if (passwdfile == NULL) {
      passwdfile = argv[i];
    } else if (username == NULL) {
      username = argv[i];
    } else {
      show_usage (argv[0]);
    }
  }
  if (username == NULL ||
      strchr (username, ':') != NULL ||
      strchr (username, ' ') != NULL) {
    show_usage (argv[0]);
  }
  contents = read_passwd_file (passwdfile, username, c_flag, &contents_len);
  passwd = get_passwd ();
  write_passwd_file (passwdfile, contents, contents_len, md5_flag,
		     username, passwd);
  return 0;
}
