/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static int
eh_connection_check_request_body_limit (eh_connection_t *ec, eh_request_t *req)
{
  const char *s;
  static const char *cont_str = "HTTP/1.1 100 Continue\r\n\r\n";
  int requestbodylimit;
  requestbodylimit = ec->app_backref->econf->requestbodylimit;
  s = req->headers.predef.expect;
  if (req->http_version_minor < 1 || s == NULL) {
    if (req->request_body_length > (size_t)requestbodylimit) {
      eh_connection_append_wvec_response (ec, req->method, "413", NULL, NULL,
					  0);
      return 1;
    }
  } else {
    if (strcasecmp (s, "100-continue") != 0 ||
	req->request_body_length > (size_t)requestbodylimit) {
      eh_connection_append_wvec_response (ec, req->method, "417", NULL, NULL,
					  0);
      return 1;
    }
    eh_connection_append_wvec (ec, (void *)cont_str, strlen (cont_str),
			       NULL, NULL);
  }
  return 0;
}

static int
eh_connection_parse_request_str (eh_connection_t *ec, char *request_str)
{
  eh_request_t *const req = &ec->current_request;
  int r;
  eh_config_vhost_t *default_vhost;

  default_vhost = ec->vserver_backref->default_vhost;
  ec->rconn = req->rconn;
  ec->last_request_has_body = (req->headers.predef.content_length != NULL);
  ec->is_http_1_0 = (req->http_version_minor == 0);
  if (req->bad_request) {
    eh_connection_append_wvec_response (ec, req->method, "400", NULL, NULL, 0);
    eh_connection_request_finish (ec);
    return 1;
  }
  if (req->method == eh_method_unknown) {
    eh_connection_append_wvec_response (ec, req->method, "501", NULL, NULL, 0);
    eh_connection_request_finish (ec);
    return 1;
  }
  if ((r = eh_request_url_to_filename (req, ec)) != 0) {
    eh_debug ("not found: filename = %s", req->filename ? req->filename : "");
    eh_connection_append_wvec_response (ec, req->method,
					r < 0 ? "400" : "404", NULL,
					NULL, 0);
    eh_connection_request_finish (ec);
    return 1;
  }
  return 0;
}


static size_t
eh_connection_reply_now (eh_connection_t *ec)
{
  eh_request_t *const req = &ec->current_request;
  size_t len_accepted = req->request_len;

  ec->app_backref->reply_count++;
  
  if (req->econf_dir_ref->ssl_verifyclient) {
    if (ec->ssl_peer == NULL || ec->ssl_x509_v_ok == 0) {
      eh_connection_append_wvec_response (ec, req->method,
					  "403", NULL, NULL, 0);
      eh_connection_request_finish (ec);
      return len_accepted;
    }
  }

  if (req->econf_dir_ref->sslrequiressl && ec->sslcon == NULL) {
    eh_connection_append_wvec_response (ec, req->method,
					"403", NULL, NULL, 0);
    eh_connection_request_finish (ec);
    return len_accepted;
  }
  
  if (eh_connection_check_auth (ec, req)) {
    eh_connection_request_finish (ec);
    return len_accepted;
  }

  // TODO: check if POST is allowed before sending 100-continue
  if (eh_connection_check_request_body_limit (ec, req)) {
    eh_connection_request_finish (ec);
    return len_accepted;
  }
  
  if (req->rhfunc) {
    const char *s, *t;
    s = req->headers.predef.content_length;
    t = req->headers.predef.transfer_encoding;
    if (s == NULL &&
	(req->method == eh_method_post || req->method == eh_method_put)) {
      eh_connection_append_wvec_response (ec, req->method,
					  "411", NULL, NULL, 0);
      eh_connection_request_finish (ec);
    } else if (t) {
      eh_connection_append_wvec_response (ec, req->method,
					  "400", NULL, NULL, 0);
      eh_connection_request_finish (ec);
    } else {
      eh_debug ("setting rhfunc");
      ec->rhandler = (*req->rhfunc)(ec, req, req->rhfunc_data);
      if (ec->rhandler) {
	const char *s;
	const char *request_str;
	size_t reqbody_len = 0, reqbody_readlen, request_len, extra_len;
	request_str = req->request_str_ref;
	request_len = req->request_len;
	extra_len = req->extra_len;
	s = req->headers.predef.content_length;
	if (s)
	  reqbody_len = atoi (s);
	s = req->headers.predef.transfer_encoding;
	reqbody_readlen = reqbody_len > extra_len ? extra_len : reqbody_len;
	if (reqbody_readlen > 0) {
	  /* we've already read some portion of the request body. send it
	     to the rhandler. */
	  eh_rhandler_on_read_request_body (ec, request_str + request_len,
					    reqbody_readlen);
	  len_accepted += reqbody_readlen;
	}
      }
    }
  } else if (S_ISDIR (req->statbuf.st_mode)) {
    eh_debug ("directory");
    eh_connection_append_wvec_directory (ec, req);
    eh_connection_request_finish (ec);
  } else {
    eh_debug ("static content");
    eh_connection_append_wvec_file (ec, req);
    eh_connection_request_finish (ec);
  }
  
  assert (ec->rhandler != NULL || ec->current_request.not_finished == 0);
  return len_accepted;
}

size_t
eh_connection_reply (eh_connection_t *ec, size_t request_str_offset,
		     size_t request_len, size_t extra_len)
{
  eh_request_t *const req = &ec->current_request;
  char *request_str = ec->readbuf.buffer + request_str_offset;
  
  eh_debug ("");
  assert (request_str[request_len - 1] == '\0');

  eh_request_init (req, request_str, request_len, extra_len,
		   ec->vserver_backref->default_vhost,
		   ec->app_backref->econf->use_access_log,
		   &ec->accesslog);

  if (eh_connection_parse_request_str (ec, request_str)) {
#ifdef DEBUG_BAD_REQUEST
    int i;
    for (i = 0; i < request_str_offset + request_len + extra_len; i++)
      if (ec->readbuf.buffer[i] == 0)
	ec->readbuf.buffer[i] = ' ';
    eh_log(EH_LOG_INFO, "BAD REQUEST buflen = %u offset = %u\n%s\n",
	ec->readbuf.buffer_len, request_str_offset,
	ec->readbuf.buffer);
#endif
    return request_len;
  }
  
  if (ec->sslcon && req->econf_dir_ref->ssl_verifyclient
      && ec->ssl_peer == NULL) {
    SSL_set_verify_depth (ec->sslcon, req->econf_dir_ref->sslverifydepth);
    SSL_set_verify (ec->sslcon,
		    SSL_VERIFY_PEER |
		    SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
		    NULL);
    eh_debug ("SSL_renegotiate");
    SSL_renegotiate (ec->sslcon);
    SSL_do_handshake (ec->sslcon);
    eh_debug ("SSL_is_init_finished: %d", SSL_is_init_finished (ec->sslcon));
    ec->ssl_renegotiate_is_in_progress = 1;
    return 0;
  }

  return eh_connection_reply_now (ec);
}

size_t
eh_connection_reply_after_renegotiation (eh_connection_t *ec)
{
  return eh_connection_reply_now (ec);
}
