/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

static void
eh_serverstatus_on_read_reqbody (eh_rhandler_t *eh, const char *buf,
				 size_t buflen)
{
  eh_connection_t *ec = (eh_connection_t *)eh->extdata;
  assert (buflen <= eh->body_length_left);
  eh_debug ("");
  eh->body_length_left -= buflen;
  if (eh->body_length_left == 0) {
    eh_debug ("finish");
    eh_connection_request_finish (ec);
  }
}

static int
eh_serverstatus_do_timeout (eh_rhandler_t *eh)
{
  return 1;
}

static void
eh_serverstatus_on_delete (eh_rhandler_t *eh)
{
  eh_debug ("");
  x_free (eh);
}

eh_rhandler_t eh_serverstatus_tmpl = {
  0, NULL,
  eh_serverstatus_on_read_reqbody,
  eh_serverstatus_do_timeout,
  eh_serverstatus_on_delete,
};

static void
eh_serverstatus_filecache (eh_connection_t *ec, char **strp)
{
  eh_filecache_t *fc;
  eh_filecache_entry_t *fcent;
  fc = ec->app_backref->fcache;
  for (fcent = fc->first; fcent; fcent = fcent->next) {
    x_append_printf (strp, "(ref=%lu len=%lu) %s\n",
		     fcent->wvec_refcount, fcent->statbuf.st_size,
		     fcent->hash_key_backref);
  }
}

static void
eh_serverstatus_connections (eh_connection_t *ec, char **strp)
{
  eh_fd_show_status (strp, ec->client_pfd->pfd.fd);
}

static char *
eh_serverstatus_get (eh_connection_t *ec, const eh_request_t *er)
{
  char *body;
  char *str;
  body = x_strdup ("<HTML>\n");
  x_append_printf (&body, "<HEAD><TITLE>%s Server Status</TITLE></HEAD>\n",
		   ESEHTTPD_NAME);
  x_append_printf (&body, "<BODY>\n");
  x_append_printf (&body, "<H1>%s Server Status</H1>\n",
		   ESEHTTPD_NAME);
  x_append_printf (&body, "<HR>\n");
  if (er->query_string && strcmp (er->query_string, "cache") == 0) {
    x_append_printf (&body, "<PRE>\n");
    eh_serverstatus_filecache (ec, &body);
    x_append_printf (&body, "</PRE>\n");
  } else if (er->query_string && strcmp (er->query_string, "conn") == 0) {
    x_append_printf (&body, "<PRE>\n");
    eh_serverstatus_connections (ec, &body);
    x_append_printf (&body, "</PRE>\n");
  } else {
    x_append_printf (&body, "<PRE>\n");
    x_append_printf (&body, "number of workers: %d\n",
		     ec->app_backref->num_child);
    x_append_printf (&body, "worker id: %d\n",
		     ec->app_backref->my_worker_id);
    x_append_printf (&body, "file descriptors in use: %d\n",
		     eh_fd_get_open_count ());
    x_append_printf (&body, "filecache entries: %d\n",
		     eh_filecache_get_num_entries (ec->app_backref->fcache));
    x_append_printf (&body, "reply count: %lu",
		     ec->app_backref->reply_count);
    x_append_printf (&body, "</PRE>\n");
    x_append_printf (&body, "<HR>");
    x_append_printf (&body, "<a href=\"?conn\">Connections</a><br>");
    x_append_printf (&body, "<a href=\"?cache\">File Cache</a><br>");
  }
  x_append_printf (&body, "<HR>\n");
  str = eh_html_strdup_server_address (er);
  x_append_printf (&body, "%s\n", str);
  x_free (str);
  x_append_printf (&body, "</BODY>\n");
  x_append_printf (&body, "</HTML>\n");
  return body;
}

eh_rhandler_t *
eh_rhandler_serverstatus_new (struct eh_connection_s *ec,
			      const eh_request_t *er, void *rhfunc_data)
{
  int reqbody_len = 0;
  eh_rhandler_t *eh;
  const char *s;
  char *message;
  eh_debug ("");
  message = eh_serverstatus_get (ec, er);
  eh_connection_append_wvec_response (ec, er->method, "200",
				      "Content-Type: text/html\r\n",
				      message, strlen (message));
  x_free (message);
  s = er->headers.predef.content_length;
  if (s) {
    reqbody_len = atoi (s);
  }
  if (reqbody_len == 0) {
    eh_connection_request_finish (ec);
    return NULL;
  }
  eh = (eh_rhandler_t *)x_malloc (sizeof (*eh));
  memcpy (eh, &eh_serverstatus_tmpl, sizeof (*eh));
  eh->body_length_left = reqbody_len;
  eh->extdata = (void *)ec;
  return eh;
}

REGISTER_HANDLER("serverstatus", eh_rhandler_serverstatus_new, NULL, 1);
