/*
 * ESE, a HyperText Transfer Protocol server
 * Copyright (C) 1996-2001 Akira Higuchi <a-higuti@math.sci.hokudai.ac.jp>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "esehttpd.h"

void
eh_strbuf_init (eh_strbuf_t *bp, size_t read_limit)
{
  bp->buffer_len = 0;
  bp->buffer_alloclen = EH_READ_MAX + 2;
  bp->buffer = (char *)x_realloc (0, bp->buffer_alloclen);
  bp->buffer[0] = '\0';
  bp->read_limit = read_limit;
  bp->read_len = 0;
  bp->write_len = 0;
}

void
eh_strbuf_discard (eh_strbuf_t *bp)
{
  bp->buffer_len = 0;
  bp->buffer_alloclen = 0;
  bp->buffer = (char *)x_realloc (bp->buffer, 0);
  bp->read_limit = 0;
  bp->read_len = 0;
  bp->write_len = 0;
}

void
eh_strbuf_set_read_limit (eh_strbuf_t *bp, size_t read_limit)
{
  bp->read_limit = read_limit;
}

static void
eh_strbuf_make_space (eh_strbuf_t *bp, size_t len)
{
  /* make enough space for storing buffer + len + '\0' */
  size_t alen;
  alen = bp->buffer_alloclen;
  while (bp->buffer_len + len >= alen) {
    alen += EH_READ_MAX;
  }
  if (alen != bp->buffer_alloclen) {
    bp->buffer_alloclen = alen;
    bp->buffer = (char *)x_realloc (bp->buffer, bp->buffer_alloclen);
  }
}

static void
eh_strbuf_make_nulterm (eh_strbuf_t *bp)
{
  assert (bp->buffer_len < bp->buffer_alloclen);
  bp->buffer[bp->buffer_len] = '\0';
}

void
eh_strbuf_append (eh_strbuf_t *bp, const char *str, size_t len)
{
  len = (bp->read_limit > len) ? len : bp->read_limit;
  if (len == 0)
    return;
  eh_strbuf_make_space (bp, len + 1);
  memcpy (bp->buffer + bp->buffer_len, str, len);
  bp->buffer_len += len;
  if (bp->read_limit != (size_t)-1)
    bp->read_limit -= len;
  eh_strbuf_make_nulterm (bp);
}

void
eh_strbuf_remove (eh_strbuf_t *bp, size_t len)
{
  len = (bp->buffer_len > len) ? len : bp->buffer_len;
  if (len == 0)
    return;
  memmove (bp->buffer, bp->buffer + len, bp->buffer_len - len);
  bp->buffer_len -= len;
  bp->write_len += len;
  eh_strbuf_make_nulterm (bp);
}

int
eh_strbuf_readok (eh_strbuf_t *bp)
{
  return (bp->read_limit > 0);
}

int
eh_strbuf_writeok (eh_strbuf_t *bp)
{
  return (bp->buffer_len > 0);
}

int
eh_strbuf_read_append (eh_strbuf_t *bp, int fd)
{
  size_t maxlen;
  int r;
  maxlen = (bp->read_limit > EH_READ_MAX) ? EH_READ_MAX : bp->read_limit;
  assert (bp->buffer);
  eh_strbuf_make_space (bp, maxlen);
  assert (bp->buffer);
  r = read (fd, bp->buffer + bp->buffer_len, maxlen);
  if (r < 0) {
    int errno_back = errno;
    eh_debug ("read -1 errno = %d", errno);
    errno = errno_back;
  }
  if (r > 0) {
    bp->buffer_len += r;
    bp->read_len += r;
    if (bp->read_limit != (size_t)-1)
      bp->read_limit -= r;
    eh_strbuf_make_nulterm (bp);
    return r;
  }
  return r;
}

int
eh_strbuf_ssl_read_append (eh_strbuf_t *bp, SSL *sslcon)
{
  size_t maxlen;
  int r;
  maxlen = (bp->read_limit > EH_READ_MAX) ? EH_READ_MAX : bp->read_limit;
  assert (bp->buffer);
  eh_strbuf_make_space (bp, maxlen);
  assert (bp->buffer);
  r = SSL_read (sslcon, bp->buffer + bp->buffer_len, maxlen);
  if (r > 0) {
    bp->buffer_len += r;
    bp->read_len += r;
    if (bp->read_limit != (size_t)-1)
      bp->read_limit -= r;
    eh_strbuf_make_nulterm (bp);
    return r;
  }
  return r;
}

int
eh_strbuf_write_remove (eh_strbuf_t *bp, int fd)
{
  int r;
  if (!eh_strbuf_writeok (bp)) {
    errno = EAGAIN;
    return -1;
  }
  r = write (fd, bp->buffer, bp->buffer_len);
  if (r > 0) {
    eh_strbuf_remove (bp, r);
  }
  return r;
}
