#include "stdafx.h"
#include "Document.h"
#include "IOBase.h"
#include <io.h>

//Implementation of IDocumentEventListener
void IDocumentEventListener::OnUpdate(IDocumentEventListener::EventType type)
{
	type;
}

//Implementation of IDocumentAccessor
IDocumentAccessor::~IDocumentAccessor()
{
}

void IDocumentAccessor::SetDocument(Document_ptr doc)
{
	doc_ = doc;
}

Document_ptr IDocumentAccessor::GetDocument() const
{
	return doc_;
}


Document::Document()
	: file_loaded_(false), modified_(false), formattype_(Format_Simple), insertmode_(true)
{
	InitFormatter();
}

void Document::Reset()
{
	//Agr[g
	modified_ = false;

	//J[\
	cursor_.reset(new CursorInfo());
	cursor_->SetPosition(0);
	cursor_->Enable(true);

	//XN[
	scroll_.reset(new ScrollInfo());

	//AhDXg
	undo_.Clear();

	//͏񏉊
	inputmanager_.reset(new InputManager());

	//񏉊
	search_.reset(new SearchInfo());
}

void Document::New()
{
	//IO
	buffer_ = Buffer_ptr(new FullFileBuffer());
	reader_ = Reader_ptr(new NullReader());
	//reader_ = Reader_ptr(new DummyReader());
	buffer_->Load(reader_);
	filepath_.clear();
	file_loaded_ = false;

	Reset();

	NotifyReconstruct();
}

bool Document::Open( const std::string& filepath )
{
	bool succeed = false;

	if(_access(filepath.c_str(),0)==0){
		//IO
		Buffer_ptr newbuffer(new FullFileBuffer());
		Reader_ptr newreader(new FileReader(filepath));
		
		succeed = newbuffer->Load(newreader);
		if(succeed){
			buffer_ = newbuffer;
			reader_ = newreader;
			file_loaded_ = true;
			filepath_ = filepath;
			Reset();
		}

		NotifyReconstruct();
	}

	return succeed;
}

bool Document::Save( const std::string& filepath )
{
	Writer_ptr newwriter(new FileWriter(filepath));
	bool succeed = buffer_->Save(newwriter);
	if(succeed){
		writer_ = newwriter;
		file_loaded_ = true;
		filepath_ = filepath;
		modified_ = false;
	}

	NotifyRefresh();

	return succeed;
}

bool Document::DoAction(ActionList list)
{
	bool succeed = Action::Redo(list,buffer_.get());

	undo_.Append(list);

	modified_ = true;

	return succeed;
}

bool Document::Undo()
{
	return undo_.Undo(buffer_.get());
}

bool Document::Redo()
{
	return undo_.Redo(buffer_.get());
}

bool Document::CanUndo() const
{
	return undo_.IsEnabledUndoing();
}

bool Document::CanRedo() const
{
	return undo_.IsEnabledRedoing();
}

ulong Document::Size() const
{
	return buffer_->Size();
}

bool Document::IsModified() const
{
	return modified_;
}

std::string Document::GetFilePath() const
{
	return filepath_;
}

bool Document::IsInsertMode() const
{
	return insertmode_;
}

void Document::SetInsertMode(bool insert)
{
	insertmode_ = insert;
}

ulong Document::Read(ulong offset, ulong size, uchar *buffer)
{
	return buffer_->Get(offset,size,buffer);
}

void Document::AddListener(IDocumentEventListener *listener)
{
	listenerlist_.push_back(listener);
}

void Document::NotifyReconstruct()
{
	for(std::list<IDocumentEventListener*>::iterator it=listenerlist_.begin(); it!=listenerlist_.end(); ++it){
		(*it)->OnUpdate(IDocumentEventListener::Reconstruction);
	}
}

void Document::NotifyRefresh()
{
	for(std::list<IDocumentEventListener*>::iterator it=listenerlist_.begin(); it!=listenerlist_.end(); ++it){
		(*it)->OnUpdate(IDocumentEventListener::Refresh);
	}
}

void Document::InitFormatter()
{
	switch(formattype_){
	case Format_Simple:
		formatter_.reset(new SimpleFormatter(*this));
		break;
	default:
		break;
	}
}

Cursor_ptr Document::GetCursor() const
{
	return cursor_;
}

Scroll_ptr Document::GetScroll() const
{
	return scroll_;
}

Formatter_ptr Document::GetFormat() const
{
	return formatter_;
}

InputManager_ptr Document::GetInputInfo() const
{
	return inputmanager_;
}

Search_ptr Document::GetSearchInfo() const
{
	return search_;
}

buffer_iterator Document::Begin() const
{
	return buffer_iterator(buffer_.get(),buffer_->Begin());
}
	
buffer_iterator Document::At(ulong offset) const
{
	buffer_iterator it = Begin()+offset;
	return it;
}

buffer_iterator Document::End() const
{
	return buffer_iterator(buffer_.get(),buffer_->End());
}

bool CheckInDataRange( Document_ptr document )
{
	return CheckInDataRange(document,document->GetCursor());
}

bool CheckInDataRange( Document_ptr document, Cursor_ptr cursor )
{
	ulong size = document->Size();
	ulong start, end;
	cursor->GetRange(start,end);

	if(!cursor->IsEnable() || size<=end || size<=start){
		return false;
	}else{
		return true;
	}
}

bool CheckInEditRange( Document_ptr document )
{
	return CheckInEditRange(document,document->GetCursor());
}

bool CheckInEditRange( Document_ptr document, Cursor_ptr cursor )
{
	ulong size = document->Size()+1;
	ulong start, end;
	cursor->GetRange(start,end);

	if(!cursor->IsEnable() || size<=end || size<=start){
		return false;
	}else{
		return true;
	}
}

void AdjustRange( Document_ptr document )
{
	Cursor_ptr cursor = document->GetCursor();
	ulong size = document->Size();
	ulong anchor = cursor->GetAnchor();
	ulong position = cursor->GetPosition();
	
	if(anchor>=size) anchor = size-1;
	if(position>=size) position = size-1;

	cursor->SetAnchor(anchor);
	cursor->SetPosition(position);
}
