#include "stdafx.h"
#include "Formatter.h"
#include "Document.h"
#include <boost/scoped_array.hpp>


uchar Ascii2char(uchar ascii)
{
	if(ascii<0x21 || 0xFE<ascii){
		return ' ';
	}else{
		return ascii;
	}
}




//implementation of IFormatter
IFormatter::~IFormatter()
{
}


//interface of SimpleFormatter::Impl
class SimpleFormatter::Impl
{
public:
	Impl(Document& document);

	ulong GetRowCount() const;
	ulong GetColumnCount(ulong lineindex) const;
	ulong GetItemCount(ulong lineindex) const;
	ulong GetMaxColumnCount() const;
	LineFormat_ptr GetLine(ulong lineindex);
	LineFormat_ptr GetHeader(ulong lineindex);
	ulong LineIndex2Offset(ulong lineindex) const;
	ulong Offset2LineIndex(ulong offset) const;

private:
	static const int linecolumncount = 16;
	static const int linecharactercount = linecolumncount*4;
	static const int itemsize = 1;
	Document* document_;

	ulong FieldSize() const;
};

//implementation of SimpleFormatter::Impl
SimpleFormatter::Impl::Impl(Document& document)
	: document_(&document)
{
}

ulong SimpleFormatter::Impl::GetRowCount() const
{
	return (FieldSize()+linecolumncount-1)/linecolumncount;
}

ulong SimpleFormatter::Impl::GetColumnCount(ulong lineindex) const
{
	lineindex;
	return GetMaxColumnCount();
}

ulong SimpleFormatter::Impl::GetItemCount(ulong lineindex) const
{
	ulong rest = FieldSize()-LineIndex2Offset(lineindex);
	return (rest>linecolumncount)?linecolumncount:rest;
}

ulong SimpleFormatter::Impl::GetMaxColumnCount() const
{
	return linecharactercount;
}

LineFormat_ptr SimpleFormatter::Impl::GetLine(ulong lineindex)
{
	LineFormat_ptr lineformat(new LineFormat);

	uchar srcbuffer[linecolumncount];
	char tmpbuffer[linecharactercount+1];

	ulong byteoffset = LineIndex2Offset(lineindex);
	ulong readsize = document_->Read(byteoffset,linecolumncount,srcbuffer);
	bool end = false;

	//16i\
	for(uint i=0; i<readsize; i++){
		ItemFormat item(i*3, 3, 0, 2, byteoffset+i, 1, true, Format::Hex); //3(2+)
		lineformat->items.push_back(item);
		uchar hi =  ((srcbuffer[i]&0xF0)>>4);
		uchar low = (srcbuffer[i]&0x0F);
		sprintf(tmpbuffer+item.offset,"%1X%1X ",hi,low);
	}
	//J[\
	if(readsize<linecolumncount){
		ItemFormat item(readsize*3, 3, 0, 2, byteoffset+readsize, 1, true, Format::Hex); //3(*3)
		lineformat->items.push_back(item);
		sprintf(tmpbuffer+item.offset,"   ");
		end = true;
	}
	//peBO
	int paddingsize = (linecolumncount - readsize - (end?1:0))*3;
	if(paddingsize){
		memset(tmpbuffer+(readsize+(end?1:0))*3,' ',paddingsize);
	}
	//\(ASCII)
	for(uint i=0; i<readsize; i++){
		ItemFormat item(linecolumncount*3+i, 1, 0, 1, byteoffset+i, 1);
		lineformat->items.push_back(item);
		uchar ch = Ascii2char(srcbuffer[i]);
		sprintf(tmpbuffer+item.offset,"%c",ch);
	}
	lineformat->text = tmpbuffer;
	
	return lineformat;
}

LineFormat_ptr SimpleFormatter::Impl::GetHeader(ulong lineindex)
{
	lineindex;

	char tmpbuffer[linecharactercount+1];

	LineFormat_ptr lineformat(new LineFormat);

	for(uint i=0; i<linecolumncount; i++){
		ItemFormat item(i*3, 3, 1, 1, i, 1);
		lineformat->items.push_back(item);
		sprintf(tmpbuffer+item.offset," %1X ",i);
	}
	for(uint i=0; i<linecolumncount; i++){
		ItemFormat item(linecolumncount*3+i, 1, 0, 1, i, 1);
		lineformat->items.push_back(item);
		sprintf(tmpbuffer+item.offset,"%1X",i);
	}
	lineformat->text = tmpbuffer;

	return lineformat;
}

ulong SimpleFormatter::Impl::LineIndex2Offset(ulong lineindex) const
{
	return lineindex*linecolumncount;
}

ulong SimpleFormatter::Impl::Offset2LineIndex(ulong offset) const
{
	return offset/linecolumncount;
}

ulong SimpleFormatter::Impl::FieldSize() const
{
	//J[\
	return document_->Size()+itemsize;
}


//implementation of SimpleFormatter
SimpleFormatter::SimpleFormatter(Document& document)
	: pimpl_(new Impl(document))
{
}

ulong SimpleFormatter::GetRowCount() const
{
	return pimpl_->GetRowCount();
}

ulong SimpleFormatter::GetColumnCount(ulong lineindex) const
{
	return pimpl_->GetColumnCount(lineindex);
}

ulong SimpleFormatter::GetItemCount(ulong lineindex) const
{
	return pimpl_->GetItemCount(lineindex);
}

ulong SimpleFormatter::GetMaxColumnCount() const
{
	return pimpl_->GetMaxColumnCount();
}

LineFormat_ptr SimpleFormatter::GetLine(ulong lineindex)
{
	return pimpl_->GetLine(lineindex);
}

LineFormat_ptr SimpleFormatter::GetHeader(ulong lineindex)
{
	return pimpl_->GetHeader(lineindex);
}

ulong SimpleFormatter::LineIndex2Offset(ulong lineindex) const
{
	return pimpl_->LineIndex2Offset(lineindex);
}

ulong SimpleFormatter::Offset2LineIndex(ulong offset) const
{
	return pimpl_->Offset2LineIndex(offset);
}
