#include "stdafx.h"
#include "IOBase.h"
#include <algorithm>

class FullFileBuffer::Impl
{
public:
	ulong Get( ulong offset, ulong size, uchar *buffer );
	bool Insert( ulong offset, ulong size, uchar *buffer );
	bool OverStrike( ulong offset, ulong size, uchar *buffer );
	bool Remove( ulong offset, ulong size );
	ulong Size() const;

	bool Load( Reader_ptr reader );
	bool Save( Writer_ptr writer );

	const uchar* Begin();
	const uchar* End();
	const uchar* Increment(const uchar*, ulong n);
	const uchar* Decrement(const uchar*, ulong n);
	bool IsEqual(const uchar*, const uchar*);

private:
	std::vector<uchar> memory_;
};

ulong FullFileBuffer::Impl::Get( ulong offset, ulong size, uchar *buffer)
{
	ulong memorysize = memory_.size();
	if(memorysize<=offset){
		return 0;
	}

	ulong readablesize = min(memorysize-offset,size);
	memcpy(buffer,&memory_[offset],readablesize);
	ulong readsize = readablesize;

	return readsize;
}

bool FullFileBuffer::Impl::Insert( ulong offset, ulong size, uchar *buffer )
{
	memory_.insert(memory_.begin()+offset,buffer,buffer+size);
	return true;
}

bool FullFileBuffer::Impl::OverStrike( ulong offset, ulong size, uchar *buffer )
{
	if(offset+size>memory_.size()){
		ulong out = offset+size-memory_.size();
		ulong in = size-out;
		std::copy(buffer,buffer+in,memory_.begin()+offset);
		memory_.insert(memory_.end(),buffer+in,buffer+size);
	}else{
		std::copy(buffer,buffer+size,memory_.begin()+offset);
	}
	return true;
}

bool FullFileBuffer::Impl::Remove( ulong offset, ulong size )
{
	memory_.erase(memory_.begin()+offset,memory_.begin()+(offset+size));
	return true;
}

ulong FullFileBuffer::Impl::Size() const
{
	return memory_.size();
}

const uchar* FullFileBuffer::Impl::Begin()
{
	return &(memory_[0]);
}

const uchar* FullFileBuffer::Impl::End()
{
	return Begin()+memory_.size();
}

const uchar* FullFileBuffer::Impl::Increment(const uchar* p, ulong n)
{
	p += n;
	return p;
}

const uchar* FullFileBuffer::Impl::Decrement(const uchar* p, ulong n)
{
	p -= n;
	return p;
}

bool FullFileBuffer::Impl::IsEqual(const uchar* lhs, const uchar* rhs)
{
	return lhs==rhs;
}

bool FullFileBuffer::Impl::Load( Reader_ptr reader )
{
	ulong size = reader->Size();
	if(size==0){
		return true;
	}

	memory_.resize(size);
	
	ulong readsize;
	if(reader->Read(0,size,&memory_[0],readsize)!=IReader::success){
		return false;
	}

	return true;
}

bool FullFileBuffer::Impl::Save( Writer_ptr writer )
{
	return writer->Write(0,Size(),&memory_[0]);
}



//implementation of FullFileBuffer
FullFileBuffer::FullFileBuffer()
	: pimpl_(new Impl)
{
}

ulong FullFileBuffer::Get( ulong offset, ulong size, uchar *buffer )
{
	return pimpl_->Get(offset,size,buffer);
}

bool FullFileBuffer::Insert( ulong offset, ulong size, uchar *buffer )
{
	return pimpl_->Insert(offset,size,buffer);
}

bool FullFileBuffer::OverStrike( ulong offset, ulong size, uchar *buffer )
{
	return pimpl_->OverStrike(offset,size,buffer);
}

bool FullFileBuffer::Remove( ulong offset, ulong size )
{
	return pimpl_->Remove(offset,size);
}

ulong FullFileBuffer::Size() const
{
	return pimpl_->Size();
}

const uchar* FullFileBuffer::Begin()
{
	return pimpl_->Begin();
}

const uchar* FullFileBuffer::End()
{
	return pimpl_->End();
}

const uchar* FullFileBuffer::Increment(const uchar* p, ulong n)
{
	return pimpl_->Increment(p,n);
}

const uchar* FullFileBuffer::Decrement(const uchar* p, ulong n)
{
	return pimpl_->Decrement(p,n);
}

bool FullFileBuffer::IsEqual(const uchar* lhs, const uchar* rhs)
{
	return pimpl_->IsEqual(lhs,rhs);
}

bool FullFileBuffer::Load(Reader_ptr reader)
{
	return pimpl_->Load(reader);
}

bool FullFileBuffer::Save(Writer_ptr writer)
{
	return pimpl_->Save(writer);
}
