
#==============================================================================#
# $Id: executable.rb,v 1.10 2003/11/17 17:45:49 yuya Exp $
#==============================================================================#

require 'exerb/utility'
require 'exerb/archive'
require 'exerb/error'

#==============================================================================#

module Exerb

  class Executable

    def initialize(core, archive, kcode = 'none')
      @core    = core
      @archive = archive
      @kcode   = kcode
    end

    attr_accessor :core, :archive, :kcode

    def pack
      packed_core    = @core.pack
      packed_archive = @archive.pack

      case @kcode.downcase.delete('-')
      when 'n', 'none' then options = ExerbHeader::OPTIONS_KCODE_NONE
      when 'e', 'euc'  then options = ExerbHeader::OPTIONS_KCODE_EUC
      when 's', 'sjis' then options = ExerbHeader::OPTIONS_KCODE_SJIS
      when 'u', 'utf8' then options = ExerbHeader::OPTIONS_KCODE_UTF8
      else raise(ExerbError, "unkown kanji code [#{@kcode}]")
      end

      exerb_header = ExerbHeader.new
      exerb_header.signature1           = ExerbHeader::SIGNATURE1
      exerb_header.signature2           = ExerbHeader::SIGNATURE2
      exerb_header.offset_of_archive    = packed_core.size
      exerb_header.options              = options

      return packed_core + packed_archive + exerb_header.pack
    end

    def output(io)
      io.write(self.pack)
    end

    def output_to_file(filepath)
      File.open(filepath, 'wb') { |file| self.output(file) }
      File.chmod(0755, filepath)
    end

  end # Executable

  class Core

    def initialize(binary)
      @binary = binary
      self.check
    end

    def self.new_from_file(path)
      return File.open(path, 'rb') { |file| self.new(file.read) }
    end

    def check
      if @binary[0, 2] != 'MZ'
        raise(ExerbError, 'the dos header of the core has invalid signature')
      end

      if @binary[-ExerbHeader.new.pack.size, 5] == 'EXERB'
        raise(ExerbError, 'the core is already joined with an archive')
      end
    end

    def pack
      return Utility.alignment16(@binary)
    end

  end # Core

  class ExerbHeader

    SIGNATURE1         = 0x52455845
    SIGNATURE2         = 0x01000042
    OPTIONS_KCODE_NONE = 0x00000000
    OPTIONS_KCODE_EUC  = 0x00000001
    OPTIONS_KCODE_SJIS = 0x00000002
    OPTIONS_KCODE_UTF8 = 0x00000003

    def initialize
      @signature1           = nil
      @signature2           = nil
      @offset_of_archive    = nil
      @options              = nil
    end

    attr_accessor :signature1, :signature2, :offset_of_archive, :options

    def pack
      return [@signature1, @signature2, @offset_of_archive, @options].pack('LLLL')
    end

  end # ExerbHeader

end # Exerb

#==============================================================================#
#==============================================================================#
