
#==============================================================================#
# $Id: executable.rb,v 1.12 2003/11/22 22:17:15 yuya Exp $
#==============================================================================#

require 'stringio'
require 'exerb/utility'
require 'exerb/archive'
require 'exerb/error'
require 'exerb/win32/pe_file'

#==============================================================================#

module Exerb

  class Executable

    def initialize(core, archive, kcode = 'none')
      @core    = core
      @archive = archive
      @kcode   = kcode
    end

    attr_accessor :core, :archive, :kcode

    def pack
      packed_core    = @core.pack
      packed_archive = @archive.pack

      case @kcode.downcase.delete('-')
      when 'n', 'none' then options = ExerbHeader::OPTIONS_KCODE_NONE
      when 'e', 'euc'  then options = ExerbHeader::OPTIONS_KCODE_EUC
      when 's', 'sjis' then options = ExerbHeader::OPTIONS_KCODE_SJIS
      when 'u', 'utf8' then options = ExerbHeader::OPTIONS_KCODE_UTF8
      else raise(ExerbError, "unkown kanji code [#{@kcode}]")
      end

      exerb_header = ExerbHeader.new
      exerb_header.signature1        = ExerbHeader::SIGNATURE1
      exerb_header.signature2        = ExerbHeader::SIGNATURE2
      exerb_header.offset_of_archive = packed_core.size
      exerb_header.options           = options

      return packed_core + packed_archive + exerb_header.pack
    end

    def write(io)
      io.write(self.pack)
    end

    def write_to_file(filepath)
      File.open(filepath, 'wb') { |file| self.write(file) }
      File.chmod(0755, filepath)
    end

  end # Executable

  class Core

    def initialize(core, rsrc = nil)
      @core = core
      @rsrc = rsrc

      self.check
    end

    attr_accessor :core, :rsrc

    def self.new_from_file(path, rsrc = nil)
      return File.open(path, 'rb') { |file| self.new(file.read, rsrc) }
    end

    def check
      if @core[0, 2] != 'MZ'
        raise(ExerbError, 'the dos header of the core has invalid signature')
      end

      if @core[-ExerbHeader.new.pack.size, 5] == 'EXERB'
        raise(ExerbError, 'the core is already joined with an archive')
      end
    end

    def pack
      if @rsrc
        pe_file = Win32::PeFile.read(StringIO.new(@core))
        fh  = pe_file.nt_headers.file_header
        oh  = pe_file.nt_headers.optional_header
        rsh = pe_file.sections.last

        if rsh.name != '.rsrc'
          raise(ExerbError, "the last section isn't resource section")
        end

        packed_rsrc  = @rsrc.pack(rsh.virtual_address)
        aligned_rsrc = Utility.alignment4k(packed_rsrc)

        @core[rsh.pointer_to_raw_data, rsh.size_of_raw_data] = aligned_rsrc

        fh.time_date_stamp                 = Time.now.to_i
        oh.size_of_initialized_data        = oh.size_of_initialized_data - rsh.size_of_raw_data + aligned_rsrc.size
        oh.size_of_image                   = oh.size_of_image            - rsh.size_of_raw_data + aligned_rsrc.size
        oh.resource_directory_virtual_size = packed_rsrc.size
        rsh.virtual_size                   = packed_rsrc.size
        rsh.size_of_raw_data               = aligned_rsrc.size

        @core[fh.position,  fh.size]  = fh.pack
        @core[oh.position,  oh.size]  = oh.pack
        @core[rsh.position, rsh.size] = rsh.pack
      end

      return Utility.alignment16(@core)
    end

  end # Core

  class ExerbHeader

    SIGNATURE1         = 0x52455845
    SIGNATURE2         = 0x01000042
    OPTIONS_KCODE_NONE = 0x00000000
    OPTIONS_KCODE_EUC  = 0x00000001
    OPTIONS_KCODE_SJIS = 0x00000002
    OPTIONS_KCODE_UTF8 = 0x00000003

    def initialize
      @signature1        = nil
      @signature2        = nil
      @offset_of_archive = nil
      @options           = nil
    end

    attr_accessor :signature1, :signature2, :offset_of_archive, :options

    def pack
      return [@signature1, @signature2, @offset_of_archive, @options].pack('LLLL')
    end

  end # ExerbHeader

end # Exerb

#==============================================================================#
#==============================================================================#
