
#==============================================================================#
# $Id: resource.rb,v 1.22 2003/11/28 09:49:42 yuya Exp $
#==============================================================================#

require 'stringio'
require 'exerb/version'
require 'exerb/error'
require 'exerb/win32/resource_directory'
require 'exerb/win32/resource_directory_root'
require 'exerb/win32/resource_entry'
require 'exerb/win32/resource_entry_data'
require 'exerb/win32/pe_file'
require 'exerb/win32/const/resource'
require 'exerb/resource/icon'
require 'exerb/resource/group_icon'
require 'exerb/resource/version_info'
require 'exerb/resource/dialog'
require 'exerb/resource/binary'

#==============================================================================#

module Exerb

  class Resource

    include Enumerable

    DEFAULT_LANG_ID = 1041 # Japanese
    RT_EXERB = 100
    ID_EXERB = 100

    def initialize
      @entries = {}
    end

    attr_reader :entries

    def self.read(io, base)
      return self.new.read(io, base)
    end

    def self.new_from_binary(bin, base)
      return self.read(StringIO.new(bin), base)
    end

    def self.new_from_pe_binary(bin)
      pe     = Win32::PeFile.new_from_binary(bin)
      rsrc   = pe.sections.find { |sec| sec.name == '.rsrc' }
      raise(ExerbError, "a resource section was not found in the core") if rsrc.nil?

      return self.new_from_binary(bin[rsrc.pointer_to_raw_data, rsrc.virtual_size], rsrc.virtual_address)
    end

    def self.new_from_pe_file(filepath)
      return File.open(filepath) { |file| self.new_from_pe_binary(file.read) }
    end

    def each
      @entries.keys.sort.each { |type|
        @entries[type].keys.sort.each { |id|
          @entries[type][id].keys.sort.each { |lang|
            yield(@entries[type][id][lang])
          }
        }
      }
    end

    def add(type, id, data, lang = DEFAULT_LANG_ID)
      @entries[type] ||= {}
      @entries[type][id] ||= {}
      @entries[type][id][lang] = Entry.new(type, id, lang, data)

      return self
    end

    def add_icon(id, data, lang = DEFAULT_LANG_ID)
      return self.add(Win32::Const::RT_ICON, id, data, lang)
    end

    def add_group_icon(id, data, lang = DEFAULT_LANG_ID)
      return self.add(Win32::Const::RT_GROUP_ICON, id, data, lang)
    end

    def add_dialog(id, data, lang = DEFAULT_LANG_ID)
      return self.add(Win32::Const::RT_DIALOG, id, data, lang)
    end

    def add_version(id, data, lang = DEFAULT_LANG_ID)
      return self.add(Win32::Const::RT_VERSION, id, data, lang)
    end

    def add_default_icon
      ico16x16x16  = DefaultIcon16x16x16.new
      ico16x16x256 = DefaultIcon16x16x256.new
      ico32x32x16  = DefaultIcon32x32x16.new
      ico32x32x256 = DefaultIcon32x32x256.new

      group_icon = GroupIcon.new
      group_icon.add(1, ico32x32x256)
      group_icon.add(2, ico32x32x16)
      group_icon.add(3, ico16x16x256)
      group_icon.add(4, ico16x16x16)

      self.add_icon(1, ico32x32x256)
      self.add_icon(2, ico32x32x16)
      self.add_icon(3, ico16x16x256)
      self.add_icon(4, ico16x16x16)
      self.add_group_icon(102, group_icon)
    end

    def add_default_dialog
      self.add_default_dialog_en
      self.add_default_dialog_ja
    end

    def add_default_dialog_en
      self.add_dialog(101, DefaultDialogEn.new, 1033)
    end

    def add_default_dialog_ja
      self.add_dialog(101, DefaultDialogJa.new)
    end

    def add_default_version_info
      return self.add_version(1, DefaultVersionInfo.new)
    end

    def add_archive(archive)
      return self.add(RT_EXERB, ID_EXERB, archive)
    end

    def pack(base, reloc = [])
      root_dir = Win32::ResourceDirectoryRoot.new

      @entries.keys.sort.each { |type|
        root_dir << Win32::ResourceDirectory.new(type) { |type_dir|
          @entries[type].keys.sort.each { |id|
            type_dir << Win32::ResourceDirectory.new(id) { |item_dir|
              @entries[type][id].keys.sort.each { |lang|
                item_dir << @entries[type][id][lang].to_resource_entry
              }
            }
          }
        }
      }

      return root_dir.pack_all(base, reloc)
    end

    def read(io, base)
      root_dir = Win32::ResourceDirectoryRoot.read(io, base)
      root_dir.entries.each { |type_dir|
        type_dir.entries.each { |item_dir|
          item_dir.entries.each { |item|
            type, id, lang = type_dir.name, item_dir.name, item.lang
            @entries[type] ||= {}
            @entries[type][id] ||= {}
            @entries[type][id][lang] = Entry.new(type, id, lang, Binary.new(item.entry_data.data))
          }
        }
      }

      return self
    end

    def merge(res)
      res.each { |entry|
        @entries[entry.type] ||= {}
        @entries[entry.type][entry.id] ||= {}
        @entries[entry.type][entry.id][entry.lang] = entry
      }

      return self
    end

    class Entry

      def initialize(type, id, lang, data)
        @type = type
        @id   = id
        @lang = lang
        @data = data
      end

      attr_accessor :type, :id, :lang, :data

      def to_resource_entry
        return Win32::ResourceEntry.new(Win32::ResourceEntryData.new(@data.pack), lang)
      end

    end # Entry

  end # Resource

end # Exerb

#==============================================================================#
#==============================================================================#
