
#==============================================================================#
# $Id: recipe.rb,v 1.9 2004/01/20 21:12:41 yuya Exp $
#==============================================================================#

require 'exerb/archive'
require 'exerb/executable'
require 'exerb/utility'
require 'exerb/error'

#==============================================================================#

module Exerb

  class Recipe

    def initialize(text, basedir)
      @basedir = basedir
      @archive = Archive.new

      @corefile = nil
      @outfile  = nil

      lineno = 0

      text.collect { |line|
        [lineno += 1, line.strip]
      }.reject { |lineno, line|
        line.empty? || /^#/ =~ line
      }.collect { |lineno, line|
        [lineno, line.split(/\t+/)]
      }.each { |line, (type, *arg)|
        case type.downcase
        when 'set_kcode'              then cmd_set_kcode(arg[0], line)
        when 'set_core_by_name'       then cmd_set_core_by_name(arg[0], line)
        when 'set_core_by_filename'   then cmd_set_core_by_filename(arg[0], line)
        when 'set_core_by_filepath'   then cmd_set_core_by_filepath(arg[0], line)
        when 'set_out_filepath'       then cmd_set_out_filepath(arg[0], line)
        when 'add_ruby_script'        then cmd_add_ruby_script(arg[0], arg[1], line)
        when 'add_bruby_binary'       then cmd_add_bruby_binary(arg[0], arg[1], line)
        when 'add_extension_library'  then cmd_add_extension_library(arg[0], arg[1], arg[2], line)
        when 'add_dynamic_library'    then cmd_add_dynamic_library(arg[0], line)
        when 'add_resource_library'   then cmd_add_resource_library(arg[0], line)
        when 'add_data_binary'        then cmd_add_data_binary(arg[0], arg[1], line)
        when 'add_plugin_by_name'     then cmd_add_plugin_by_name(arg[0], line)
        when 'add_plugin_by_filename' then cmd_add_plugin_by_filename(arg[0], line)
        when 'add_plugin_by_filepath' then cmd_add_plugin_by_filepath(arg[0], line)
        else raise(ExerbError, "unknown directive '#{type}' in line #{line}")
        end
      }
    end

    attr_reader :basedir, :archive, :corefile, :outfile

    def self.new_from_file(filepath)
      text    = File.open(filepath) { |file| file.read }
      basedir = File.dirname(File.expand_path(filepath))
      return self.new(text, basedir)
    end

    def expand_filepath(filename, filepath, line)
      path = File.expand_path(filepath || filename, @basedir)
      raise(ExerbError, "no such file '#{path}' in line #{line}") unless File.exist?(path)
      return path
    end
    protected :expand_filepath

    def cmd_set_kcode(kcode, line)
      unless ['n', 'none', 'e', 'euc', 's', 'sjis', 'u', 'utf8'].include?(kcode.downcase.delete('-'))
        raise(ExerbError, "unknown kanji code '#{kcode}' in line #{line}")
      end
      @archive.kcode = kcode
    end

    def cmd_set_core_by_name(name, line)
      @corefile = Utility.find_core_by_name(name)
      raise(ExerbError, "unknown core name '#{name}' in line #{line}")      unless @corefile
      raise(ExerbError, "no such core file '#{@corefile}' in line #{line}") unless File.exist?(@corefile)
    end

    def cmd_set_core_by_filename(filename, line)
      @corefile = Utility.find_core_by_filename(filename)
      raise(ExerbError, "unknown core file name '#{filename}' in line #{line}") unless @corefile
      raise(ExerbError, "no such core file '#{@corefile}' in line #{line}")     unless File.exist?(@corefile)
    end

    def cmd_set_core_by_filepath(filepath, line)
      @corefile = File.expand_path(filepath, @basedir)
      raise(ExerbError, "no such core file '#{@corefile}' in line #{line}") unless File.exist?(@corefile)
    end

    def cmd_set_out_filepath(filepath, line)
      @outfile = File.expand_path(filepath, @basedir)
      outdir   = File.dirname(@outfile)
      raise(ExerbError, "no such directory '#{outdir}' in line #{line}") unless File.directory?(outdir)
    end

    def cmd_add_ruby_script(filename, filepath, line)
      @archive.add_ruby_script_from_file(filename, expand_filepath(filename, filepath, line))
    end

    def cmd_add_bruby_binary(filename, filepath, line)
      @archive.add_bruby_binary_from_file(filename, expand_filepath(filename, filepath, line))
    end

    def cmd_add_extension_library(filename, filepath, option, line)
      options = option.to_s.downcase.split(/,/)
      flag    = 0
      flag   |= FileTable::Entry::FLAG_NO_REPLACE_FUNCTION if options.delete('no_replace_function')
      raise(ExerbError, "unknown options #{options.inspect} in line #{line}") unless options.empty?
      @archive.add_extension_library_from_file(filename, expand_filepath(filename, filepath, line), flag)
    end

    def cmd_add_dynamic_library(filepath, line)
      path = File.expand_path(filepath, @basedir)
      raise(ExerbError, "no such file '#{path}' in line #{line}") unless File.exist?(path)
      @archive.add_dynamic_library_from_file(path)
    end

    def cmd_add_resource_library(filepath, line)
      path = File.expand_path(filepath, @basedir)
      raise(ExerbError, "no such file '#{path}' in line #{line}") unless File.exist?(path)
      @archive.add_resource_library_from_file(path)
    end

    def cmd_add_data_binary(internal_name, filepath, line)
      @archive.add_data_binary_from_file(internal_name, expand_filepath(internal_name, filepath, line))
    end

    def cmd_add_plugin_by_name(name, line)
      path = Utility.find_plugin_by_name(name)
      raise(ExerbError, "unknown plugin name '#{name}' in line #{line}") unless path
      raise(ExerbError, "no such plugin file '#{path}' in line #{line}") unless File.exist?(path)
      @archive.add_plugin_from_file(path)
    end

    def cmd_add_plugin_by_filename(filename, line)
      path = Utility.find_plugin_by_filename(filename)
      raise(ExerbError, "unknown plugin file name '#{filename}' in line #{line}") unless path
      raise(ExerbError, "no such plugin file '#{path}' in line #{line}")          unless File.exist?(path)
      @archive.add_plugin_from_file(path)
    end

    def cmd_add_plugin_by_filepath(filepath, line)
      path = File.expand_path(filepath, @basedir)
      raise(ExerbError, "no such file '#{path}' in line #{line}") unless File.exist?(path)
      @archive.add_plugin_from_file(path)
    end

  end # Recipe

end # Exerb

#==============================================================================#
#==============================================================================#
