require 'gtk2'
require 'kconv'
require 'open3'
require 'zlib'
require 'logger'
require 'gconf2'

require 'remastering/remastertool_const'

class RTUtility

	MSG_ERR_INITIALIZE = "Cannot initialize doubly."

	# for logging
	CONF_KEY_LOG_DIR = "/apps/remasteringtool/log/logDirectory"
	CONF_KEY_LOG_LEVEL = "/apps/remasteringtool/log/logLevel"
	LOGFILE = "remastertool_" + $$.to_s + ".log"	# ログファイル
	LOG_LV_HASH = { 0 => Logger::UNKNOWN, 1 => Logger::ERROR, 2 => Logger::WARN, 3 => Logger::DEBUG }

	@@mainObj	= nil
	@@mainWindow = nil

	@@waitWindow = nil
	@@apt_conf = Hash.new {|hash, key| hash[key] = ""}
	@@initialized = false
	@@modified = false
	@@plugins = nil
	@@nativePackage	= nil
	@@image_dir = nil
	@@work_dir = nil
	@@logger = Logger.new("/dev/null")	# loggerインスタンス
	@@logger_dir = ""					# logディレクトリ
	@@logger_lv = 0						# logレベル

	CMD_APTCONFIG = "apt-config dump"	# for get apt configuration
	RMTOOL_BAK_EXT = ".remasterbak"

	bindtextdomain(PROG_NAME, nil, nil, "UTF-8")

	# 画面表示リテラル
	ABOUT_NAME_LITERAL = _("About %s plugin")
	OPTION_NAME_LITERAL = _("Setting %s plugin")

public
    #=== リマスタリングツール本体のインスタンスを設定する
    #
    #リマスタリングツールの#初期化を行う。
    #
	def RTUtility.setObjs(obj, wnd)
		if @@mainObj
			@@logger.error MSG_ERR_INITIALIZE if @@logger
			raise MSG_ERR_INITIALIZE
		end
		@@mainObj	= obj
		@@mainWindow = wnd
	end

    #=== プロセスで一意なロガーインスタンスの取得
    #
    #メインウインドウのオブジェクトからloggerインスタンスを取得して返却する
	#
	#復帰値:: loggerインスタンス
    #
	def RTUtility.get_logger
		@@logger
	end

    #=== プロセスで一意なロガーインスタンスの作成
    #
    #オプション設定で設定したログ情報、または、指定された情報からログインスタンスを作成する
	#
	#dir::ログ出力先ディレクトリ(nilの場合はgconfから取得する）
	#level::ログ出力レベル(nilの場合はgconfから取得する）
	#
	#復帰値:: なし
    #
	def RTUtility.set_logger(dir = nil, level = nil)
		gconf = GConf::Client.default
		@@logger_dir = dir
		@@logger_lv = level

		# ログディレクトリの決定
		if @@logger_dir.nil?
			if gconf[CONF_KEY_LOG_DIR]
				@@logger_dir = gconf[CONF_KEY_LOG_DIR]
			else
				@@logger_dir = LOGGING_DIR
			end
		end
		# ログ出力レベルの決定
		if @@logger_lv.nil?
			if gconf[CONF_KEY_LOG_LEVEL]
				@@logger_lv = gconf[CONF_KEY_LOG_LEVEL]
			else
				@@logger_lv = 0
			end
		end

		# logの初期化
		if @@logger_lv > 0
			@@logger = Logger.new(File.join(@@logger_dir, LOGFILE))
			@@logger.level = LOG_LV_HASH[@@logger_lv]
		else
			# ログを出力しない場合は既存のログをクローズして /dev/null にログファイルを設定する。
			# このときなるべくIOが発生しないように最高レベルのログのみ出力するようレベルを設定しておく
			@@logger.close if @@logger
			@@logger = Logger.new("/dev/null")
			@@logger.level = Logger::FATAL
		end

		# 引数のログディレクトリ、ログレベルの双方が指定された場合は設定内容の変更なのでgconfに設定する
		if dir && level
			gconf[CONF_KEY_LOG_DIR] = @@logger_dir
			gconf[CONF_KEY_LOG_LEVEL] = @@logger_lv
		end
	end

    #=== ログオプションの取得
    #
    #オプション設定で設定したログ出力先、ログ出力レベルを返却する
	#
	#復帰値:: Array [ ログ出力ディレクトリ, ログ出力レベル ]
    #
	def RTUtility.get_log_option
		set_logger unless @@logger

		[ @@logger_dir, @@logger_lv ]
	end
	#=== プラグインのロード
	#
	#プラグインディレクトリ(remastering/pugin)下のプラグインをロードする
	#
	#plugin_type::ロードするプラグインのタイプ。ずべてのタイプが対象の場合、nilを指定する
	#復帰値:: なし
	#
	def RTUtility.load_plugin(plugin_type = nil)
		return if @@plugins
		# プラグインハッシュを作成
		hash = {PLUGIN_TYPE_OS 		 => Hash.new,	PLUGIN_TYPE_MEDIA	=> Hash.new,
				PLUGIN_TYPE_PACKAGE  => Hash.new,	PLUGIN_TYPE_TEST 	=> Hash.new,
				PLUGIN_TYPE_EMULATOR => Hash.new,	PLUGIN_TYPE_EXPORT	=> Hash.new}

		check_hash = Hash.new
		# get remastering tool plugin directory
		$:.each { | x |
			dir_name = File.join(x, REMASTERING_MODULE_DIR, PLUGIN_DIR)
			next unless File.directory? dir_name
			Dir[File.join(dir_name, "*")].sort.each { | plugin|
				# loading plugin
				plgin_name = File.join(plugin, File.basename(plugin)) + ".rb"
				plgin_class = File.basename(plgin_name, ".rb").capitalize
				if check_hash[plgin_class] == nil && File.readable?(plgin_name)
					check_hash[plgin_class] = true
					$:.push File.dirname(plgin_name)
#					load plgin_name
					require plgin_name
					# Make class name
					# create plugin object
					plgin_obj = Object.const_get(plgin_class).new(File.dirname(plgin_name))
					# Get plugin type & store plugin object to hash(key=displayName, value=object)
					if (plgin_obj.start_plugin(PROG_VERSION) == false) ||
							(plugin_type  && plgin_obj.get_plugin_type != plugin_type)
						plgin_obj.end_plugin
						$:.pop
						next
					end
					(hash[plgin_obj.get_plugin_type])[plgin_obj.get_plugin_name] = plgin_obj
				end
			}
		}

		@@plugins = hash
	end

	#=== プラグインの終了
	#
	#ロード済の全プラグインの終了処理を呼び出す
	#
	#復帰値:: なし
	#
	def RTUtility.unload_plugin
		@@plugins.each_value{|hash|
			hash.each_value { |obj| obj.end_plugin }
		}
		@@plugins = nil
	end

    #=== OSプラグインインスタンス格納ハッシュの取得
    #
    #OSプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
    #
    #復帰値::OSプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_os_plugins
		@@plugins[PLUGIN_TYPE_OS]
	end

    #=== メディアプラグインインスタンス格納ハッシュの取得
    #
    #メディアプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
    #
    #復帰値::メディアプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_media_plugins
		@@plugins[PLUGIN_TYPE_MEDIA]
	end

    #=== パッケージプラグインインスタンス格納ハッシュの取得
    #
    #パッケージプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
    #
    #復帰値::パッケージプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_package_plugins
		@@plugins[PLUGIN_TYPE_PACKAGE]
	end

    #=== テストプラグインインスタンス格納ハッシュの取得
    #
    #テストプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
    #
    #復帰値::テストプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_test_plugins
		@@plugins[PLUGIN_TYPE_TEST]
	end

	#=== エミュレータプラグインインスタンス格納ハッシュの取得
	#
	#エミュレータプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
	#
	#復帰値::エミュレータプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_emulator_plugins
		@@plugins[PLUGIN_TYPE_EMULATOR]
	end

	#=== エエクスポートプラグインインスタンス格納ハッシュの取得
	#
	#エクスポートプラグインのインスタンスを格納しているハッシュオブジェクトを返却する。
	#
	#復帰値::エクスポートプラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_export_plugins
		@@plugins[PLUGIN_TYPE_EXPORT]
	end

	#=== リマスタリング対象OSプラグインの設定
	#
	#リマスタリング対象となるOSのプラグインインスタンスをインスタンス変数に設定する。
	#
	#name::リマスタリング対象OSのプラグイン表示名
	#復帰値::なし
	#
	def RTUtility.set_os_plugin(name)
		if /^(.+?):(\d+)(?::in `(.*)')?/ =~ caller.first && File.basename($1) == "remastertool_init.rb"
			@@remasterOs = @@plugins[PLUGIN_TYPE_OS][name]
			@@initialized = name ? true	: false		# OS plugin is specified. then initializing is done.
			@@mainObj.change_pane(nil) unless @@initialized
		end
	end

	#=== OSプラグインインスタンスの取得
	#
	#リマスタリング対象OSに対応するOSプラグインのインタンスを返却する。
	#
	#復帰値::OSプラグインインスタンス
	#
	def RTUtility.get_os_plugin
		@@remasterOs
	end

	#=== ネイティブパッケージプラグインの設定
	#
	#リマスタリング対象となるOSのネイティブパッケージプラグインのインスタンスを設定する。
	#
	#name::パッケージプラグイン表示名
	#復帰値::なし
	#
	def RTUtility.set_native_pkg_plugin(name)
		@@nativePackage =  @@plugins[PLUGIN_TYPE_PACKAGE][name]
	end

	#===ネイティブパッケージプラグインのインスタンスの取得
	#
	#リマスタリング対象OSに対応するネイティブパッケージプラグインのインタンスを返却する。
	#
	#復帰値::ネイティブパッケージプラグインインスタンス
	#
	def RTUtility.get_native_pkg_plugin
		@@nativePackage
	end

	#===全プラグインのインスタンスの取得
	#
	#全プラグインのインスタンスを格納しているハッシュオブジェクトを返却する。リ
	#
	#復帰値::全プラグインのインスタンスを格納しているハッシュオブジェクト
	#
	def RTUtility.get_plugins
		@@plugins
	end

    #=== イメージファイル格納ディレクトリの取得
    #
    #リマスタリングツールの#初期化を行う。
    #
    #復帰値::イメージファイル格納ディレクトリ
    #
    def RTUtility.get_image_dir
		return @@image_dir if @@image_dir
		# get image file directory and save to instance variable
		$:.each { | x |
			dir_name = File.join(x, REMASTERING_MODULE_DIR, IMAGE_DIR)
			next unless File.directory? dir_name
			@@image_dir = dir_name
			break
		}
		@@image_dir
	end

    #=== gladeファイル名格納ディレクトリの取得
    #
    #リマスタリングツールのgladeファイル名格納ディレクトリを取得する。
    #
    #復帰値:: gladeファイルの格納ディレクトリ
    #        gladeファイル格納ディレクトリが見付からない場合nilを返却する。
    #
    def RTUtility.get_glade_dir(prog_path)
        $:.each { |x|
            name = File.join(x, REMASTERING_MODULE_DIR, GLADE_DIR)
            return name if FileTest.readable?(File.join(name, prog_path))
         }
        nil
    end

	#=== リマスタリング作業で使用している作業用ディレクトリの取得
	#
	#リマスタリング作業で使用している作業用ディレクトリ返却する。
	#
	#復帰値:: リマスタリング作業で使用している作業用ディレクトリ
	#
	def RTUtility.get_work_dir
		@@work_dir
	end

	#=== リマスタリング作業で使用している作業用ディレクトリの設定
	#
	#リマスタリング作業で使用している作業用ディレクトリを設定する。
	#
	#dir:: 作業用ディレクトリ
	#復帰値:: なし
	#
	def RTUtility.set_work_dir(dir)
		@@work_dir = dir
	end

    #=== メッセージの出力
    #
    #メッセージウインドウに指定メッセージを追加表示する。メッセージウインドウがない場合、
	#起動端末に出力する。
    #
    #復帰値::イメージファイル格納ディレクトリ
    #
	def RTUtility.addMessage(msg)
		if @@mainObj
			@@mainObj.addMessage(msg) 
		else
			puts msg
		end
	end

    #=== メインウインドウのインスタンスの取得
    #
    #メインウインドウのインスタンスを返却する。
    #メッセージダイアログなどの親ウインドウを指定する場合に使用する。
    #
    #復帰値::メインウインドウのインスタンス
    #
	def RTUtility.mainWindow
		@@mainWindow
	end

    #=== サイドバーボタンのenable/disableの切替え
    #
    #サイドバーボタンのenable/disableの切替えを行う。
    #注）disable指定されてもすべてのボタンがdisableになるわけではなく、
	#初期状態（初期設定、パッケージ作成、仮想マシンテストツール起動のボタン
	#はenableのままになる。
    #
    #enable::サイドバーボタンの状態(true=enable指定、false=disable指定)
    #復帰値::なし
	#
	def RTUtility.enable_sidebar_button(enable)
		@@mainObj.enable_sidebar_button enable
	end

    #=== 初期設定の完了確認
    #
    #初期設定処理が完了しているか否かを返却する．
    #
    #復帰値::初期設定の完了状態(true=完了, false=未)
	#
	def RTUtility.isInitialized?
		@@initialized
	end

    #=== お待ちくださいウインドウの表示
    #
    #利用者に処理を実行中であることを示すおまちくださいウインドウを表示する．
    #
    #復帰値::なし
	#
	def RTUtility.show_wait_window(parent, text)
		if @@waitWindow == nil
			begin
				# prepare processing dialog
				style = Gtk::Style.new
				style.font_desc = Pango::FontDescription.new("Sans 24")
				style.set_bg(Gtk::STATE_ACTIVE, 0, 65535, 65535)
				image_path = get_image_dir
				if image_path
					if  File.readable?(File.join(image_path, IMAGE_WAIT))
						anim = Gdk::PixbufAnimation.new(File.join(image_path, IMAGE_WAIT))
						img  = Gtk::Image.new(anim)
					else
						image_path = nil
					end
				end
				@@label = Gtk::Label.new(text)
				@@label.style = style
				hbox = Gtk::HBox.new(false, 10)
				hbox.pack_start(@@label, true, true, 40)
				hbox.pack_start(img, false, true, 0) if image_path
				window = Gtk::Window.new
				window.add(hbox)
				window.set_transient_for(parent)
				window.set_modal(true)
				window.title = PROG_HELP_NAME
				window.set_window_position(Gtk::Window::POS_CENTER_ON_PARENT)
				window.signal_connect("delete-event") { true }
				@@waitWindow = window
			rescue
#				puts $!.to_s
			end
		else
			@@waitWindow.resize(10, 10)		# サイズ調整
			@@label.text = text
		end
		@@waitWindow.show_all
		mainWindow.window.set_cursor(Gdk::Cursor.new(Gdk::Cursor::WATCH))
	end

    #=== お待ちくださいウインドウの終了
    #
    #利用者に処理を実行中であることを示すおまちくださいウインドウを終了する．
    #
    #復帰値::なし
	#
	def RTUtility.end_wait_window
		if @@waitWindow
			@@waitWindow.hide
			mainWindow.window.set_cursor(nil)
		end
	end

	#=== リマスタリング対象OS上でのOSコマンド実行
	#
	#リマスタリング対象OS上で指定されたOSコマンドを実行し、結果を配列に格納して返却する。
	#
	#cmd:: 実行するするOSコマンド
	#parent:: お待ちくださいwindowを表示する際にお待ちくださいwindowの親windowとなるwidget
	#          (お待ちくださいwindowが不要な場合はnilを指定する)
	#wait_msg:: お待ちくださいwindowに表示するメッセージを指定する（parent引数がnilの場合は参照されない)
	#復帰値:: 実行結果配列(1行が1要素に格納されている1次元配列)
	#
	def RTUtility.exec_remasterOS_command_terminal(cmd, parent = nil, wait_msg = "")
		addMessage(cmd + "\n")
		status = 0
		return status unless @@initialized
		isBackup = false

		open("|-") { |f|
			if f == nil
				# 子プロセス側の処理(リマスタリング対象OSにchrootした側の処理)
cmd_file = <<_EOF_
#!/usr/bin/env ruby
Dir.chroot("__ROOTDIR__")
Dir.chdir("/")
system("mount -t proc /proc /proc")
File.delete("/tmp/remaster_cmd.__PID__") if File.exist?("/tmp/remaster_cmd.__PID__")
system("LC_ALL=C __COMMAND__ || (echo $? > /tmp/remaster_cmd.__PID__ ; read)")
system("umount /proc")
exit!
_EOF_
				cmd_file.gsub!("__ROOTDIR__", get_os_plugin.get_root_dir)
				cmd_file.gsub!("__PID__", $$.to_s)
				cmd_file.gsub!("__COMMAND__", cmd)
				write_file(File.join("/tmp", PROG_NAME + $$.to_s), cmd_file)
				
				fname = File.join(get_os_plugin.get_root_dir, get_os_plugin.get_resolvconf_path)
				if FileTest.exist?(fname)
					fnamebak = fname + RMTOOL_BAK_EXT
					FileUtils.mv(fname, fnamebak)
					isBackup = true
				end
				write_file(fname, @@mainObj.get_resolv_conf)

				# 実行したコマンドの終了ステータスを保存する
				system(sprintf(@@mainObj.get_terminal_cmd, "ruby " + File.join("/tmp", PROG_NAME + $$.to_s)))
				res_file = File.join(get_os_plugin.get_root_dir, "/tmp/remaster_cmd." + $$.to_s)
				begin
					status = read_file(res_file).to_i
				rescue
				ensure
					FileUtils.rm_f(res_file)
				end
				FileUtils.rm_f(File.join("/tmp", PROG_NAME + $$.to_s))

				# /etc/resolv.confをバックアップした場合は元に戻す
 				FileUtils.mv(fnamebak, fname) if isBackup

				# コマンドの終了ステータスを自身の終了ステータスとして終了する。
				exit!( status )
			else
				# 親ウインドウが指定されている場合はwaitウインドウを表示する
				show_wait_window(parent, wait_msg) if parent

				# 実行コマンドが生きている間、Xのevent loopを回して画面の描画を阻害しないようにする。
				status_array = nil
				begin
					until (status_array = Process.waitpid2(f.pid, Process::WNOHANG))
						Gtk.main_iteration while Gtk.events_pending?
						sleep 0.2
					end
				rescue
				end

				# 子プロセスの終了を待つ
				begin
					status = Process.wait2[1].exitstatus
				rescue
					status =  status_array[1].exitstatus if status_array
				end
				# wait ウインドウを止める
				end_wait_window if parent
			end
		}
		status
	end

	#=== リマスタリング対象OS上でのOSコマンド実行
	#
	#リマスタリング対象OS上で指定されたOSコマンドを実行し、結果を配列に格納して返却する。
	#
	#cmd:: 実行するするOSコマンド
	#outAry:: コマンドの標準出力を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準出力が不要な場合はnilを指定する)
	#errAry:: コマンドの標準エラー出力を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準エラー出力が不要な場合はnilを指定する)
	#inAry:: コマンドの標準入力に渡す内容を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準入力への入力が不要な場合はnilを指定する)
	#parent:: お待ちくださいwindowを表示する際にお待ちくださいwindowの親windowとなるwidget
	#          (お待ちくださいwindowが不要な場合はnilを指定する)
	#wait_msg:: お待ちくださいwindowに表示するメッセージを指定する（parent引数がnilの場合は参照されない)
	#isDisplayMsg:: 実行したコマンドの出力をメッセージ表示域に表示するか否かを指定する。true=表示する、false=表示しない。
	#net:: コマンドの実行にネットワークを使用するか否かを指定する。true=使用する、false=使用しない。
	#復帰値:: 実行結果配列(1行が1要素に格納されている1次元配列)
	#
	def RTUtility.exec_remasterOS_command(cmd, outAry = nil, errAry = nil, inAry = nil, parent = nil, wait_msg = "", isDisplayMsg = true, net = false)
		addMessage(cmd + "\n")
		status = 0
		return status unless @@initialized
		isBackup = false
		pp = IO.pipe

		open("|-") { |f|
			if f == nil
				# 子プロセス側の処理(リマスタリング対象OSにchrootした側の処理)
				pp[0].close
				STDERR.reopen(pp[1])

				# リマスタリング対象OSのルートディレクトリにchrootする
 				Dir.chroot(get_os_plugin.get_root_dir)
				Dir.chdir("/")

				# ネットワーク指定時は/etc/resolv.confをバックアップし、オプション設定
				# で指定された内容を書き出す。
				if net
					# Backup /etc/resolv.conf
 					fname = get_os_plugin.get_resolvconf_path
 					if FileTest.exist?(fname)
 						fnamebak = fname + RMTOOL_BAK_EXT
 						FileUtils.mv(fname, fnamebak)
 						isBackup = true
 					end
  					write_file(fname, @@mainObj.get_resolv_conf)
				end

				# /procファイルシステムをマウントする
 				system("mount -t proc /proc /proc")

				# リマスタリング対象OS上でコマンドを実行する
				open("|#{cmd}"){ |fpkg|
					fpkg.each{ |line|puts line; STDOUT.flush }
				}

				# 実行したコマンドの終了ステータスを保存する
				status = $?.exitstatus

				# /procファイルシステムをアンマウントする
 				system("umount /proc")

				# /etc/resolv.confをバックアップした場合は元に戻す
 				FileUtils.mv(fnamebak, fname) if isBackup

				# コマンドの終了ステータスを自身の終了ステータスとして終了する。
				exit!( status )
			else
				# 親ウインドウが指定されている場合はwaitウインドウを表示する
				show_wait_window(parent, wait_msg) if parent
				# 標準エラー出力の内容を取得する
				pp[1].close
				thrd = Thread.new {
					pp[0].each{|line|
						utfText = Kconv.toutf8(line)
						errAry.push utfText.chomp if errAry
						addMessage(utfText) if isDisplayMsg
					}
				}

				# 実行コマンドが生きている間ループしてコマンドの出力を受け取る。
				# 受け取る合間にXのevent loopを回して画面の描画を阻害しないようにする。
				status_array = nil
				begin
					until (status_array = Process.waitpid2(f.pid, Process::WNOHANG))
						Gtk.main_iteration while Gtk.events_pending?
						if IO.select([f], nil, nil, 0.2)
							outText = f.gets
							if outText
								utfText = Kconv.toutf8(outText)
								outAry.push utfText.chomp if outAry
								addMessage(utfText) if isDisplayMsg
							end
						end
					end
				rescue
					# 実行中のコマンドを停止させるために、他のスレッドから本スレッドで例外を発生
					# させる場合があるので、それを捕まえて実行中のコマンドを殺す。
					begin
						Process.kill(9, f.pid)
					rescue
						# Kill発行のタイミングによってはコマンドがすでに終了している場合がある
					end
				end
				f.each{|line|
					utfText = Kconv.toutf8(line)
					outAry.push utfText.chomp if outAry
					addMessage(utfText) if isDisplayMsg
				}

				# 子プロセスの終了を待つ
				begin
					status = Process.wait2[1].exitstatus
				rescue
					status =  status_array[1].exitstatus if status_array
				end
				pp[0].close
				thrd.exit

				# wait ウインドウを止める
				end_wait_window if parent
			end
		}
		status
	end

	#=== 作業用OS上でのOSコマンド実行
	#
	#作業用OS上で指定されたOSコマンドを実行し、結果を配列に格納して返却する。
	#
	#cmd:: 実行するするOSコマンド
	#outAry:: コマンドの標準出力を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準出力が不要な場合はnilを指定する)
	#errAry:: コマンドの標準エラー出力を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準エラー出力が不要な場合はnilを指定する)
	#inAry:: コマンドの標準入力に渡す内容を格納する配列。1行が1要素に格納されている1次元配列
	#         （標準入力への入力が不要な場合はnilを指定する)
	#parent:: お待ちくださいwindowを表示する際にお待ちくださいwindowの親windowとなるwidget
	#          (お待ちくださいwindowが不要な場合はnilを指定する)
	#wait_msg:: お待ちくださいwindowに表示するメッセージを指定する（parent引数がnilの場合は参照されない)
	#isDisplayMsg:: 実行したコマンドの出力をメッセージ表示域に表示するか否かを指定する。true=表示する、false=表示しない。
	#復帰値:: 実行結果配列(1行が1要素に格納されている1次元配列)
	#
	def RTUtility.exec_workOS_command(cmd, outAry = nil, errAry = nil, inAry = nil, parent = nil, wait_msg = "", isDisplayMsg = true)
		addMessage(cmd + "\n") if isDisplayMsg
		resultAry = Array.new

		pp = IO.pipe
		stderr = STDERR.dup
		STDERR.reopen(pp[1])

		# 親ウインドウが指定されている場合はwaitウインドウを表示する
		show_wait_window(parent, wait_msg) if parent

		thrd = Thread.start {
			sleep 3 if inAry		# 子プロセスの入力準備が完了するまで3秒ほど待つ
			pp[0].each{|line|
				utfText = Kconv.toutf8(line)
				errAry.push utfText.chomp if errAry
				addMessage(utfText)
			}
		}

		open("|#{cmd}", "w+"){ |f|
			if inAry
				sleep 3
				inAry.each{|x| f.puts(x); sleep 0.1 }
			end
			f.flush
			# 実行コマンドが生きている間ループしてコマンドの出力を受け取る。
			# 受け取る合間にXのevent loopを回して画面の描画を阻害しないようにする。
			status_array = nil
			begin
				until (status_array = Process.waitpid2(f.pid, Process::WNOHANG))
					if Thread.current == Thread.main
						Gtk.main_iteration while Gtk.events_pending?
					end
					if IO.select([f], nil, nil, 0.2)
						outText = f.gets
						if outText
							utfText = Kconv.toutf8(outText)
							outAry.push utfText.chomp if outAry
							addMessage(utfText) if isDisplayMsg
						end
					end
				end
			rescue
				Process.kill(9, f.pid)
			end
			f.each{|line|
				utfText = Kconv.toutf8(line)
				outAry.push utfText.chomp if outAry
				addMessage(utfText) if isDisplayMsg
			}
		}
		status = 0
		begin
			status = $?.exitstatus
		rescue
			status =  status_array[1].exitstatus if status_array
		end
		STDERR.reopen(stderr)
		pp[1].close
		thrd.exit

		# wait ウインドウを止める
		end_wait_window if parent
		status
	end

	#=== APTコマンドの設定取得
	#
	#aptコマンドの設定情報のうち引数keyで取得した設定に対応する情報を返却する。
	#
	#key:: 取得する情報の設定名
	#復帰値:: 取得する情報の設定名に対応する値（未設定のキーに対しては空文字列を返却）
	#
	def RTUtility.get_apt_setting(key)
		if @@apt_conf.empty?
			regexp = /(\S+) \"(\S+)\";/
			open("|#{CMD_APTCONFIG}") {|apt|
				apt.each { |line|
					line.scan(regexp) { |name, value|
						@@apt_conf[name] = value
					}
				}
			}
		end

		@@apt_conf[key]
	end

	#=== ファイルの読み込み
	#
	#指定したファイルを読み込み１行／１要素の配列で返却する。
	#
	#file:: 読み込みを行うファイルのパス
	#remaster:: ファイルの格納場所。true=リマスタリング対象OS, false=作業用OS
	#復帰値:: なし
	#
	def RTUtility.read_file(file, remaster = false)
		path = remaster ? get_remaster_path_in_own(file) : file
		contents = ""
		open(path){|f|
			f.each{ |line|
				contents << Kconv.toutf8(line)
			}
		}
		contents
	end

	#=== gzipファイルの読み込み
	#
	#指定したgzip圧縮ファイルを読み込み１行／１要素の配列で返却する。
	#
	#file:: 読み込みを行うgzipファイルのパス
	#remaster:: ファイルの格納場所。true=リマスタリング対象OS, false=作業用OS
	#復帰値:: なし
	#
	def RTUtility.read_gzip_file(file, remaster = false)
		path = remaster ? get_remaster_path_in_own(file) : file
		contents = ""
		Zlib::GzipReader.open(path){|f|
			f.each{ |line|
				contents << Kconv.toutf8(line)
			}
		}
		contents
	end

	#=== ファイルの書き込み
	#
	#指定した内容を指定パスのファイルに書き込む。
	#
	#file:: 書き込み先ファイルのパス
	#contents:: ファイルに書き込む内容
	#remaster:: ファイルの格納場所。true=リマスタリング対象OS, false=作業用OS
	#復帰値:: なし
	#
	def RTUtility.write_file(file, contents, remaster = false)
		path = remaster ? get_remaster_path_in_own(file) : file
		FileUtils.mkdir_p(File.dirname(path))
		open(path, "w"){ |f|
			f.write contents
		}
	end

	#=== aboutダイアログの表示
	#
	#基本情報のみのアバウトダイアログをモーダルダイアログとして表示する。
	#
	#name:: aboutダイアログに表示するプログラム名
	#version:: aboutダイアログに表示するプログラムのバージョン
	#copyright:: aboutダイアログに表示するプログラム名のコピーライト
	#license:: aboutダイアログに表示するプログラム名のライセンス
	#復帰値:: なし
	#
	def RTUtility.show_about(name, version, copyright, license)
		about = Gtk::AboutDialog.new
		about.set_name(name )
		about.set_version(version)
		about.set_copyright(copyright) if copyright
		about.set_license(license) if license
		about.set_transient_for(mainWindow)
		about.run
	end

	#=== OSイメージの再作成が必要であることを通知
	#
	#ファイル操作、パッケージのインストール／アンインストールによりOSイメージ
	#の再作成が必要であることを通知する。
	#この呼び出しにより、ディストリビューション作成画面の「OSイメージの作成」、「プロファイリング」、
	#「メディアの作成」の各手順が未実行状態になる。
	#
	#復帰値:: なし
	#
	def RTUtility.need_osimage_remake
		@@mainObj.need_osimage_remake
	end

	#=== treeview(ListView)中に既に同一のデータが格納されているかチェック
	#
	#リストビュー中に既に同一のデータが格納されているかチェックする。
	#
	#treeview:: リストの内容をチェックするGTK::Treeviewクラスのwidget
	#ary:: 存在をチェックするデータ
	#復帰値:: BOOL。true=存在する。false=存在しない。
	#
	def RTUtility.is_list_exist?(treeview, ary)
		return false unless (iter = treeview.model.iter_first)
		begin
			ary.each_with_index{|str, i|
				raise if str != iter[i]
			}
			return true
		rescue
		end while iter.next!

		false
	end

	#=== treeviewへのデータの追加
	#
	#リストビュー中に指定されたデータを追加する。
	#
	#treeview:: データを追加するGTK::Treeviewクラスのwidget
	#ary:: 追加するデータ
	#allowDup:: 重複許可。true=データの重複を許可。false=データの重複は禁止。
	#復帰値:: BOOL。true=データを追加した。false=データを追加しなかった（既に同一データが存在）。
	#
	def RTUtility.add_list(treeview, ary, allowDup = false)
		# 同一エントリを居かしない場合、エントリチェックメソッドを呼びだして同一エントリ
		# の有無を確認する。同一エントリが存在する場合は追加を行わない
		return false if allowDup == false && is_list_exist?(treeview, ary)
		iter = treeview.model.append
		ary.each_with_index{|str, i| iter[i] = str }
		true
	end

	#=== treeviewからのデータ削除
	#
	#リストビューで選択中のデータをリストから削除する。
	#
	#treeview:: データを削除するGTK::Treeviewクラスのwidget
	#復帰値:: BOOL。true=データを削除した。false=データを追加しなかった（データが未選択）。
	#
	def RTUtility.del_list(treeview)
		dels = Array.new
		treeview.selection.selected_each{|model, path, iter|
			dels.push(Gtk::TreeRowReference.new(model,path))
		}
		return false if dels.empty?

		model = treeview.model
		dels.each{ |rowref|
			(path = rowref.path) and model.remove(model.get_iter(path))
		}

		true
	end

    #=== リマスタリングツールのオプションメニューの初期設定
    #
    #リマスタリングツールのオプションメニューの初期設定を行う。
	#各プラグインにoptionメニューの有無を問い合わせ、「有」の場合はオプション
	#メニューにそのプラグインに対するメニューアイテムを作成する。
    #
    #widget::オプションメニューのwidget
    #
	def RTUtility.create_menu_option(widget)
#		workAry = Array.new
		get_plugins.each_value {|hash|
			hash.each { |key, val|
				if val.have_option?
					item_name = sprintf(OPTION_NAME_LITERAL, key)
					item = Gtk::MenuItem.new(item_name)
					item.signal_connect("activate"){ |w|
						val.show_option
					}
					widget.append(item)
#					workAry.push(item)
				end
			}
		}
		widget.show_all
#		workAry
	end

    #=== リマスタリングツールのヘルプメニューの初期設定
    #
    #リマスタリングツールのヘルプメニューの初期設定を行う。
	#各プラグインにHelpメニューの有無を問い合わせ、「有」の場合はヘルプメニューに
	#そのプラグインに対するメニューアイテムを作成する。
    #
    #widget::Helpメニューのwidget
    #
	def RTUtility.create_menu_help(widget)
		# newしたメニュー項目をwidgetにappendするだけだとGCによってなくなるので、
		# 配列にメニュー項目を格納して呼び出し元に返すようにした。
#		workAry = Array.new
		get_plugins.each_value {|hash|
			hash.each { |key, val|
				if val.have_about?
					item_name = sprintf(ABOUT_NAME_LITERAL, key)
					item = Gtk::MenuItem.new(item_name)
					item.signal_connect("activate"){ |w|
						val.show_about
					}
					widget.append(item)
#					workAry.push(item)
				end
			}
		}
		widget.show_all
#		workAry
	end

    #=== リマスタリング環境のインポート
    #
    #リマスタリング環境をリマスタリング環境ファイルから復元する。
	#importメソッドでは利用者が指定したリマスタリング環境ファイルを
	#各画面、各プラグインに反映する処理を実施する。
    #
    #復帰値::なし
    #
	def RTUtility.import
		@@mainObj.import
	end

private

	#
	# Get remastering OS path
	#
	def RTUtility.get_remaster_path_in_own(file)
		File.join(get_os_plugin.get_root_dir, file)
	end
end
