package jp.sourceforge.foolishmerge;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * 設定クラス
 */
public class FoolishMergeConfig {

	/**
	 * 設定ファイル名
	 */
	private static final String FILENAME = "foolishmerge.properties";

	/**
	 * キーのプリフィックス
	 */
	private static final String PREFIX = "foolishmerge.";

	/**
	 * プロパティ
	 */
	private static Properties config = null;

	// 設定を読み込む。
	// 優先度は 設定ファイル > システムプロパティ
	static {
		// システムプロパティをフィールドにセット。
		config = System.getProperties();

		// カレントスレッドからクラスローダを取得。
		ClassLoader cloader = Thread.currentThread().getContextClassLoader();

		// 取得できなかった場合は自身のクラスローダを取得。
		if (cloader == null) {
			cloader = FoolishMergeConfig.class.getClassLoader();
		}

		// 設定ファイルを読み込む。
		InputStream in = cloader.getResourceAsStream(FILENAME);

		try {
			// 設定ファイルを読み込めた場合、プロパティにロード。
			if (in != null) {
				config.load(in);
				in.close();
			}
		} catch (IOException e) {
			new FoolishMergeException(e);
		}
	}

	/**
	 * コンフリクト箇所の開始セパレータを取得する。
	 * 
	 * @return 開始セパレータ(デフォルトは「<<<<<<<」)
	 */
	public static String getStartSeparator() {
		// 開始セパレータを取得。
		// デフォルトはdiffコマンドに倣って「<<<<<<<」。
		return config.getProperty(PREFIX + "start.separator", "<<<<<<<");
	}

	/**
	 * コンフリクト箇所の終了セパレータを取得する。
	 * 
	 * @return 終了セパレータ(デフォルトは「>>>>>>>」)
	 */
	public static String getEndSeparator() {
		// 終了セパレータを取得。
		// デフォルトはdiffコマンドに倣って「>>>>>>>」。
		return config.getProperty(PREFIX + "end.separator", ">>>>>>>");
	}

	/**
	 * コンフリクト箇所のリビジョンセパレータを取得する。
	 * 
	 * @return リビジョンセパレータ(デフォルトは「=======」)
	 */
	public static String getRevisionSeparator() {
		// リビジョンセパレータを取得。
		// デフォルトはdiffコマンドに倣って「=======」。
		return config.getProperty(PREFIX + "revision.separator", "=======");
	}

	/**
	 * 改行コードを取得する。
	 * 
	 * @return 改行コード
	 */
	public static String getLineSeparator() {
		// 改行コードを取得。
		return config.getProperty("line.separator");
	}

}
