package jp.sourceforge.foolishmerge.diff;

import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.foolishmerge.FoolishMergeConfig;

/**
 * 差分クラス。
 * 行差分を複数格納する。
 */
public class Delta implements Comparable {

	/**
	 * 状態 - 追加
	 */
	public static final int ADD = 1;

	/**
	 * 状態 - 削除
	 */
	public static final int DEL = -1;

	/**
	 * 状態 - 変更
	 */
	public static final int CHANGE = 0;

	/**
	 * 状態 - 状態なし
	 */
	public static final int NO_STATE = Integer.MIN_VALUE;

	/**
	 * 行差分
	 */
	private LineDifference[] lines = null;

	/**
	 * 行差分 - 編集元(削除分)
	 */
	private LineDifference[] original = null;

	/**
	 * 行差分 - 編集後(追加分)
	 */
	private LineDifference[] modified = null;

	/**
	 * 状態
	 */
	private int state = NO_STATE;

	/**
	 * 改行コード
	 */
	private static final String NL = FoolishMergeConfig.getLineSeparator();

	/**
	 * コンフリクトなし(差分1 &lt; 差分2)<BR>
	 * <CODE>
	 * ┌<BR>
	 * │<BR>
	 * └<BR>
	 * 　┐<BR>
	 * 　│<BR>
	 * 　┘
	 * </CODE>
	 */
	public static final int NO_CONFLICT_LT = -2;

	/**
	 * コンフリクト(差分1 &lt;= 差分2)<BR>
	 * <CODE>
	 * ┌　┌┐　┐<BR>
	 * │┐││┌│<BR>
	 * └│└│└│<BR>
	 * 　┘　┘　┘
	 * </CODE>
	 */
	public static final int CONFLICT_LT = -1;

	/**
	 * コンフリクト(差分1 == 差分2)<BR>
	 * <CODE>
	 * ┌┐┌　　┐<BR>
	 * │││┐┌│<BR>
	 * ││││││<BR>
	 * └┘└┘└┘
	 * </CODE>
	 */
	public static final int CONFLICT_EQ = 0;

	/**
	 * コンフリクト(差分1 &gt;= 差分2)<BR> 
	 * <CODE>
	 * 　┐┌┐┌<BR>
	 * ┌││││┐<BR>
	 * │┘│┘│┘<BR>
	 * └　└　└
	 * </CODE>
	 */
	public static final int CONFLICT_GT = 1;

	/**
	 * コンフリクトなし(差分1 &gt; 差分2)<BR>
	 * <CODE>
	 * 　┐<BR>
	 * 　│<BR>
	 * 　┘<BR>
	 * ┌<BR>
	 * │<BR>
	 * └
	 * </CODE>
	 */
	public static final int NO_CONFLICT_GT = 2;

	/**
	 * チャンク(行差分のリスト)から差分を構築する。
	 * 
	 * @param chunk チャンク
	 */
	public Delta(List chunk) {
		// 行差分をフィールドにセット。
		lines =
			(LineDifference[]) chunk.toArray(new LineDifference[chunk.size()]);

		// 行差分を格納するリストを生成。
		List org = new LinkedList();
		List mod = new LinkedList();

		// 行差分を編集元と編集後に振り分ける。
		for (int i = 0; i < lines.length; i++) {
			switch (lines[i].getState()) {
				case LineDifference.ADD :
					// 追加行は「編集後」。
					mod.add(lines[i]);
					break;

				case LineDifference.DEL :
					// 削除行は「編集元」。
					org.add(lines[i]);
					break;
			}
		}

		// 編集元、編集後の行差分をフィールドにセット。
		original =
			(LineDifference[]) org.toArray(new LineDifference[org.size()]);
		modified =
			(LineDifference[]) mod.toArray(new LineDifference[mod.size()]);

		// ステータスをセット。
		if (original.length > 0 && modified.length > 0) {
			// 追加行、削除行がともにある場合は「変更」。
			state = CHANGE;
		} else if (original.length > 0) {
			// 削除行のみの場合は「削除」。
			state = DEL;
		} else if (modified.length > 0) {
			// 追加行のみの場合は「追加」。
			state = ADD;
		}

	}

	/**
	 * 行差分を取得する。
	 * 
	 * @return 行差分
	 */
	public LineDifference[] getLineDifference() {
		// 行差分を返す。
		return lines;
	}

	/**
	 * 編集元文書の行を取得する。
	 * 
	 * @return 編集元文書の行
	 */
	public String[] getOrgLines() {
		// バッファを生成。
		String[] lines = new String[original.length];

		// 差分情報から行の文字列を取得。
		for (int i = 0; i < original.length; i++) {
			lines[i] = original[i].getLine();
		}

		// 編集元文書の行を返す。
		return lines;
	}

	/**
	 * 編集元(削除分)の行差分を取得する。
	 * 
	 * @return 編集元の行差分
	 */
	public LineDifference[] getOrgDifference() {
		// 編集元の行差分を返す。
		return original;
	}

	/**
	 * 編集後文書の行を取得する。
	 * 
	 * @return 編集後文書の行
	 */
	public String[] getModLines() {
		// バッファを生成。
		String[] lines = new String[modified.length];

		// 差分情報から行の文字列を取得。
		for (int i = 0; i < modified.length; i++) {
			lines[i] = modified[i].getLine();
		}

		// 編集後文書の行を返す。
		return lines;
	}

	/**
	 * 編集後(追加分)の行差分を取得する。
	 * 
	 * @return 編集後の行差分
	 */
	public LineDifference[] getModDifference() {
		// 編集後の行差分を返す。
		return modified;
	}

	/**
	 * 差分のステータスを取得する。
	 * 
	 * @return 差分のステータス(ADD/DEL/CHANGE/NO_STATE)
	 */
	public int getState() {
		// ステータスを返す。
		return state;
	}

	/**
	 * 編集元文書にパッチを当てる。
	 * 
	 * @param doc 編集元文書
	 * @param offset オフセット
	 * @return パッチサイズ
	 */
	public int patch(List doc, int offset) {
		int size = 0;

		// 行差分の件数分繰り返す。
		for (int i = 0; i < lines.length; i++) {
			// 行のパッチを当てる。
			lines[i].patch(doc, offset + size);

			// パッチサイズを計算。
			switch (lines[i].getState()) {
				case LineDifference.ADD :
					// 追加の場合
					size++;
					break;

				case LineDifference.DEL :
					// 削除の場合
					size--;
					break;
			}
		}

		// パッチサイズを返す。
		return size;
	}

	/**
	 * 渡された差分とコンフリクトしているかどうかを返す。
	 * 
	 * @param delta 比較する差分
	 * @return コンフリクトしている場合はtrue
	 */
	public boolean isConflict(Delta delta) {
		// 渡された差分と比較。
		int confCnst = compareTo(delta);

		if (confCnst == NO_CONFLICT_LT || confCnst == NO_CONFLICT_GT) {
			// コンフリクトしていない場合はfalseを返す。
			return false;
		} else {
			// コンフリクトしている場合はtrueを返す。
			return true;
		}
	}

	/**
	 * 渡された差分と順序を比較する。
	 * 
	 * @param obj 比較する差分
	 * @return コンフリクト定数
	 */
	public int compareTo(Object obj) {
		// 差分にキャスト。
		Delta delta = (Delta) obj;

		// 戻り値に「コンフリクト(差分1 == 差分2)」をセット。
		// ┌┐ ┌     ┐
		// ││ │┐ ┌│
		// ││ ││ ││
		// └┘ └┘ └┘
		int ret = CONFLICT_EQ;

		// 修正開始/終了行数を取得。
		int first1 = getOrgFirstLineNum();
		int last1 = getOrgLastLineNum();
		int first2 = delta.getOrgFirstLineNum();
		int last2 = delta.getOrgLastLineNum();

		if (last1 < first2) {
			// 差分1の末尾が差分2の先頭より前にある場合、
			// 戻り値に「コンフリクトなし(差分1 < 差分2)」をセット。
			// ┌
			// │
			// └
			//   ┐
			//   │
			//   ┘	
			ret = NO_CONFLICT_LT;
		} else if (first1 > last2) {
			// 差分2の末尾が差分1の先頭より前にある場合、
			// 戻り値に「コンフリクトなし(差分1 > 差分2)」をセット。
			//   ┐
			//   │
			//   ┘	
			// ┌
			// │
			// └
			ret = NO_CONFLICT_GT;
		} else if (last1 < last2) {
			// 差分1の末尾が差分2の末尾より前にある場合、
			// 戻り値に「コンフリクト(差分1 <= 差分2)」をセット。
			// ┌   ┌┐   ┐
			// │┐ ││ ┌│
			// └│ └│ └│
			//   ┘   ┘   ┘
			ret = CONFLICT_LT;
		} else if (last1 > last2) {
			// 差分2の末尾が差分1の末尾より前にある場合、
			// 戻り値に「コンフリクト(差分1 >= 差分2)」をセット。
			//   ┐ ┌┐ ┌
			// ┌│ ││ │┐
			// │┘ │┘ │┘
			// └   └   └
			ret = CONFLICT_GT;
		}

		// コンフリクト定数を返す。
		return ret;
	}

	/**
	 * 差分の文字列表現を取得する。
	 * 
	 * @return 差分の文字列表現
	 */
	public String toString() {
		// diffコマンドの標準的な出力を返す。
		return toNormalFormat();
	}

	/**
	 * 差分をdiffコマンドのnormal形式の文字列として取得する。
	 * 
	 * @see <A HREF="http://www.gnu.org/software/diffutils/manual/html_node/Detailed-Normal.html">Detailed Description of Normal Format</A>
	 * @return 差分の文字列表現
	 */
	public String toNormalFormat() {
		// バッファを生成。
		StringBuffer buf = new StringBuffer();

		// 編集元文書の開始行、終了行を取得。
		int orgFirst = getOrgFirstLineNum() + 1;
		int orgLast = getOrgLastLineNum() + 1;
		// 編集後文書の開始行、終了行を取得。
		int modFirst = getModFirstLineNum() + 1;
		int modLast = getModLastLineNum() + 1;

		String orgLineNum = null;
		String modLineNum = null;

		// 編集元文書の修正範囲を取得。
		if (orgFirst == orgLast) {
			// 開始行と終了行が同じ場合は開始行を修正範囲とする。
			orgLineNum = String.valueOf(orgFirst);
		} else {
			// 開始行と終了行が異なる場合は開始行～終了を修正範囲とする。
			orgLineNum = orgFirst + "," + orgLast;
		}

		// 編集後文書の修正範囲を取得。
		if (modFirst == modLast || state == DEL) {
			// 開始行と終了行が同じ、または削除の場合は開始行を修正範囲とする。
			modLineNum = String.valueOf(modFirst);
		} else {
			// 開始行と終了行が異なる場合は開始行～終了を修正範囲とする。
			modLineNum = modFirst + "," + modLast;
		}

		// バッファに編集元文書の修正範囲を追加。
		buf.append(orgLineNum);

		// バッファに変更コマンドを追加。
		switch (state) {
			case ADD :
				buf.append("a");
				break;

			case DEL :
				buf.append("d");
				break;

			case CHANGE :
				buf.append("c");
				break;
		}

		// バッファに編集後文書の修正範囲を追加。
		buf.append(modLineNum);
		buf.append(NL);

		int prevState = LineDifference.NO_STATE;
		int crntState = LineDifference.NO_STATE;

		// バッファに修正行の情報を追加。
		for (int i = 0; i < lines.length; i++) {
			crntState = lines[i].getState();

			// ステータスが変わったらセパレータを追加。
			if (prevState != LineDifference.NO_STATE
				&& crntState != prevState) {
				buf.append("---" + NL);
			}

			prevState = crntState;

			switch (crntState) {
				case LineDifference.ADD :
					// 追加行の場合
					buf.append("> ");
					buf.append(lines[i].getLine());
					break;

				case LineDifference.DEL :
					// 削除行の場合
					buf.append("< ");
					buf.append(lines[i].getLine());
					break;
			}

			buf.append(NL);
		}

		// 末尾の改行を削除。 
		if (buf.length() >= NL.length()) {
			buf.delete(buf.length() - NL.length(), buf.length());
		}

		// 差分の文字列表現を返す。
		return buf.toString();
	}

	/**
	 * 編集元文書の修正開始行を取得する。
	 * 
	 * @return 編集元文書の修正開始行
	 */
	public int getOrgFirstLineNum() {
		// 編集元の行差分が存在しない場合
		// 編集後の行差分から編集元文書の行数を取得。
		if (original.length < 1) {
			return lines[0].getOrgLineNum();
		}

		// 編集元の行差分から行数を取得して返す。
		return original[0].getOrgLineNum();
	}

	/**
	 * 編集元文書の修正終了行を取得する。
	 * 
	 * @return 編集元文書の修正終了行
	 */
	public int getOrgLastLineNum() {
		// 編集元の行差分が存在しない場合
		// 編集後の行差分から編集元文書の行数を取得。
		if (original.length < 1) {
			return lines[lines.length - 1].getOrgLineNum();
		}

		// 編集元の行差分から行数を取得して返す。
		return original[original.length - 1].getOrgLineNum();
	}

	/**
	 * 編集後文書の修正開始行を取得する。
	 * 
	 * @return 編集後文書の修正開始行
	 */
	public int getModFirstLineNum() {
		// 編集後の行差分が存在しない場合
		// 編集元の行差分から編集後文書の行数を取得。
		if (modified.length < 1) {
			return lines[0].getModLineNum();
		}

		// 編集後の行差分から行数を取得して返す。
		return modified[0].getModLineNum();
	}

	/**
	 * 編集後文書の修正終了行を取得する。
	 * 
	 * @return 編集後文書の修正終了行
	 */
	public int getModLastLineNum() {
		// 編集後の行差分が存在しない場合
		// 編集元の行差分から編集後文書の行数を取得。
		if (modified.length < 1) {
			return lines[lines.length - 1].getOrgLineNum();
		}

		// 編集後の行差分から行数を取得して返す。
		return modified[modified.length - 1].getModLineNum();
	}

}
