# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import re
import copy
import time
import codecs

from bbs_type_exception import BbsTypeError

subject_reg_expr = re.compile("(?P<id>(?:(?!<>).)*).dat<>(?P<title>(?:(?!<>).)*)\((?P<res>\d*)\)")
dat_reg_expr = re.compile("(?P<name>(?:(?!<>).)*)<>(?P<mail>(?:(?!<>).)*)<>(?P<date>(?:(?!<>).)*)<>(?P<msg>(?:(?!<>).)*)<>(?P<title>(?:(?!<>).)*)")


class BaseType:
    """Base of bbs type. Do not instantiate directly"""

    bbs_type = "other"
    _base_reg = None
    _cgi_reg = None
    subject_reg = subject_reg_expr
    dat_reg = dat_reg_expr
    encoding = "cp932"

    def __init__(self, uri):
        self.uri = uri
        self.host = None
        self.board = None
        self.thread = None

        self._parse_uri(uri)

    def _parse_uri(self, uri):

        m = self._base_reg.match(self.uri)
        if m:
            self.host = m.group("host")
            self.board = m.group("board")
        else:
            m = self._cgi_reg.match(self.uri)
            if m:
                self.host = m.group("host")
                self.board = m.group("board")
                self.thread = m.group("thread")
            else:
                raise BbsTypeError, \
                      "the uri %s does not represent %s" \
                      % (self.uri, self.bbs_type)

    def is_board(self):
        return not self.thread

    def is_thread(self):
        return self.thread

    def is_same_board(self, another):
        return self.bbs_type == another.bbs_type \
               and self.host == another.host \
               and self.board == another.board

    def clone_with_thread(self, thread):
        if not thread:
            raise ValueError, "parameter must not be empty"
        another = copy.copy(self)
        another.thread = thread
        another.uri = another.get_thread_uri()
        return another

    def get_uri_base(self):
        return "http://" + self.host + "/" + self.board + "/"

    def get_subject_txt_uri(self):
        return self.get_uri_base() + "subject.txt"

    def get_dat_uri(self):
        if not self.thread:
            raise BbsTypeError, "not specified thread"
        return self.get_uri_base() + "dat/" + self.thread + ".dat"

    def get_thread_uri(self):
        if not self.thread:
            raise BbsTypeError, "not specified thread"
        return "http://" + self.host + "/test/read.cgi/" + \
                  self.board + "/" + self.thread + "/"

    def get_post_uri(self):
        return "http://" + self.host + "/test/bbs.cgi"

    def build_post_dict(self, name, mail, msg):
        post_dict = {}
        post_dict["bbs"] = self.board
        post_dict["key"] = self.thread
        post_dict["time"] = self.thread
        post_dict["submit"] = u"\u66f8\u304d\u8fbc\u3080".encode(
            "cp932", "replace")
        post_dict["FROM"] = name.encode("cp932", "replace")
        post_dict["mail"] = mail.encode("cp932", "replace")
        post_dict["MESSAGE"] = msg.encode("cp932", "replace")
        return post_dict

    def set_extra_post(self, post_dict):
        return post_dict

    def set_extra_dat_request(self, request, thread):
        return request

    def get_title_from_dat(self, line):
        if line:
            line = line.decode(self.encoding, "replace")
            m = self.dat_reg.match(line)
            if m:
                return m.group("title")
        return ""

    def get_board_dir_path(self):
        """Returns board dir path from logs dir downward, not full path"""

        return self.bbs_type + "/" + self.host + "/" + self.board
