# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import os
import time
import gobject
import gconf
import traceback
import sys
import itertools

import board_data
import uri_opener
import misc
from threadlistmodel import ThreadListModel
from BbsType import bbs_type_judge_uri
import config
import session
import winwrapbase
from misc import ThreadInvoker
import bookmark_list
import bookmark_window
import board_plugins
import board_column
import board_states

GLADE_FILENAME = "board_window.glade"

def open_board(uri, update=False):
    if not uri:
        raise ValueError, "parameter must not be empty"

    bbs_type = bbs_type_judge_uri.get_type(uri)
    uri = bbs_type.get_uri_base()  # use strict uri

    winwrap = session.get_window(uri)
    if winwrap:
        # already opened
        winwrap.window.present()
        if update:
            winwrap.load(update)
    else:
        winwrap = WinWrap(bbs_type.uri)  # pass the original uri
        winwrap.load(update)


class WinWrap(winwrapbase.WinWrapBase, board_data.BoardData):

    def __init__(self, uri):

        self.bbs_type = bbs_type_judge_uri.get_type(uri)
        board_data.BoardData.__init__(self, self.bbs_type)

        glade_path = os.path.join(config.glade_dir, GLADE_FILENAME)
        self.widget_tree = gtk.glade.XML(glade_path)
        self._get_widgets()

        self.widget_tree.signal_autoconnect(self)

        self.window.set_title(self.bbs_type.uri)
        self.treeview.set_model(ThreadListModel())
        self.toolbar.unset_style()

        renderer = gtk.CellRendererText()

        for column_class in board_column.tree_view_column_list:
            column_class(renderer, self.treeview)

        self.treeview.set_fixed_height_mode(True)

        # menu plugins
        board_plugins.load(self.widget_tree)

        self.restore()
        self.window.show()

        self.created()

    def _get_widgets(self):
        self.window = self.widget_tree.get_widget("window_board")
        self.treeview = self.widget_tree.get_widget("treeview")
        self.popupmenu = self.widget_tree.get_widget("popup_treeview_menu")
        self.toolbar = self.widget_tree.get_widget("toolbar")
        self.statusbar = self.widget_tree.get_widget("appbar")
        self.progress = self.statusbar.get_progress()
        self.filterbar = self.widget_tree.get_widget(
            "bonobodockitem_filterbar")
        self.entry_filterbar = self.widget_tree.get_widget("entry_filterbar")

    def set_status(self, text):
        self.statusbar.set_status(text)

    def set_fraction(self, fraction):
        self.progress.set_fraction(fraction)

    def destroy(self):
        self.save()
        self.window.destroy()

    def get_uri(self):
        return self.bbs_type.get_uri_base()

    def open_thread(self):
        treeselection = self.treeview.get_selection()
        model, iter = treeselection.get_selected()
        if not iter:
            return

        dic = model.get_dict(iter)
        thread = dic["id"]
        title = dic["title"]
        print thread + ':"' + title + '"', "activated"

        res = dic["res"]
        lineCount = dic["lineCount"]

        update = res > lineCount

        bbs_type_for_thread = self.bbs_type.clone_with_thread(thread)
        uri_opener.open_uri(bbs_type_for_thread.get_thread_uri(), update)

    def update_datastore(self, new_list):
        print "reflesh datastore"

        self.treeview.get_model().set_list(new_list)

        # redraw visible area after set list to model
        self.treeview.queue_draw()

        print "end"

    def on_thread_idx_updated(self, thread_uri, idx_dic):
        if not thread_uri or not idx_dic:
            return

        # nothing to do if thread_uri does not belong to this board.
        bbs_type = bbs_type_judge_uri.get_type(thread_uri)
        if not bbs_type.is_thread() \
               or not bbs_type.is_same_board(self.bbs_type):
            return

        thread = bbs_type.thread

        model = self.treeview.get_model()
        if model:
            idx_dic["id"] = thread
            try:
                idx_dic["lastModified"] =  misc.httpdate_to_secs(
                    idx_dic["lastModified"])
            except ValueError:
                idx_dic["lastModified"] = 0
            model.modify_row(idx_dic)

    def load(self, update=False):

        def set_id(thread_id, item_dict):
            item_dict["id"] = thread_id
            return item_dict

        def conv_dictdict_to_listdict(dictdict):
            key_iter = dictdict.iterkeys()
            value_iter = dictdict.itervalues()
            iterable = itertools.imap(set_id, key_iter, value_iter)

            # remove not in subject.txt and not cache.
            iterable = itertools.ifilter(
                lambda dic: dic["num"] > 0 or dic["lineCount"] > 0, iterable)

            return [item_dict for item_dict in iterable]

        def load_local():
            datalist = self.load_idxfiles()
            self.merge_local_subjecttxt(datalist)
            new_list = conv_dictdict_to_listdict(datalist)
            gobject.idle_add(self.update_datastore, new_list)

        def get_remote(datalist):
            print "start get subject.txt"
            self.merge_remote_subjecttxt(datalist)
            new_list = conv_dictdict_to_listdict(datalist)
            gobject.idle_add(self.update_datastore, new_list)

        def on_end():
            def reset_progress():
                self.progress.set_fraction(0.0)
                self.progress.hide()
            gobject.idle_add(reset_progress)

        def deep_copy():
            def init_some(dic):
                dic["num"] = 0
                dic["oldRes"] = dic["res"]
                dic["res"] = 0
                dic["average"] = 0
                return dic

            model = self.treeview.get_model()
            if model:
                iterable = model.original_list

                iterable = itertools.imap(lambda dic: dic.copy(), iterable)
                iterable = itertools.imap(init_some, iterable)

                new_dict = dict([(dic["id"], dic) for dic in iterable])
            else:
                new_dict = {}
            return new_dict

        sbj_path = misc.get_board_subjecttxt_path(self.bbs_type)
        sbj_exists = os.path.exists(sbj_path)

        self.progress.show()

        if update or not sbj_exists:
            new_dict = deep_copy()
            t = ThreadInvoker(on_end, lambda *x: get_remote(new_dict))
            t.start()
        else:
            t = ThreadInvoker(on_end, load_local)
            t.start()

    def save(self):
        try:
            states_path = misc.get_board_states_path(self.bbs_type)
            dirname = os.path.dirname(states_path)

            # save only if board dir exists.
            if os.path.exists(dirname):
                window_width, window_height = self.window.get_size()
                toolbar_visible = self.toolbar.parent.get_property("visible")
                statusbar_visible = self.statusbar.get_property("visible")
                filterbar_visible = self.filterbar.get_property("visible")

                columns = self.treeview.get_columns()
                order = ""
                width = ""
                for column in columns:
                    if order:
                        order += ","
                    order += column.id
                    if width:
                        width += ","
                    width += str(column.get_width())

                sort_column_name, sort_reverse = \
                                  self.treeview.get_model().get_sort()
                sort_reverse = str(sort_reverse)

                f = file(states_path, "w")

                f.write("columns=" + order + "\n")
                f.write("widths=" + width + "\n")
                f.write("sort_column=" + sort_column_name + "\n")
                f.write("sort_reverse=" + sort_reverse + "\n")
                f.write("window_width=" + str(window_width) + "\n")
                f.write("window_height=" + str(window_height) + "\n")
                f.write("toolbar_visible=" + str(toolbar_visible) + "\n")
                f.write("statusbar_visible=" + str(statusbar_visible) + "\n")
                f.write("filterbar_visible=" + str(filterbar_visible) + "\n")

                f.close()
        except:
            traceback.print_exc()

    def restore(self):
        try:
            states_dict = board_states.states_file_to_dict(self.bbs_type)

            self.treeview.get_model().sort(
                states_dict["sort_column"], True, states_dict["sort_reverse"])

            # set column order before set column width
            treeviewcolumn = dict(
                [(cln.id, cln) for cln in self.treeview.get_columns()])
            base_column = None
            for column_name in states_dict["columns"]:
                if column_name in treeviewcolumn:
                    column = treeviewcolumn[column_name]
                    self.treeview.move_column_after(column, base_column)
                    base_column = column

            # set column width afeter set column order
            for width, column in itertools.izip(
                states_dict["widths"], self.treeview.get_columns()):
                if width:
                    column.set_fixed_width(width)

            self.window.set_default_size(
                states_dict["window_width"], states_dict["window_height"])

            if not states_dict["toolbar_visible"]:
                gobject.idle_add(self.toolbar.parent.hide,
                                 priority=gobject.PRIORITY_HIGH)
            if not states_dict["statusbar_visible"]:
                gobject.idle_add(self.statusbar.hide,
                                 priority=gobject.PRIORITY_HIGH)
            if not states_dict["filterbar_visible"]:
                gobject.idle_add(self.filterbar.hide,
                                 priority=gobject.PRIORITY_HIGH)
        except:
            traceback.print_exc()

    def on_menu_toolbar_activate(self, widget):
        if self.toolbar.parent.get_property("visible"):
            self.toolbar.parent.hide()
        else:
            self.toolbar.parent.show()

    def on_menu_statusbar_activate(self, widget):
        if self.statusbar.get_property("visible"):
            self.statusbar.hide()
        else:
            self.statusbar.show()


    # signal handlers

    # menu and toolbutton

    def on_menu_add_bookmark_activate(self, widget):
        bookmark_list.bookmark_list.add_bookmark_with_edit(
            uri=self.bbs_type.uri)

    def on_menu_close_activate(self, widget):
        self.destroy()

    def on_menu_delete_activate(self, widget):
        selection = self.treeview.get_selection()
        model, iter = selection.get_selected()
        if not iter:
            return
        thread = model.get_dict(iter)["id"]
        
        bbs_type_for_thread = self.bbs_type.clone_with_thread(thread)

        dat_path = misc.get_thread_dat_path(bbs_type_for_thread)
        try:
            os.remove(dat_path)
        except OSError:
            traceback.print_exc()
        idx_path = misc.get_thread_idx_path(bbs_type_for_thread)
        try:
            os.remove(idx_path)
        except OSError:
            traceback.print_exc()
        states_path = misc.get_thread_states_path(bbs_type_for_thread)
        try:
            os.remove(states_path)
        except OSError:
            traceback.print_exc()

    def on_menu_filter_activate(self, widget):
        self.filterbar.show()
        self.entry_filterbar.grab_focus()

    def on_menu_manage_bookmarks_activate(self, widget):
        bookmark_window.open()

    def on_menu_open_activate(self, widget):
        self.open_thread()

    def on_menu_quit_activate(self, widget):
        session.main_quit()

    def on_menu_refresh_activate(self, widget):
        self.load(True)


    # window

    def on_window_board_delete_event(self, widget, event):
        self.save()
        return False

    def on_window_board_destroy(self, widget):
        self.popupmenu.destroy()
        for column in self.treeview.get_columns():
            column.destroy()
        self.destroyed()


    # treeview

    def on_treeview_row_activated(self, widget, path, view_column):
        self.open_thread()

    def on_treeview_button_press_event(self, widget, event):
        if event.button == 3:
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pthinfo = widget.get_path_at_pos(x, y)
            if pthinfo is not None:
                path, col, cellx, celly = pthinfo
                widget.grab_focus()
                widget.set_cursor(path, col, 0)
                self.popupmenu.popup(None, None, None, event.button, time)
            return 1


    # filterbar

    def on_entry_filterbar_activate(self, widget):
        text = widget.get_text()

        def func(model, item):
            try:
                item["title"].index(text)
            except ValueError:
                return False
            else:
                return True

        model = self.treeview.get_model()
        if model:
            if text:
                filter_func = func
            else:
                filter_func = None
            model.refilter(filter_func)

    def on_toolbutton_filterbar_close_clicked(self, widget):
        self.filterbar.hide()

    def on_button_filterbar_clear_clicked(self, widget):
        self.entry_filterbar.set_text("")
        model = self.treeview.get_model()
        if model:
            model.refilter(None)
