# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade
import traceback
import gnome
import os.path
import itertools
import sys

from bookmark_core import Bookmark
from bookmark_editwindow import BookmarkEditWindow
from BbsType import bbs_type_exception
import bookmark_list
import uri_opener
import winwrapbase
import session
import config

GLADE_FILENAME = "bookmark_window.glade"

def open():
    winwrap = session.get_window(WinWrap.uri)
    if winwrap:
        # already opened
        winwrap.window.present()
    else:
        winwrap = WinWrap()


class WinWrap(winwrapbase.WinWrapBase):

    uri = "namari://bookmarks-manager"

    def __init__(self):
        self.bookmarklist = bookmark_list.bookmark_list
        self.bookmarklist.set_view(self)

        glade_path = os.path.join(config.glade_dir, GLADE_FILENAME)
        self.widget_tree = gtk.glade.XML(glade_path)
        self.window = self.widget_tree.get_widget("window_bookmark")
        self.window.connect("destroy", self.on_destroy)
        self.treeview_categories = self.widget_tree.get_widget(
            "treeview_categories")
        self.treeview_categories.connect(
            "cursor-changed", self.on_treeview_categories_cursor_changed)
        self.treeview_bookmarks = self.widget_tree.get_widget(
            "treeview_bookmarks")
        self.treeview_bookmarks.connect(
            "row-activated", self.on_treeview_bookmarks_row_activated)
        self.treeview_bookmarks.connect(
            "cursor-changed", self.on_treeview_bookmarks_cursor_changed)
        self.treeview_bookmarks.connect(
            "button-press-event",
            self.on_treeview_bookmarks_button_press_event)
        self.statusbar = self.widget_tree.get_widget("statusbar")
        self.statusbar.push(
            self.statusbar.get_context_id("bookmarks"), "Ready.")
        self.menu_bookmark = self.widget_tree.get_widget("menu_bookmark")

        self.widget_tree.signal_autoconnect(self)

        renderer = gtk.CellRendererText()

        treeviewcolumn = gtk.TreeViewColumn("category", renderer, text=0)
        treeviewcolumn.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        self.treeview_categories.append_column(treeviewcolumn)

        treeviewcolumn = gtk.TreeViewColumn("bookmark", renderer, text=0)
        treeviewcolumn.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        self.treeview_bookmarks.append_column(treeviewcolumn)

        self.treeview_categories.set_fixed_height_mode(True)
        self.treeview_bookmarks.set_fixed_height_mode(True)

        self.treeview_categories.set_model(gtk.ListStore(str, int))
        self.treeview_bookmarks.set_model(gtk.ListStore(str, object))

        self.current_category = ""
        self.current_category_type = 0

        self.update_categories()

        self.created()

    def on_destroy(self, widget):
        self.destroyed()

    def get_uri(self):
        return self.uri

    def destroyed(self):
        self.bookmarklist.set_view(None)
        winwrapbase.WinWrapBase.destroyed(self)

    def destroy(self):
        self.window.destroy()

    def on_treeview_categories_cursor_changed(self, widget):
        selection = self.treeview_categories.get_selection()
        model, iter = selection.get_selected()
        if model and iter:
            category = model.get_value(iter, 0)
            category_type = model.get_value(iter, 1)
            if category and category != self.current_category:
                self.update_bookmarks(category, category_type)

    def _get_selected_bookmark(self):
        selection = self.treeview_bookmarks.get_selection()
        model, iter = selection.get_selected()
        if model and iter:
            return model.get_value(iter, 1)

    def on_treeview_bookmarks_cursor_changed(self, widget):
        bookmark = self._get_selected_bookmark()
        if bookmark:
            self.statusbar.pop(self.statusbar.get_context_id("bookmarks"))
            self.statusbar.push(
                self.statusbar.get_context_id("bookmarks"), bookmark.uri)

    def on_treeview_bookmarks_row_activated(self, widget, path, column):
        model = self.treeview_bookmarks.get_model()
        if model:
            iter = model.get_iter(path)
            if iter:
                self._open_bookmark(model.get_value(iter, 1))

    def on_treeview_bookmarks_button_press_event(self, widget, event):
        if event.button == 3:
            x = int(event.x)
            y = int(event.y)
            time = event.time
            pthinfo = widget.get_path_at_pos(x, y)
            if pthinfo is not None:
                path, col, cellx, celly = pthinfo
                widget.grab_focus()
                widget.set_cursor(path, col, 0)
                self.menu_bookmark.popup(None, None, None, event.button, time)
            return True

    def on_update_bbsmenu(self, widget):
        self.bookmarklist.update_bbsmenu()
        self.update_categories()

    def _open_bookmark(self, bookmark):
        try:
            uri_opener.open_uri(bookmark.uri)
        except bbs_type_exception.BbsTypeError:
            self._open_bookmark_with_web_browser(bookmark)

    def _open_bookmark_with_web_browser(self, bookmark):
        gnome.url_show(bookmark.uri)

    def update_bookmarks(self, category, category_type):
        if category_type == 0:
            bookmarks = self.bookmarklist.get_bookmark_list_in_category(
                category)
        else:
            bookmarks = self.bookmarklist.get_non_category_bookmark()

        model = self.treeview_bookmarks.get_model()
        model.clear()

        for bookmark in bookmarks:
            model.append([bookmark.name, bookmark])
        self.treeview_bookmarks.set_model(model)

        self.current_category = category
        self.current_category_type = category_type

        category_model = self.treeview_categories.get_model()
        if category_model:
            for row in category_model:
                text = row.model.get_value(row.iter, 0)
                cat_type = row.model.get_value(row.iter, 1)
                if text == category and cat_type == category_type:
                    selection = self.treeview_categories.get_selection()
                    selection.select_iter(row.iter)
                    self.treeview_categories.scroll_to_cell(row.path)
                    break

        self.statusbar.pop(self.statusbar.get_context_id("bookmarks"))

    def update_categories(self):
        model = self.treeview_categories.get_model()
        model.clear()
        categories = self.bookmarklist.get_category_list()
        model.append(["Non Category", 1])
        for category in categories:
            model.append([category, 0])
        self.treeview_categories.set_model(model)

        if self.current_category:
            self.update_bookmarks(self.current_category,
                                  self.current_category_type)
        else:
            self.update_bookmarks("", 0)

    def on_editbookmark_complete(self, old_bookmark, new_bookmark):
        changed = False
        changed = changed or old_bookmark.name != new_bookmark.name
        changed = changed or old_bookmark.uri != new_bookmark.uri
        changed = changed or \
                  len(old_bookmark.categories) != len(new_bookmark.categories)
        if not changed:
            for cat, kat in itertools.izip(old_bookmark.categories,
                                           new_bookmark.categories):
                if cat not in new_bookmark.categories:
                    changed = True
                    break
                if kat not in old_bookmark.categories:
                    changed = True
                    break
        if changed:
            old_bookmark.name = new_bookmark.name
            old_bookmark.uri = new_bookmark.uri
            old_bookmark.categories = new_bookmark.categories
            self.update_categories()
            self.bookmarklist.save()
        else:
            print "not changed"
        
    def on_newbookmark_complete(self, bookmark):
        self.bookmarklist.add_new_bookmark(bookmark)
        self.update_categories()

    # menu handler

    def on_new_activate(self, widget):
        BookmarkEditWindow(self.on_newbookmark_complete)

    def on_open_activate(self, widget):
        bookmark = self._get_selected_bookmark()
        if bookmark:
            self._open_bookmark(bookmark)

    def on_open_web_activate(self, widget):
        bookmark = self._get_selected_bookmark()
        if bookmark:
            self._open_bookmark_with_web_browser(bookmark)

    def on_properties_activate(self, widget):
        bookmark = self._get_selected_bookmark()
        if bookmark:
            BookmarkEditWindow(
                lambda new_bookmark: \
                self.on_editbookmark_complete(bookmark, new_bookmark),
                bookmark)

    def on_save_activate(self, widget):
        self.bookmarklist.save()

    def on_close_activate(self, widget):
        self.destroy()

    def on_quit_activate(self, widget):
        session.main_quit()

    def on_delete_activate(self, widget):
        bookmark = self._get_selected_bookmark()
        if bookmark:
            self.bookmarklist.delete_bookmark(bookmark)
            self.update_categories()

    def on_refresh_bbsmenu_activate(self, widget):
        self.bookmarklist.update_bbsmenu()
        self.update_categories()
