# Copyright (C) 2006 by Aiwota Programmer
# aiwotaprog@tetteke.tk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import os.path
import re
import threading
from datetime import tzinfo, timedelta, datetime
import itertools

import config
from BbsType import bbs_type_exception

REG_EXPR_HTTPDATE = re.compile("[^ ,]{3}, (?P<day>\d{2}) (?P<month>(?:Jan)|(?:Feb)|(?:Mar)|(?:Apr)|(?:May)|(?:Jun)|(?:Jul)|(?:Aug)|(?:Sep)|(?:Oct)|(?:Nov)|(?:Dec)) (?P<year>\d{4}) (?P<hour>\d{2}):(?P<minute>\d{2}):(?P<second>\d{2}) GMT")
MON_DICT = {"Jan":1, "Feb":2, "Mar":3, "Apr":4, "May":5, "Jun":6, "Jul":7,
            "Aug":8, "Sep":9, "Oct":10, "Nov":11, "Dec":12}

def _check_thread(bbs_type):
    """if bbs_type is not thread, raise BbsTypeError"""

    if not bbs_type.is_thread():
        raise bbs_type_exception.BbsTypeError, \
              "the bbs_type does not represent thread: " + bbs_type.uri

def get_logs_dir_path():
    return os.path.join(config.get_config_dir_path(), "logs")

def get_thread_dat_dir_path(bbs_type):
    """Returns dir path for saving thread dat file"""

    return os.path.join(get_board_dir_path(bbs_type), "dat")

def get_thread_idx_dir_path(bbs_type):
    """Returns dir path for saving thread index file"""

    return os.path.join(get_board_dir_path(bbs_type), "idx")

def get_thread_states_dir_path(bbs_type):
    """Returns dir path for saving thread states file"""

    return os.path.join(get_board_dir_path(bbs_type), "states")

def get_thread_dat_path(bbs_type):
    """Returns thread dat file path"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_dat_dir_path(bbs_type),
                        bbs_type.thread + ".dat")

def get_board_subjecttxt_path(bbs_type):
    """Returns subject.txt file path"""

    return os.path.join(get_board_dir_path(bbs_type), "subject.txt")

def get_thread_states_path(bbs_type):
    """Returns thread states file path"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_states_dir_path(bbs_type),
                        bbs_type.thread + ".states")

def get_board_states_path(bbs_type):
    """Returns board states file path"""

    return os.path.join(get_board_dir_path(bbs_type), "board.states")

def get_board_idx_path(bbs_type):
    """Returns board idx file path"""

    return os.path.join(get_board_dir_path(bbs_type), "subject.idx")

def get_board_dir_path(bbs_type):
    """Returns board dir path"""

    return os.path.join(get_logs_dir_path(), bbs_type.get_board_dir_path())

def get_thread_idx_path(bbs_type):
    """Returns idx file path of thread"""

    _check_thread(bbs_type)

    return os.path.join(get_thread_idx_dir_path(bbs_type),
                        bbs_type.thread + ".idx")

def get_board_cache_path(bbs_type):
    """Returns .cache file path of board"""

    return os.path.join(get_thread_idx_dir_path(bbs_type), ".cache")

ZERO = timedelta(0)
HOUR = timedelta(hours=1)
class UTC(tzinfo):
    """UTC"""

    def utcoffset(self, dt):
        return ZERO

    def tzname(self, dt):
        return "UTC"

    def dst(self, dt):
        return ZERO

utc = UTC()

epoch = datetime(1970, 1, 1, 0, 0, 0, 0, utc)

def httpdate_to_secs(httpdate):
    """Returns the seconds since the epoch"""

    if not httpdate:
        return 0

    m = REG_EXPR_HTTPDATE.match(httpdate)
    if m:
        tm_day = int(m.group("day"))
        tm_mon = MON_DICT[m.group("month")]
        tm_year = int(m.group("year"))
        tm_hour = int(m.group("hour"))
        tm_min = int(m.group("minute"))
        tm_sec = int(m.group("second"))
        
        d = datetime(tm_year, tm_mon, tm_day, tm_hour, tm_min, tm_sec, 0, utc)
        delta = d - epoch
        return delta.days*24*60*60 + delta.seconds
    else:
        raise ValueError


class ThreadInvoker(threading.Thread):
    def __init__(self, on_end, *methods):
        super(ThreadInvoker, self).__init__()
        self.on_end = on_end
        self.methods = methods
    def run(self):
        try:
            for m in self.methods:
                m()
        finally:
            self.on_end()


class FileWrap:
    def __init__(self, path, mode="a+"):
        self.__file = None
        self.__path = path
        self.__mode = mode
    def __del__(self):
        self.close()
    def seek(self, size):
        self.file().seek(size)
    def write(self, data):
        self.file().write(data)
    def writelines(self, sequence):
        self.file().writelines(sequence)
    def close(self):
        if self.__file:
            self.__file.close()
            self.__file = None
    def file(self):
        if not self.__file:
            basedir = os.path.dirname(self.__path)
            if not os.path.isdir(basedir):
                os.makedirs(basedir)
            self.__file = file(self.__path, self.__mode)
        return self.__file

def unpack_ifilter(predicate, iterable):
    """For multiple argument"""
    for item in iterable:
        if predicate(*item):
            yield item

def split_key_and_value(key_equal_value):
    try:
        index = key_equal_value.index("=")
    except ValueError:
        pass
    else:
        key = key_equal_value[:index]
        value = key_equal_value[index+1:]
        return key, value

def tabbed_to_dict_generator(tabbed, sep="\t"):
    iterable = tabbed.rstrip().split(sep)
    iterable = itertools.imap(split_key_and_value, iterable)
    iterable = itertools.ifilter(None, iterable)
    return iterable

def tabbed_to_dict(tabbed, sep="\t"):
    """Creates a dict from key equal value pairs seperated with tab"""
    return dict([pair for pair in tabbed_to_dict_generator(tabbed, sep)])
