/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <gtk/gtk.h>

#include "fileutil.h"

gboolean
is_readable(gchar *path)
{
	struct stat file;
	gint uid = getuid();
	gint gid = getgid();

	if (!path) return FALSE;
	if (stat(path, &file)) return FALSE;

	if (file.st_mode & S_IROTH)
		return TRUE;
	else if (gid == file.st_gid && (file.st_mode & S_IRGRP))
		return TRUE;
	else if (uid == file.st_uid && (file.st_mode & S_IRUSR))
		return TRUE;

	return FALSE;
}

gboolean
is_writable(gchar *path)
{
	gint uid = getuid();
	gint gid = getgid();
	struct stat file;

	if (stat(path, &file)) return FALSE;

	if (file.st_mode & S_IWOTH)
		return TRUE;
	else if (gid == file.st_gid && (file.st_mode && S_IWGRP))
		return TRUE;
	else if (uid == file.st_uid && (file.st_mode && S_IWUSR))
		return TRUE;

	return FALSE;
}

GList *
get_dir_entries(gchar *src_path,
		gboolean dot_filter,
		gboolean recursive)
{
	struct dirent *entry;
	DIR *dir;
	gchar *path;
	GList *entries = NULL;

	if (!src_path) return NULL;
	if (!g_file_test(src_path, G_FILE_TEST_IS_DIR)) return NULL;

	path = g_strdup(src_path);

	dir = opendir(path);
	if (!dir) return NULL;
  
	while ((entry = readdir(dir)) != NULL) {
		gchar *name, *new_path;

		name = entry->d_name;
		if ((name[0] == '.') &&
		    ((name[1] == '\0') || (name[1] == '.'))) continue;
		if (dot_filter && name[0] == '.') continue;

		if (strcmp(path, "/"))
			new_path = g_strconcat(path, "/", name, NULL);
		else
			new_path = g_strconcat("/", name, NULL);

		if (!g_file_test(new_path, G_FILE_TEST_EXISTS)) {
			g_free(new_path);
			continue;
		}

		if (g_file_test(new_path, G_FILE_TEST_IS_DIR) && recursive) {
			GList *entries_node = NULL;

			entries_node = get_dir_entries(new_path, dot_filter, recursive);
	  
			entries = g_list_concat(entries_node, entries);
		}
		else {
			entries = g_list_append(entries, new_path);
		}
	} 
	closedir(dir);

	if (recursive) entries = g_list_append(entries, path);

	return entries;
}

void
free_dir_entries(GList *entries)
{
	GList *node;

	if (!entries) return;

	for (node = entries; node; node = node->next) {
		if (!node->data) continue;
		g_free(node->data);
	}

	g_list_free(entries);
}

/* gint */
/* count_dir_entries(gchar *path, */
/* 		  gboolean dot_filter) */
/* { */
/*   struct dirent *entry; */
/*   DIR *dir; */
/*   gint count = 0; */

/*   if (!path) return -1; */
/*   if (!g_file_test(path, G_FILE_TEST_IS_DIR)) return -1; */

/*   dir = opendir(path); */
/*   if (!dir) return -1; */
  
/*   while (entry = readdir(dir)) */
/*     { */
/*       if ((entry->d_name[0] == '.') && */
/* 	  (entry->d_name[1] == '\0') || (entry->d_name[1] == '.')) continue; */
 
/*       if (dot_filter && entry->d_name[0] == '.') continue; */

/*       count++; */
/*     } */
/*   closedir(dir); */

/*   return count; */
/* } */

gboolean
copy_file(gchar *src_path,
	  gchar *dest_path)
{
	gint size;
	gchar buf[BUFSIZ];
	FILE *src, *dest;

	if (!src_path || !dest_path) return FALSE;
	if (!g_file_test(src_path, G_FILE_TEST_IS_REGULAR)) return FALSE;

	src = fopen(src_path, "rb");
	if (!src) return FALSE;

	dest = fopen(dest_path, "wb");
	if (!dest) {
		fclose(src);
		return FALSE;
	}

	while ((size = fread(buf, sizeof(char), BUFSIZ, src)) > 0) {
		fwrite(buf, sizeof(char), size, dest);
		if (ferror(dest)) {
			fclose(src);
			fclose(dest);

			return FALSE;
		}
	}

	fclose(src);
	fclose(dest);

	return TRUE;
}

gboolean
move_file(gchar *src_path,
	  gchar *dest_path)
{
	/*   if (!copy_file(src_path, dest_path) || */
	/*       !remove(src_path)) return FALSE; */

	if (!copy_file(src_path, dest_path)) return FALSE;

	if (unlink(src_path)) return FALSE;

	return TRUE;
}

gboolean
delete_file(gchar *path)
{
	if (!path) return FALSE;

	if (unlink(path)) return FALSE;

	return TRUE;
}

gboolean
make_dir(gchar *path)
{
	if (!path) return FALSE;

	if (mkdir(path, S_IRWXU)) return FALSE;

	return TRUE;
}

gboolean
copy_dir(gchar *src_path,
	 gchar *dest_path)
{
	struct stat status;

	if (!src_path || !dest_path) return FALSE;
	if (stat(src_path, &status)) return FALSE;
	if (!S_ISDIR(status.st_mode)) return FALSE;

	if (!mkdir(dest_path, status.st_mode)) return FALSE;

	return TRUE;
}

gboolean
move_dir(gchar *src_path,
	 gchar *dest_path)
{
	GList *first, *entries = NULL;

	if (!copy_dir(src_path, dest_path)) return FALSE;

	first = get_dir_entries(src_path, TRUE, TRUE);
	for (entries = first; entries; entries = entries->next) {
		if (remove(entries->data)) {
			free_dir_entries(first);
			return FALSE;
		}
	}
	
	free_dir_entries(first);

	return TRUE;
}

gboolean
delete_dir(gchar *path)
{
	if (!path) return FALSE;

	if (rmdir(path)) return FALSE;

	return TRUE;
}

