/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <unistd.h>

#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba-ui.h"
#include "file-utils.h"
#include "dirview.h"
#include "dialog.h"

/*
 *---------------------------------------------------------
 *       Progress Dialog
 *---------------------------------------------------------
 */
static void fb_progress_dialog_class_init(FbProgDialogClass *klass);
static void fb_progress_dialog_init(FbProgDialog *dialog);
static void fb_progress_dialog_response(GtkDialog *dialog, gint arg);
static void fb_progress_dialog_finalize(GObject *object);

static GtkDialogClass *parent_class = NULL;

GType
fb_progress_dialog_get_type(void)
{
  static GType object_type = 0;

  if (!object_type) {
	  static const GTypeInfo object_info = {
		  sizeof (FbProgDialogClass),
		  NULL, /* base_init */
		  NULL, /* base_finalize */
		  (GClassInitFunc) fb_progress_dialog_class_init,
		  NULL, /* class_finalize */
		  NULL, /* class_data */
		  sizeof (FbProgDialog),
		  32, /* n_preallocs */
		  (GInstanceInitFunc) fb_progress_dialog_init,
	  };

	  object_type = g_type_register_static(GTK_TYPE_DIALOG, "FbProgDialog",
					       &object_info, 0);
  }

  return object_type;
}

static void
fb_progress_dialog_class_init(FbProgDialogClass *klass)
{
	GObjectClass *object_class;
	GtkDialogClass *dialog_class;

	parent_class = g_type_class_peek_parent(klass);
	object_class = (GObjectClass *)klass;
	dialog_class = (GtkDialogClass *)klass;

	object_class->finalize = fb_progress_dialog_finalize;

	dialog_class->response = fb_progress_dialog_response;
}

static void
fb_progress_dialog_init(FbProgDialog *dialog)
{
	GtkWidget *main_vbox, *align;
	GtkWidget *label, *message, *progress;

	gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 120);
	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
	gtk_dialog_add_button(GTK_DIALOG(dialog),
			      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

	main_vbox = GTK_DIALOG(dialog)->vbox;

	label = gtk_label_new(NULL);
	gtk_box_pack_start(GTK_BOX(main_vbox), label, TRUE, FALSE, 0);
	gtk_widget_show(label);

	align = gtk_alignment_new(0.5, 0.5, 0.7, 1.0);
	gtk_box_pack_start(GTK_BOX(main_vbox), align, TRUE, FALSE, 0);
	gtk_widget_show(align);

	progress = gtk_progress_bar_new();
	gtk_container_add(GTK_CONTAINER(align), progress);
	gtk_widget_show(progress);

	message = gtk_label_new(NULL);
	gtk_box_pack_start(GTK_BOX(main_vbox), message, TRUE, FALSE, 0);
	gtk_widget_show(message);

	dialog->label = label;
	dialog->progress = progress;
	dialog->message = message;
	dialog->interrupt = FALSE;
}

GtkWidget *
fb_progress_dialog_new(gchar *text)
{
	GObject *object;

	object = g_object_new(FB_TYPE_PROGRESS_DIALOG, NULL);

	if (text) {
		FbProgDialog *dialog;

		dialog = FB_PROGRESS_DIALOG(object);
		fb_progress_dialog_set_label(dialog, text);
	}

	return GTK_WIDGET(object);
}

static void
fb_progress_dialog_response(GtkDialog *dialog,
			    gint arg)
{
	if (arg == GTK_RESPONSE_CANCEL)
		FB_PROGRESS_DIALOG(dialog)->interrupt = TRUE;
}

static void
fb_progress_dialog_finalize(GObject *object)
{
	FbProgDialog *dialog;

	dialog = FB_PROGRESS_DIALOG(object);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(object);
}

void
fb_progress_dialog_set_label(FbProgDialog *dialog,
			     gchar *text)
{
	g_return_if_fail(FB_IS_PROGRESS_DIALOG(dialog));
	g_return_if_fail(text != NULL);

	gtk_label_set_text(GTK_LABEL(dialog->label), text);
}

void
fb_progress_dialog_set_message(FbProgDialog *dialog,
			       gchar *text)
{
	g_return_if_fail(FB_IS_PROGRESS_DIALOG(dialog));
	g_return_if_fail(text != NULL);

	gtk_label_set_text(GTK_LABEL(dialog->label), text);
}

void
fb_progress_dialog_set_progress(FbProgDialog *dialog,
				gdouble fraction,
				gchar *message)
{
	g_return_if_fail(FB_IS_PROGRESS_DIALOG(dialog));

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(dialog->progress), fraction);

	if (message)
		gtk_label_set_text(GTK_LABEL(dialog->message), message);
}

/*
 *---------------------------------------------------------
 *       About Dialog
 *---------------------------------------------------------
 */
GtkWidget *
fb_about_dialog_new(GtkWidget *parent)
{
	GtkWidget *dialog, *box, *label;

	dialog = gtk_dialog_new_with_buttons(_("About - Futaba"),
					     GTK_WINDOW(parent),
					     GTK_DIALOG_MODAL |
					     GTK_DIALOG_NO_SEPARATOR,
					     GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
					     NULL);
	gtk_window_set_default_size(GTK_WINDOW(dialog), 300, 120);
	g_signal_connect_swapped(G_OBJECT(dialog), "response",
				 G_CALLBACK(gtk_widget_destroy), dialog);

	box = GTK_DIALOG(dialog)->vbox;

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), 
			     "<span weight=\"bold\" size=\"xx-large\">Futaba</span>");
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), "<b>Futaba is Image Viewer.</b>");
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	label = gtk_label_new("Copyright (C) 2005 Futaba Project");
	gtk_box_pack_end(GTK_BOX(box), label, FALSE, FALSE, 0);
	gtk_widget_show(label);


	return dialog;
}

/*
 *-------------------------------------------------
 *           property dialog
 *-------------------------------------------------
 */
#define LABEL_TAB 15
#define LABEL_TAB2 0

GtkWidget *
fb_property_dialog_new(Futaba *futaba)
{
	gchar *name, *size, *path, *owner, *group, *permission, *mtime;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	GtkWidget *dialog;
	GtkWidget *table, *label;

	if (!futaba->treeview) return;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return;

	gtk_tree_model_get(model, &iter,
			   COLUMN_PATH, &path,
			   COLUMN_DISPLAY_PATH, &name,
			   COLUMN_PERMISSION, &permission,
			   COLUMN_OWNER, &owner,
			   COLUMN_GROUP, &group,
			   COLUMN_DISPLAY_SIZE, &size,
			   COLUMN_DISPLAY_MTIME, &mtime,
			   -1);


	dialog = gtk_dialog_new_with_buttons(_("Property"),
					     GTK_WINDOW(futaba->top),
					     GTK_DIALOG_MODAL |
					     GTK_DIALOG_NO_SEPARATOR,
					     GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
					     NULL);
	/*   gtk_window_set_default_size(GTK_WINDOW(dialog), 300, 200); */
	g_signal_connect_swapped(G_OBJECT(dialog), "response",
				 G_CALLBACK(gtk_widget_destroy), dialog);

	/* dialog->vbox area */
	table = gtk_table_new(7, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);
	gtk_widget_show(table);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>name:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 0, 1,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(name);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 0, 1,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>size:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 1, 2,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(size);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 1, 2,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>path:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 2, 3,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(path);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 2, 3,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>owner:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 3, 4,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(owner);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 3, 4,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>group:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 4, 5,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(group);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 4, 5,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>permission:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 5, 6,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(permission);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 5, 6,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	label = gtk_label_new(NULL);
	gtk_label_set_markup(GTK_LABEL(label), _("<b>mtime:</b>"));
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 0, 1, 6, 7,
			 GTK_FILL, 0, LABEL_TAB, 0);
	gtk_widget_show(label);

	label = gtk_label_new(mtime);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label,
			 1, 2, 6, 7,
			 GTK_FILL, 0, LABEL_TAB2, 0);
	gtk_widget_show(label);

	g_free(name);
	g_free(size);
	g_free(path);
	g_free(owner);
	g_free(group);
	g_free(permission);
	g_free(mtime);

	return dialog;
}

/*
 *-------------------------------------------------
 *           mkdir dialog
 *-------------------------------------------------
 */
static void cb_activate(GtkEntry *entry, gpointer data);
static void cb_mkdir_response(GtkDialog *dialog, gint arg, gpointer data);

static void
cb_activate(GtkEntry *entry,
	    gpointer data)
{
	GtkDialog *dialog = data;

	gtk_dialog_response(dialog, GTK_RESPONSE_OK);
}

static void
cb_mkdir_response(GtkDialog *dialog,
		  gint arg,
		  gpointer data)
{
	Futaba *futaba = data;

	switch (arg) {
	case GTK_RESPONSE_CANCEL:
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	case GTK_RESPONSE_OK: {
		gchar *string, *path;
		GtkWidget *entry = g_object_get_data(G_OBJECT(dialog), "entry");

		/* 	string = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1); */
		/* 	if (!string) return; */

		/* 	if (futaba->workdir == "/") */
		/* 	  path = g_strconcat("/", string, NULL); */
		/* 	else */
		/* 	  path = g_strconcat(futaba->workdir, "/", string, NULL); */
		/* 	fb_dirview_add_new_dir(futaba, path); */

		/* 	g_free(path); */
		/* 	gtk_widget_destroy(GTK_WIDGET(dialog)); */
	}
		break;
	}
}

GtkWidget *
fb_mkdir_dialog_new(Futaba *futaba)
{
	GtkWidget *dialog;
	GtkWidget *hbox, *label, *entry;

	dialog = gtk_dialog_new_with_buttons(_("Property"),
					     GTK_WINDOW(futaba->top),
					     GTK_DIALOG_MODAL |
					     GTK_DIALOG_NO_SEPARATOR,
					     GTK_STOCK_OK, GTK_RESPONSE_OK,
					     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					     NULL);
	gtk_window_set_default_size(GTK_WINDOW(dialog), 350, 100);
	g_signal_connect(G_OBJECT(dialog), "response",
			 G_CALLBACK(cb_mkdir_response), futaba);

	/* dialog->vbox area */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("name:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	entry = gtk_entry_new_with_max_length(64);
	g_signal_connect(G_OBJECT(entry), "activate",
			 G_CALLBACK(cb_activate), dialog);
	g_object_set_data(G_OBJECT(dialog), "entry", entry);
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_widget_show(entry);

	return dialog;
}
/*
 *-------------------------------------------------
 *           bulk change dialog
 *-------------------------------------------------
 */
GtkWidget *
fb_bulk_change_dialog_new(Futaba *futaba)
{
	GtkWidget *dialog;
	
	dialog = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dialog), _("change name in bulk"));
	gtk_window_set_default_size(GTK_WINDOW(dialog), 350, 100);
	gtk_window_set_destroy_with_parent(GTK_WINDOW(dialog), TRUE);
	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	return dialog;
}

/*
 *-----------------------------------------------------
 *       futaba rmdir function
 *-----------------------------------------------------
 */
static gboolean
fb_rm_file(gchar *path)
{
	GtkWidget *dialog;

	if (! path || ! *path) return FALSE;
	if (!path_is_exists(path)) return FALSE;

	if (unlink (path)) {
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						_("Couldn't delete \"%s\""), path);
		g_signal_connect_swapped(G_OBJECT(dialog), "response",
					 G_CALLBACK(gtk_widget_destroy), dialog);
		gtk_widget_show(dialog);

		return FALSE;
	}

	return TRUE;
}

static gboolean
fb_rm_dir(gchar *path)
{
	gint length, pos, result, is_delete;
	gfloat fraction;
	gchar *entry;
	GList *first, *entries;
	GtkWidget *prog_dlg, *dialog;

	if (! path || ! *path) return FALSE;
	if (!path_is_dir(path)) return FALSE;

	first = get_path_list(path, TRUE);

	length = g_list_length(first);
	if (length == 1) {
		path_list_free(first);

		if (rmdir (path)) {
			dialog = gtk_message_dialog_new(NULL,
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_ERROR,
							GTK_BUTTONS_OK,
							_("Couldn't delete"));
			g_signal_connect_swapped(G_OBJECT(dialog), "response",
						 G_CALLBACK(gtk_widget_destroy), dialog);
			gtk_widget_show(dialog);

			return FALSE;
		}

		return TRUE;
	}
	/* case: directory isn't empty */
	dialog = gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					_("Directory isn't empty.\nDelete it recursively ?"));
	result = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);

	if (result == GTK_RESPONSE_CANCEL) {
		path_list_free(first);
      
		return FALSE;
	}

	prog_dlg = fb_progress_dialog_new(_("Deleting..."));
	gtk_widget_show(prog_dlg);

	pos = 0;
	result = GTK_RESPONSE_NONE;
	for (entries = first; entries; entries = entries->next) {
		while (gtk_events_pending()) gtk_main_iteration();

		if (FB_PROGRESS_DIALOG(prog_dlg)->interrupt) {
			gtk_widget_destroy(prog_dlg);
			path_list_free(first);

			return FALSE;
		}

		entry = entries->data;
		if (path_is_dir( entry))
			is_delete = rmdir (entry);
		else
			is_delete = unlink (entry);

		if (is_delete && (result != GTK_RESPONSE_YES)) {
			dialog = gtk_message_dialog_new(NULL,
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_QUESTION,
							GTK_BUTTONS_YES_NO,
							_("Couldn't delete \"%s\"\nDo you continue ?"), entry);
			result = gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);

			if (result == GTK_RESPONSE_NO) {
				gtk_widget_destroy(prog_dlg);
				path_list_free(first);

				return FALSE;
			}
		}

		pos++;
		fraction = (gfloat) pos / (gfloat) length;
		fb_progress_dialog_set_progress(FB_PROGRESS_DIALOG(prog_dlg),
						fraction, entry);
	}

	path_list_free(first);
	gtk_widget_destroy(prog_dlg);

	return TRUE;
}

gboolean
fb_rmdir_start(gchar *path)
{
	gint result;
	GtkWidget *dialog;

	if (!path) return FALSE;
	if (!path_is_exists(path)) return FALSE;

	dialog = gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					_("Are you sure you want to permanently delete \"%s\"\n?"), path);
	result = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);

	if (result == GTK_RESPONSE_CANCEL) return FALSE;

	/* delete file */
	if (path_is_dir(path))
		result = fb_rm_dir(path);
	else
		result = fb_rm_file(path);

	return result;
}

/*
 *-------------------------------------------
 *     futaba paste function
 *-------------------------------------------
 */
static gboolean
fb_paste_file(gchar *src_path,
	      gchar *dest_path,
	      gboolean is_copy)
{
	gint is_rewrite = GTK_RESPONSE_NONE;
	GtkWidget *dialog;

	if (!src_path || !dest_path) return FALSE;

	if (path_is_exists(dest_path)) {
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_QUESTION,
						GTK_BUTTONS_OK_CANCEL,
						_("File exists\nRewrite  ?"));
		is_rewrite = gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);

		if (is_rewrite == GTK_RESPONSE_CANCEL) return FALSE;
	}

	if (!file_copy(src_path, dest_path)) {
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						_("Writing failed"));
		g_signal_connect_swapped(G_OBJECT(dialog), "response",
					 G_CALLBACK(gtk_widget_destroy), dialog);
		gtk_widget_show(dialog);

		return FALSE;
	}

	if (!is_copy) {
		if (unlink(src_path)) {
			dialog = gtk_message_dialog_new(NULL,
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_ERROR,
							GTK_BUTTONS_OK,
							_("Copy done\nBut couldn't delete"));
			g_signal_connect_swapped(G_OBJECT(dialog), "response",
						 G_CALLBACK(gtk_widget_destroy), dialog);
			gtk_widget_show(dialog);
		}
	}

	return TRUE;
}

static gchar *
convert_path(gchar *src_path,
	     gchar *pattern,
	     gchar *replace)
{
	gint i, len;
	gchar *tmp, *new_path;

	if (!src_path || !pattern || !replace) return NULL;

	i = 0;
	len = 0;

	while (pattern[len] != '\0') len++;

	tmp = g_strdup(src_path);
	while (tmp[i] == pattern[i]) {
		tmp[i] = ' ';
		i++;

		if (tmp[i] == '\0' || pattern[i] == '\0') break;
	}
	if (i != len) return NULL;

	tmp = g_strchug(tmp);

	if (tmp)
		new_path = g_strconcat(replace, tmp, NULL);
	else
		new_path = g_strdup(replace);

	g_free(tmp);

	return new_path;
}

static gboolean
fb_paste_dir(gchar *src_path,
	     gchar *dest_path,
	     gboolean is_copy)
{
	gboolean result = TRUE;
	gint length, pos, response;
	gfloat fraction;
	gchar *src_file, *dest_file;
	GList *first, *entries;
	GtkWidget *prog_dlg, *dialog;

	if (! src_path || ! *src_path) return FALSE;
	if (! dest_path || ! *dest_path) return FALSE;

	first = get_path_list(src_path, TRUE);
	first = g_list_reverse(first);

	pos = 0;
	length = g_list_length(first);
	response = GTK_RESPONSE_NONE;

	prog_dlg = fb_progress_dialog_new(_("Copy..."));
	gtk_widget_show(prog_dlg);

	for (entries = first; entries; entries = entries->next) {
		while (gtk_events_pending()) gtk_main_iteration();

		if (FB_PROGRESS_DIALOG(prog_dlg)->interrupt) {
			gtk_widget_destroy(prog_dlg);
			path_list_free(first);

			return FALSE;
		}

		src_file = entries->data;
		/* FIX ME (whether copy only displayed file) */
		dest_file = convert_path(src_file, src_path, dest_path);
		if (!dest_file) continue;

		/* check destination */
		if ((response != GTK_RESPONSE_APPLY) && path_is_exists (dest_file)) {
			dialog = gtk_message_dialog_new(NULL,
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_QUESTION,
							GTK_BUTTONS_NONE,
							_("\"%s\" exists!!\nRewrite ?"), dest_file);
			gtk_dialog_add_buttons(GTK_DIALOG(dialog),
					       GTK_STOCK_NO, GTK_RESPONSE_NO,
					       GTK_STOCK_YES, GTK_RESPONSE_YES,
					       _("All"), GTK_RESPONSE_APPLY,
					       NULL);
			response = gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);

			if (response == GTK_RESPONSE_NO) {
				gtk_widget_destroy(prog_dlg);
				path_list_free(first);
				g_free(dest_file);
	     
				return FALSE;
			}
		}

		if (path_is_dir(src_file))
			result = MIN(result, dir_copy(src_file, dest_file));
		else
			result = MIN(result, file_copy(src_file, dest_file));
		g_free(dest_file);

		pos++;
		fraction = (gfloat) pos / (gfloat) length;
		fb_progress_dialog_set_progress(FB_PROGRESS_DIALOG(prog_dlg),
						fraction, src_file);
	}

	if (!result) {
		dialog = gtk_message_dialog_new(NULL,
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						_("Couldn't copy completely"));
		g_signal_connect_swapped(G_OBJECT(dialog), "response",
					 G_CALLBACK(gtk_widget_destroy), dialog);
		gtk_widget_show(dialog);

		return FALSE;
	}

	if (!is_copy && result) {
		gint is_delete = 0;

		pos = 0;
		fb_progress_dialog_set_label(FB_PROGRESS_DIALOG(prog_dlg), _("Delete..."));
		fb_progress_dialog_set_progress(FB_PROGRESS_DIALOG(prog_dlg), 0, NULL);

		for (entries = g_list_reverse(first); entries; entries = entries->next) {
			while (gtk_events_pending()) gtk_main_iteration();

			if (path_is_dir( entries->data))
				is_delete = MIN(is_delete, rmdir(entries->data));
			else
				is_delete = MIN(is_delete, unlink(entries->data));

			pos++;
			fraction = (gfloat) pos / (gfloat) length;
			fb_progress_dialog_set_progress(FB_PROGRESS_DIALOG(prog_dlg),
							fraction, entries->data);
		}

		if (is_delete) {
			dialog = gtk_message_dialog_new(NULL,
							GTK_DIALOG_MODAL,
							GTK_MESSAGE_ERROR,
							GTK_BUTTONS_OK,
							_("Copy done\nBut couldn't delete completely"));
			g_signal_connect_swapped(G_OBJECT(dialog), "response",
						 G_CALLBACK(gtk_widget_destroy), dialog);
			gtk_widget_show(dialog);
		}
	}
  
	path_list_free(first);
	gtk_widget_destroy(prog_dlg);

	return TRUE;
}

gboolean
fb_paste_start(Futaba *futaba,
	       gchar *dest_path)
{
	gboolean result;

	if (!futaba->paste_path) return FALSE;
	if (! dest_path || ! *dest_path) return FALSE;

	if (path_is_dir(futaba->paste_path))
		result = fb_paste_dir(futaba->paste_path, dest_path, futaba->is_copy);
	else
		result = fb_paste_file(futaba->paste_path, dest_path, futaba->is_copy);

	if (result) fb_dirview_add_file(futaba, dest_path);

	return result;
}
