/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba-ui.h"
#include "history.h"
#include "dirview.h"
#include "file-utils.h"

#define HISTORY_STOCK 25

typedef struct _FbHistory FbHistory;

struct _FbHistory
{
	GtkWidget *item;
	gchar *path;
	gchar *label;
	gchar *message;
	guint id;
	FbFileType type;
};

static FbHistory *history_new(Futaba *futaba, gchar *path, guint length);
static void history_destroy(FbHistory *fhis);
static void dir_history_remove(FbDirHistory *history, gint nth);

static void
cb_change_dir(GtkMenuItem *item,
	      gpointer data)
{
	gint i, new_pos;
	FbHistory *fhis = g_object_get_data(G_OBJECT(item), "history");
	Futaba *futaba = data;

	fb_dirview_set_history(futaba, fhis->path);

	new_pos = futaba->history->length - fhis->id - 1;

	/* change pulldown menu item */
	if (new_pos > futaba->history->position) {
		for (i = futaba->history->position + 1; i <= new_pos; i++) {
			fhis = g_list_nth_data(futaba->history->record, i);
			g_object_ref(fhis->item);
			gtk_container_remove(GTK_CONTAINER(futaba->history->prev), fhis->item);
		}

		for (i = new_pos - 1; i >= futaba->history->position; i--) {
			fhis = g_list_nth_data(futaba->history->record, i);
			gtk_menu_shell_prepend(GTK_MENU_SHELL(futaba->history->next), fhis->item);
			gtk_widget_show(fhis->item);
		}
	}
	else {
		for (i = futaba->history->position - 1; i >= new_pos; i--) {
			fhis = g_list_nth_data(futaba->history->record, i);
			g_object_ref(fhis->item);
			gtk_container_remove(GTK_CONTAINER(futaba->history->next), fhis->item);
		}

		for (i = new_pos + 1; i <= futaba->history->position; i++) {
			fhis = g_list_nth_data(futaba->history->record, i);
			gtk_menu_shell_prepend(GTK_MENU_SHELL(futaba->history->prev), fhis->item);
			gtk_widget_show(fhis->item);
		}
	}
  
	futaba->history->position = new_pos;

	fb_toolbar_set_sensitive(futaba);
}

static FbHistory *
history_new(Futaba *futaba,
	    gchar *set_path,
	    guint length)
{
	gchar *base, *label, *message, *path;
	GtkWidget *item, *icon;
	FbFileType type;
	FbHistory *fhis = g_new0(FbHistory, 1);

	if (!set_path) return NULL;

	path = g_strdup(set_path);
	base = g_path_get_basename(path);
	label = str_local_to_utf8(base);
	g_free(base);

	message = g_strdup_printf(_("Go to \"%s\""), path);
	item = gtk_image_menu_item_new_with_label(label);
	g_signal_connect(G_OBJECT(item), "activate",
			 G_CALLBACK(cb_change_dir), futaba);
	g_object_set_data(G_OBJECT(item), "history", fhis);
	fb_widget_set_message(item, message);

	if (path_is_dir(path)) {
		type = DIRECTORY;
		icon = gtk_image_new_from_stock("fb-directory", GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), icon);
		gtk_widget_show(icon);
	}
	else if (path_is_file(path) && file_is_image(path)) {
		type = IMAGE;
	}

	fhis->item = item;
	fhis->path = path;
	fhis->label = label;
	fhis->message = message;
	fhis->id = length;
	fhis->type = type;

	return fhis;
}

static void
history_destroy(FbHistory *fhis)
{
	if (!fhis) return;

	gtk_widget_destroy(fhis->item);
	g_free(fhis->path);
	g_free(fhis->label);
	g_free(fhis->message);
	g_free(fhis);
}

static void
dir_history_remove(FbDirHistory *history,
		   gint nth)
{
	FbHistory *fhis;

	fhis = g_list_nth_data(history->record, nth);
	history->record = g_list_remove(history->record, fhis);
	history_destroy(fhis);

	history->length--;
}

FbDirHistory *
fb_dir_history_new(void)
{
	FbDirHistory *history;

	history = g_new0(FbDirHistory, 1);
	history->prev = gtk_menu_new();
	history->next = gtk_menu_new();
	history->record = NULL;
	history->position = 0;
	history->length = 0;

	return history;
}

void
fb_dir_history_add(Futaba *futaba,
		   gchar *path)
{
	gint i;
	FbHistory *fhis;

	if (futaba->history->position > 0) {
		for (i = 0; i < futaba->history->position; i++)
			dir_history_remove(futaba->history, 0);

		futaba->history->position = 0;
	}

	if (futaba->history->length > HISTORY_STOCK)
		dir_history_remove(futaba->history, HISTORY_STOCK);

	/* keep stock new fhis */
	fhis = history_new(futaba, path, futaba->history->length);
	futaba->history->record = g_list_prepend(futaba->history->record, fhis);
	futaba->history->length++;

	/* add prev fhis to "prev" popdown menu */
	if (futaba->history->length > 1) {
		fhis = g_list_nth_data(futaba->history->record, 1);
		gtk_menu_shell_prepend(GTK_MENU_SHELL(futaba->history->prev), fhis->item);
		gtk_widget_show(fhis->item);
	}

	fb_toolbar_set_sensitive(futaba);
}

gchar *
fb_dir_history_back(FbDirHistory *history)
{
	FbHistory *fhis;

	if (history->position > (history->length - 2)) return NULL;

	/* add stock fhis to "next" pulldown menu */
	fhis = g_list_nth_data(history->record, history->position);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(history->next), fhis->item);
	gtk_widget_show(fhis->item);

	history->position++;

	/* remove prev fhis, and keep stock */
	fhis = g_list_nth_data(history->record, history->position);
	g_object_ref(fhis->item);
	gtk_container_remove(GTK_CONTAINER(history->prev), fhis->item);

	return g_strdup(fhis->path);
}

gchar *
fb_dir_history_next(FbDirHistory *history)
{
	FbHistory *fhis;

	if (history->position <= 0) return NULL;

	/* add fhis to "prev" pulldown menu */
	fhis = g_list_nth_data(history->record, history->position);
	gtk_menu_shell_prepend(GTK_MENU_SHELL(history->prev), fhis->item);
	gtk_widget_show(fhis->item);

	history->position--;

	/* remove next fhis */
	fhis = g_list_nth_data(history->record, history->position);
	g_object_ref(fhis->item);
	gtk_container_remove(GTK_CONTAINER(history->next), fhis->item);

	return g_strdup(fhis->path);
}

void
fb_dir_history_clear(Futaba *futaba)
{
	GList *node;
	FbHistory *fhis;

	if (!futaba->history) return;

	node = g_list_first(futaba->history->record);
	for (; node; node = g_list_next(node)) {
		fhis = node->data;
		history_destroy(fhis);
	}

	g_list_free(futaba->history->record);
	futaba->history->record = NULL;
	futaba->history->position = 0;
	futaba->history->length = 0;

	fb_dir_history_add(futaba, futaba->workdir);
}
