/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "fb-image.h"
#include "fb-conf.h"
#include "pixbuf-utils.h"

#define CURSOR_WATCH 1000
#define CURSOR_TIMEOUT 2000

static void     fb_image_class_init          (FbImageClass     *klass);
static void     fb_image_init                (FbImage          *image);
static void     fb_image_finalize            (GObject          *object);
static void     fb_image_size_allocate       (GtkWidget        *widget,
					      GtkAllocation    *allocation);
static void     fb_image_realize             (GtkWidget        *widget);
static gboolean fb_image_button_press        (GtkWidget        *widget,
					      GdkEventButton   *button);
static gboolean fb_image_expose_event        (GtkWidget        *widget,
					      GdkEventExpose   *event);
static gboolean fb_image_motion_notify       (GtkWidget        *widget,
					      GdkEventMotion   *event);
static gboolean fb_image_enter_notify        (GtkWidget        *widget,
					      GdkEventCrossing *event);
static gboolean fb_image_leave_notify        (GtkWidget        *widget,
					      GdkEventCrossing *event);
static FbTile  *fb_image_get_tile_at_pointer (FbImage          *image);
static void     fb_image_move_image          (FbImage          *image,
					      gint              dx,
					      gint              dy);
static void     fb_image_magnify_init        (FbImage          *image,
					      gint              x,
					      gint              y);
static void     fb_image_magnify_fin         (FbImage          *image);
static void     fb_image_magnify             (FbImage          *image,
					      gint              x,
					      gint              y);
/* util */
static void     store_status                 (FbImage          *image);
static void     restore_status               (FbImage          *image);
static void     cursor_unvisible             (GtkWidget        *widget);
static gboolean cursor_timeout_func          (gpointer          data);
static gboolean cursor_watch_func            (gpointer          data);
static gboolean rectangle_diff               (GdkRectangle     *src,
					      GdkRectangle     *dest,
					      GdkRectangle     *harea,
					      GdkRectangle     *varea);

static GtkWidgetClass *parent_class = NULL;
/*
 *-------------------------------------------------
 *       Utility
 *-------------------------------------------------
 */

static void
cursor_unvisible(GtkWidget *widget)
{
	GdkPixmap *src;
	GdkCursor *cursor;

	src = gdk_bitmap_create_from_data(NULL, "\0\0\0", 1, 1);

	cursor = gdk_cursor_new_from_pixmap(src, src,
					    &widget->style->fg[GTK_STATE_ACTIVE],
					    &widget->style->bg[GTK_STATE_ACTIVE],
					    0, 0);

	gdk_window_set_cursor(widget->window, cursor);

	gdk_cursor_unref(cursor);
	gdk_pixmap_unref(src);

}

static gboolean
cursor_timeout_func (gpointer data)
{
	FbImage *image;

	if (! GTK_IS_WIDGET (data)) return FALSE;

	image = FB_IMAGE (data);
	if (! image->is_enter) return FALSE;

	cursor_unvisible (GTK_WIDGET (data));

	image->cursor_timer_id = -1;

	return FALSE;
}

static gboolean
cursor_watch_func (gpointer data)
{
	FbImage *image;

/* 	g_print ("FbImage::cursor watch\n"); */
	if (! FB_IS_IMAGE (data)) return FALSE;

	image = FB_IMAGE (data);
	if (! image->is_enter) return FALSE;

        if (! image->cursor_move && image->cursor_timer_id == -1) {
		image->cursor_timer_id = g_timeout_add (CURSOR_TIMEOUT,
							(GSourceFunc) cursor_timeout_func, image);
	}

	return TRUE;
}

/* void */
/* fb_zoom_window_draw(FbImageCanvas *fic, */
/* 		    gint x, */
/* 		    gint y) */
/* { */
/* 	gint src_x, src_y; */
/* 	GdkPoint points[5]; */
/* 	GdkRectangle *zwin, *dest; */

/* 	if (!GDK_IS_PIXMAP(fic->zoom)) fb_imagecanvas_load_zoom(fic); */

/* 	dest = g_new0(GdkRectangle, 1); */
/* 	zwin = g_new0(GdkRectangle, 1); */
/* 	zwin->x = x - (zoom_width / 2); */
/* 	zwin->y = y - (zoom_height / 2); */
/* 	zwin->width = zoom_width; */
/* 	zwin->height = zoom_height; */

/* 	if (!gdk_rectangle_intersect(fic->canvas, zwin, dest)) { */
/* 		g_free(dest); */
/* 		g_free(zwin); */
/* 		return; */
/* 	} */

/* 	src_x = ((x + fic->x0 - fic->canvas->x) * zoom_scale) - (zoom_width / 2); */
/* 	src_y = ((y + fic->y0 - fic->canvas->y) * zoom_scale) - (zoom_height / 2); */

/* 	gdk_draw_drawable(fic->widget->window, */
/* 			  fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)], */
/* 			  fic->zoom, */
/* 			  src_x, src_y, dest->x, dest->y, */
/* 			  dest->width, dest->height); */

/* 	/* zoom window frame */
/* 	points[0].x = zwin->x + 1; */
/* 	points[0].y = zwin->y + 1; */
/* 	points[1].x = points[0].x; */
/* 	points[1].y = zwin->y + zoom_height - 2; */
/* 	points[2].x = zwin->x + zoom_width - 2; */
/* 	points[2].y = points[1].y; */
/* 	points[3].x = points[2].x; */
/* 	points[3].y = points[0].y; */
/* 	points[4].x = points[0].x; */
/* 	points[4].y = points[0].y; */

/* 	/*   points[0].x = zwin->x; */ 
/* 	/*   points[0].y = zwin->y; */ 
/* 	/*   points[1].x = points[0].x; */ 
/* 	/*   points[1].y = zwin->y + zoom_height; */ 
/* 	/*   points[2].x = zwin->x + zoom_width; */ 
/* 	/*   points[2].y = points[1].y; */ 
/* 	/*   points[3].x = points[2].x; */ 
/* 	/*   points[3].y = points[0].y; */ 
/* 	/*   points[4].x = points[0].x; */ 
/* 	/*   points[4].y = points[0].y; */ 

/* 	gdk_draw_lines(fic->widget->window, */
/* 		       fic->widget->style->white_gc, */
/* 		       points, 5); */

/* 	g_free(dest); */
/* 	g_free(zwin); */
/* } */

/*
 *-----------------------------------------------
 *       Tile
 *-----------------------------------------------
 */
enum {
	ROTATE_NONE = 1 << 0,
	ROTATE_90   = 1 << 1,
	ROTATE_180  = 1 << 2,
	ROTATE_270  = 1 << 3
};

struct _FbTile
{
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap; /* not use */

	gchar *path;

	GdkRectangle *area;
	gint image_x, image_y;
	gfloat scale;
	gint raw_image_w, raw_image_h;

	gint rotate;
	gboolean fitting;
	gboolean mirroring;
};

static FbTile *fb_tile_new (const gchar *path);
static void fb_tile_free_image (FbTile *tile);
static void fb_tile_free (FbTile *tile);
static void fb_tile_calc_fit_scale (FbTile *tile);
static void fb_tile_load_pixbuf (FbTile *tile);
static void fb_tile_draw (FbTile *tile,
			  GtkWidget *widget,
			  GdkRectangle *draw_area);
static void fb_tile_redraw (FbTile *tile,
			    GtkWidget *widget);

static FbTile *
fb_tile_new (const gchar *path)
{
	FbTile *tile = g_new0 (FbTile, 1);

	tile->pixbuf = NULL;
	tile->pixmap = NULL;

	tile->area = g_new0 (GdkRectangle, 1);

	tile->path = g_strdup (path);
/* 	gdk_pixbuf_get_file_info (tile->path, &tile->raw_image_w, &tile->raw_image_h); */
	tile->image_x = -1;
	tile->image_y = -1;
	tile->scale = -1;

	tile->rotate = ROTATE_NONE;
	tile->fitting = TRUE;
	tile->mirroring = FALSE;

	return tile;
}

static void
fb_tile_free_image (FbTile *tile)
{
	if (tile->pixbuf) {
		g_object_unref (tile->pixbuf);
		tile->pixbuf = NULL;
	}

	if (tile->pixmap) {
		g_object_unref (tile->pixmap);
		tile->pixmap = NULL;
	}

}

static void
fb_tile_free (FbTile *tile)
{
	fb_tile_free_image (tile);

	if (tile->path) {
		g_free (tile->path);
		tile->path = NULL;
	}

	if (tile->area) {
		g_free (tile->area);
		tile->area = NULL;
	}

	g_free (tile);
}

static void
fb_tile_calc_fit_scale (FbTile *tile)
{
	gfloat scale;

	g_return_if_fail (tile != NULL);

	scale = (gfloat) tile->area->height / (gfloat) tile->raw_image_h;

	if (tile->area->width < (tile->raw_image_w * scale))
		scale = (gfloat) tile->area->width / (gfloat) tile->raw_image_w;

	tile->scale = scale;
}

static void
fb_tile_load_pixbuf (FbTile *tile)
{
	gint img_w, img_h;
	GdkPixbuf *pixbuf, *tmp;

	/* note! have already setted tile->area */
	g_return_if_fail (tile != NULL);

	pixbuf = gdk_pixbuf_new_from_file (tile->path, NULL);
	g_return_if_fail (pixbuf != NULL);

	switch (tile->rotate) {
	case ROTATE_90: {
		tmp = pixbuf_copy_rotate_90 (pixbuf, FALSE);
		g_object_unref (pixbuf);
		pixbuf = tmp;
	}
		break;
	case ROTATE_180: {
		tmp = pixbuf_copy_mirror (pixbuf, TRUE, TRUE);
		g_object_unref (pixbuf);
		pixbuf = tmp;
	}
		break;
	case ROTATE_270: {
		tmp = pixbuf_copy_rotate_90 (pixbuf, TRUE);
		g_object_unref (pixbuf);
		pixbuf = tmp;
	}
		break;
	}

	if (tile->mirroring) {
		tmp = pixbuf_copy_mirror (pixbuf, TRUE, FALSE);
		g_object_unref (pixbuf);
		pixbuf = tmp;
	}

	tile->raw_image_w = gdk_pixbuf_get_width (pixbuf);
	tile->raw_image_h = gdk_pixbuf_get_height (pixbuf);

	if (tile->fitting || tile->scale <= 0) {
		fb_tile_calc_fit_scale (tile);
	}

	img_w = (gfloat) tile->raw_image_w * tile->scale;
	img_h = (gfloat) tile->raw_image_h * tile->scale;
/* 	g_print ("pixbuf w %d h %d ", tile->raw_image_w, tile->raw_image_h);  */
/* 	g_print ("scale %f ", tile->scale); */
/* 	g_print ("w %d h %d\n", img_w, img_h); */

	tile->image_x = (tile->area->width - img_w) / 2;
	tile->image_y = (tile->area->height - img_h) / 2;

/* 	tile->pixbuf = gdk_pixbuf_new_from_file_at_size (tile->path, img_w, img_h, NULL); */
	tile->pixbuf = gdk_pixbuf_scale_simple (pixbuf, img_w, img_h, GDK_INTERP_HYPER);
	g_object_unref (pixbuf);
}

static void
fb_tile_draw (FbTile *tile,
	      GtkWidget *widget,
	      GdkRectangle *draw_area)
{
	gint pixbuf_x, pixbuf_y, pixbuf_w, pixbuf_h;
	GdkRectangle *intersect;

	g_return_if_fail (tile != NULL);

	if (! tile->pixbuf) fb_tile_load_pixbuf (tile);

	pixbuf_w = gdk_pixbuf_get_width (tile->pixbuf);
	pixbuf_h = gdk_pixbuf_get_height (tile->pixbuf);

	intersect  = g_new0 (GdkRectangle, 1);
	intersect->x = tile->area->x + tile->image_x;
	intersect->y = tile->area->y + tile->image_y;
	intersect->width = pixbuf_w;
	intersect->height = pixbuf_h;

/* 	g_print ("expose x %d y %d w %d h %d\n", draw_area->x, draw_area->y, draw_area->width, draw_area->height); */
/* 	g_print ("tile   x %d y %d w %d h %d\n", tile->area->x, tile->area->y, tile->area->width, tile->area->height); */
/* 	g_print ("image  x %d y %d w %d h %d\n", intersect->x, intersect->y, intersect->width, intersect->height); */
	if (gdk_rectangle_intersect (draw_area, intersect, intersect)) {
		gdk_rectangle_intersect (tile->area, intersect, intersect);
/* 		g_print ("draw   x %d y %d w %d h %d\n", intersect->x, intersect->y, intersect->width, intersect->height); */

		pixbuf_x = intersect->x - tile->area->x - tile->image_x;
		pixbuf_y = intersect->y - tile->area->y - tile->image_y;
		if ((pixbuf_x + intersect->width) > pixbuf_w) {
			intersect->width = pixbuf_w - pixbuf_x;
		}
		if ((pixbuf_y + intersect->height) > pixbuf_h) {
			intersect->height = pixbuf_h - pixbuf_x;
		}
/* 		g_print ("src position  : x %d y %d\n", pixbuf_x, pixbuf_y); */

		gdk_draw_pixbuf(widget->window, 
				widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
				tile->pixbuf, pixbuf_x, pixbuf_y,
				intersect->x, intersect->y,
				intersect->width, intersect->height,
				GDK_RGB_DITHER_NORMAL, -1, -1);
	}
	g_free (intersect);
}

static void
fb_tile_redraw (FbTile *tile,
		GtkWidget *widget)
{
	g_return_if_fail (tile != NULL);
	/* NOTE (manipulating big image, performance will be slow) */
	fb_tile_free_image (tile);

	gdk_draw_rectangle (widget->window,
			    widget->style->bg_gc[GTK_STATE_NORMAL],
			    TRUE,
			    tile->area->x, tile->area->y,
			    tile->area->width, tile->area->height);

	fb_tile_draw (tile, widget, tile->area);
}

/*
 *-----------------------------------------------
 *     Image
 *-----------------------------------------------
 */
GType
fb_image_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbImageClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_image_class_init,
			NULL,
			NULL,
			sizeof (FbImage),
			32,
			(GInstanceInitFunc) fb_image_init,
		};

		object_type = g_type_register_static (GTK_TYPE_WIDGET, "FbImage",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_image_class_init (FbImageClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	gobject_class->finalize = fb_image_finalize;

/* 	object_class->destroy = fb_image_destroy; */

	widget_class->button_press_event = fb_image_button_press;
	widget_class->enter_notify_event = fb_image_enter_notify;
	widget_class->expose_event = fb_image_expose_event;
	widget_class->leave_notify_event = fb_image_leave_notify;
	widget_class->motion_notify_event = fb_image_motion_notify;
	widget_class->size_allocate = fb_image_size_allocate;
	widget_class->realize = fb_image_realize;
}

static void
fb_image_init (FbImage *image)
{

	image->tiles = NULL;
	image->nth_tile = -1;

	image->magnifying = FALSE;
	image->magnify_w = -1;
	image->magnify_h = -1;
	image->magnify_power = -1;
	image->magnify_pixbuf = NULL;
	image->prev_magnify_area = g_new0 (GdkRectangle, 1);

	image->open_way = LEFT_OPEN;

	image->timer_id = g_timeout_add (CURSOR_WATCH,
					 (GSourceFunc) cursor_watch_func, image);
	image->cursor_timer_id = -1;
	image->cursor_move = FALSE;
	image->is_enter = TRUE;

	image->pointer_last_x = -1;
	image->pointer_last_y = -1;

	image->drag_data = NULL;

	restore_status (image); 
}

static void
restore_status (FbImage *image)
{
	fb_conf_get_value ("magnify width", &image->magnify_w, FB_CONF_TYPE_INT, "100");
	fb_conf_get_value ("magnify height", &image->magnify_h, FB_CONF_TYPE_INT, "100");
	fb_conf_get_value ("magnify power", &image->magnify_power, FB_CONF_TYPE_FLOAT, "1.5");
}

static void
store_status (FbImage *image)
{
	fb_conf_set_value ("magnify width", &image->magnify_w, FB_CONF_TYPE_INT);
	fb_conf_set_value ("magnify height", &image->magnify_h, FB_CONF_TYPE_INT);
	fb_conf_set_value ("magnify power", &image->magnify_power, FB_CONF_TYPE_FLOAT);
}

static void
fb_image_finalize (GObject *object)
{
	GList *t;
	FbImage *image = FB_IMAGE (object);

/* 	g_print ("FbImage::finalize\n"); */

	store_status (image);

	for (t = image->tiles; t; t = t->next) {
		fb_tile_free (t->data);
	}
	g_list_free (image->tiles);

	g_free (image->prev_magnify_area);

	if (image->timer_id > 0) {
		g_source_remove (image->timer_id);
		image->timer_id = -1;
	}
	if (image->cursor_timer_id > 0) {
		g_source_remove (image->cursor_timer_id);
		image->cursor_timer_id = -1;
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void     
fb_image_realize (GtkWidget *widget)
{
	GdkWindowAttr attributes;
	gint attributes_mask;

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.event_mask = gtk_widget_get_events (widget)
		| GDK_SCROLL_MASK
		| GDK_POINTER_MOTION_MASK
		| GDK_BUTTON_PRESS_MASK
		| GDK_BUTTON_RELEASE_MASK
		| GDK_EXPOSURE_MASK
		| GDK_ENTER_NOTIFY_MASK
		| GDK_LEAVE_NOTIFY_MASK;

      
	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
      
	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
					 &attributes, attributes_mask);
	gdk_window_set_user_data (widget->window, widget);
  
	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

static gboolean
fb_image_button_press (GtkWidget *widget,
		       GdkEventButton *event)
{
	FbImage *image = FB_IMAGE (widget);

/* 	g_print ("FbImage::button-press\n"); */

	switch (event->button) {
	case 1:
		break;
	case 2: {
		image->magnifying = ! image->magnifying;
		if (image->magnifying)
			fb_image_magnify_init (image, event->x, event->y);
		else
			fb_image_magnify_fin (image);
	}
		break;
	}

	return TRUE;
}

static gboolean
fb_image_motion_notify(GtkWidget *widget,
		       GdkEventMotion *event)
{
	gint dx, dy;
	FbImage *image = FB_IMAGE (widget);

/* 	g_print ("FbImage::motion-notify x %d y %d\n", event->x, event->y); */
	image->cursor_move = TRUE;

	if (image->pointer_last_x == -1) 
		image->pointer_last_x = event->x;
	if (image->pointer_last_y == -1) 
		image->pointer_last_y = event->y;

	dx = event->x - image->pointer_last_x;
	dy = event->y - image->pointer_last_y;

	if (image->cursor_timer_id == -1) {
		if (! image->magnifying)
			gdk_window_set_cursor(widget->window, NULL);
	}
	else {
		g_source_remove (image->cursor_timer_id);
		image->cursor_timer_id = -1;
	}

	/* drag motion */
	if (event->state == GDK_BUTTON1_MASK)
		fb_image_move_image (image, dx, dy);

	/* magnify */
	if (image->magnifying && event->state != GDK_BUTTON1_MASK)
		fb_image_magnify (image, event->x, event->y);

	image->pointer_last_x = event->x;
	image->pointer_last_y = event->y;
	image->cursor_move = FALSE;

	return TRUE;
}

static gboolean
fb_image_enter_notify (GtkWidget *widget,
		       GdkEventCrossing *event)
{
	FbImage *image = FB_IMAGE (widget);

	if (event->type == GDK_ENTER_NOTIFY) {
		image->is_enter = TRUE;
		image->timer_id = g_timeout_add(CURSOR_WATCH,
						(GSourceFunc) cursor_watch_func, image);

		if (image->magnifying)
			cursor_unvisible (widget);
	}

	return TRUE;
}

static gboolean
fb_image_leave_notify (GtkWidget *widget,
		       GdkEventCrossing *event)
{
	FbImage *image = FB_IMAGE (widget);

	if (event->type == GDK_LEAVE_NOTIFY) {
/* 		g_print ("FbImage::leave-notify\n"); */
		image->is_enter = FALSE;
		if (image->timer_id > 0) {
			g_source_remove (image->timer_id);
			image->timer_id = -1;
		}

		if (image->magnifying) {
			gdk_window_invalidate_rect (widget->window, image->prev_magnify_area, TRUE);
			gdk_window_set_cursor (widget->window, NULL);
		}
	}

	return TRUE;
}

static gboolean
fb_image_expose_event(GtkWidget *widget,
		      GdkEventExpose *event)
{
	GList *t;
	FbTile *tile;

	if (! GTK_WIDGET_MAPPED (widget)) return;

	for (t = FB_IMAGE (widget)->tiles; t; t = t->next) {
		tile = t->data;

		fb_tile_draw (tile, widget, &event->area);
	}

	return FALSE;
}

static void
fb_image_size_allocate(GtkWidget *widget,
		       GtkAllocation *allocation)
{
	gdouble scale_w, scale_h;
	GList *t;
	FbTile *tile;
	FbImage *image = FB_IMAGE (widget);

/* 	g_print ("FbImage::size-allocate\n"); */
/* 	g_print ("widget   x %d y %d w %d h %d\n", */
/* 		 widget->allocation.x , widget->allocation.y , widget->allocation.width, widget->allocation.height); */
/* 	g_print ("allocate x %d y %d w %d h %d\n",  */
/* 		 allocation->x , allocation->y , allocation->width ,allocation->height); */
	scale_w = (gdouble) allocation->width / (gdouble) widget->allocation.width;
	scale_h = (gdouble) allocation->height / (gdouble) widget->allocation.height;

	if (scale_w <= 0) scale_w = 1.0;
	if (scale_h <= 0) scale_h = 1.0;

	for (t = image->tiles; t; t = t->next) {
		tile = t->data;

		if (tile->area->x != allocation->x)
			tile->area->x *= scale_w;
		if (tile->area->y != allocation->y)
			tile->area->y *= scale_h;
		tile->area->width *= scale_w;
		tile->area->height *= scale_h;
/* 		g_print ("tile     x %d y %d w %d h %d\n", */
/* 			 tile->area->x,tile->area->y,tile->area->width,tile->area->height); */
/* 		g_print ("scale x %f y %f\n", scale_w, scale_h); */

		/* adjustment value of area */
		if ((widget->allocation.width * scale_w) < allocation->width) {
			if (tile->area->x != allocation->x)
				tile->area->x++;
			tile->area->width++;
		}
		if ((widget->allocation.height * scale_h) < allocation->height) {
			if (tile->area->y != allocation->y)
				tile->area->y++;
			tile->area->height++;
		}

		tile->area->x = MAX (tile->area->x, allocation->x);
		tile->area->y = MAX (tile->area->y, allocation->y);
		tile->area->width = MIN (tile->area->width, allocation->width);
		tile->area->height = MIN (tile->area->height, allocation->height);

		/* update position of image */
		if (tile->fitting) {
			fb_tile_free_image (tile);
		}
		tile->image_x = (tile->area->width + tile->area->x - (tile->raw_image_w * tile->scale)) / 2;
		tile->image_y = (tile->area->height + tile->area->y - (tile->raw_image_h * tile->scale)) / 2;
	}

	if (GTK_WIDGET_CLASS (parent_class)->size_allocate)
		GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
}

GtkWidget *
fb_image_new (void)
{
	GObject *object = g_object_new (FB_TYPE_IMAGE, NULL);

	return GTK_WIDGET (object);
}

gint
fb_image_get_n_images (FbImage *image)
{
	g_return_if_fail (FB_IS_IMAGE (image));

	return g_list_length (image->tiles);
}

static void
fb_image_move_image (FbImage *image,
		     gint dx,
		     gint dy)
{
	gint img_x, img_y, img_w, img_h;
	gint draw_x, draw_y, draw_w, draw_h;
	GList *t;
	FbTile *tile;
	GtkWidget *widget;

	g_return_if_fail (FB_IS_IMAGE (image));

	widget = GTK_WIDGET (image);

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile || tile->fitting) return;

	img_w = gdk_pixbuf_get_width (tile->pixbuf);
	img_h = gdk_pixbuf_get_height (tile->pixbuf);
	if (img_w <= tile->area->width && img_h <= tile->area->height) return;

	img_x = CLAMP ((tile->image_x + dx), (tile->area->width - img_w), 0);
	img_y = CLAMP ((tile->image_y + dy), (tile->area->height - img_h), 0);
	if (img_x == tile->image_x && img_y == tile->image_y) return;

/* 	g_print ("prev image x %d y %d\n", tile->image_x, tile->image_y); */
	if (img_w > tile->area->width) tile->image_x = img_x;
	if (img_h > tile->area->height) tile->image_y = img_y;
/* 	g_print ("next image x %d y %d\n", tile->image_x, tile->image_y); */

	img_x = MAX (- tile->image_x, 0);
	img_y = MAX (- tile->image_y, 0);
	draw_x = MAX (tile->area->x, ((tile->area->width - img_w) / 2));
	draw_y = MAX (tile->area->y, ((tile->area->height - img_h) / 2));
	draw_w = MIN (img_w, tile->area->width);
	draw_h = MIN (img_h, tile->area->height);
	if (img_x + draw_w > img_w)
		draw_w = img_w - img_x;
	if (img_y + draw_h > img_h)
		draw_h = img_h - img_y;
/* 	g_print ("image x %4d y %4d\n", img_x, img_y); */
/* 	g_print ("draw  x %4d y %4d w %4d h %4d\n", draw_x, draw_y, draw_w, draw_h); */

	gdk_draw_rectangle (widget->window,
			    widget->style->bg_gc[GTK_STATE_NORMAL],
			    TRUE,
			    tile->area->x, tile->area->y,
			    tile->area->width, tile->area->height);

	gdk_draw_pixbuf (widget->window,
			 widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
			 tile->pixbuf, img_x, img_y,
			 draw_x, draw_y,
			 draw_w, draw_h,
			 GDK_RGB_DITHER_NORMAL, 0, 0);
/* 	fb_tile_draw (tile, widget, tile->area); */
}

static FbTile *
fb_image_get_tile_at_pointer (FbImage *image)
{
	gint p_x, p_y;
	GList *t;
	FbTile *tile;

	g_return_val_if_fail (FB_IS_IMAGE (image), NULL);

	if (g_list_length (image->tiles) == 0) return NULL;

	gtk_widget_get_pointer (GTK_WIDGET (image), &p_x, &p_y);

	for (t = image->tiles; t; t = t->next) {
		tile = t->data;

		if (((tile->area->x < p_x) && (p_x < tile->area->x + tile->area->width)) &&
		    ((tile->area->y < p_y) && (p_y < tile->area->y + tile->area->height)))
			return tile;
	}

	return tile;
}

void
fb_image_add_image(FbImage *image,
		   const gchar *path,
		   GtkOrientation way)
{
	gint tile_x, tile_y, tile_w, tile_h;
	GList *t;
	FbTile *parent, *child;

	g_return_if_fail (FB_IS_IMAGE (image));

	parent = fb_image_get_tile_at_pointer (image);
	if (parent) {
/* 		g_print ("src   x %4d y %4d w %4d h %4d\n", */
/* 			 parent->area->x, parent->area->y, parent->area->width, parent->area->height); */
		if (way == GTK_ORIENTATION_HORIZONTAL) {
			if (image->open_way == RIGHT_OPEN) {
				/* note!! not enough test */
				tile_w = parent->area->width / 2;
				tile_x = parent->area->x + tile_w;
				parent->area->width = tile_w;
			}
			else if (image->open_way == LEFT_OPEN) {
				tile_x = parent->area->x;
				tile_w = parent->area->width / 2;
				parent->area->x = tile_x + tile_w;
				parent->area->width = tile_w;
		/* 			g_print ("left  x %4d y %4d w %4d h %4d\n", */
		/* 				 parent->area->x, parent->area->y, parent->area->width, parent->area->height); */
			}
			tile_y = parent->area->y;
			tile_h = parent->area->height;
			fb_tile_redraw (parent, GTK_WIDGET (image));
		}
		else if (way == GTK_ORIENTATION_VERTICAL) {
			if (image->open_way == RIGHT_OPEN) {
				tile_y = parent->area->y;
				tile_h = parent->area->height / 2;
				parent->area->y = tile_y + tile_h;
				parent->area->height = tile_h;
			}
			else if (image->open_way == LEFT_OPEN) {
				/* note!! not enough test */
				tile_h = parent->area->height / 2;
				tile_y = parent->area->y + tile_h;
				parent->area->height = tile_h;
		/* 			g_print ("left  x %4d y %4d w %4d h %4d\n", */
		/* 				 parent->area->x, parent->area->y, parent->area->width, parent->area->height); */
			}
			tile_x = parent->area->x;
			tile_w = parent->area->width;
			fb_tile_redraw (parent, GTK_WIDGET (image));
		}
	}
	else {
/* 		tile_x = GTK_WIDGET (image)->allocation.x; */
/* 		tile_y = GTK_WIDGET (image)->allocation.y; */
		tile_x = 0;
		tile_y = 0;
		tile_w = GTK_WIDGET (image)->allocation.width;
		tile_h = GTK_WIDGET (image)->allocation.height;
	}

	child = fb_tile_new (path);
	child->area->x = tile_x;
	child->area->y = tile_y;
	child->area->width = tile_w;
	child->area->height = tile_h;
 
	fb_tile_redraw (child, GTK_WIDGET (image));

	image->tiles = g_list_append (image->tiles, child);
	image->nth_tile = g_list_index (image->tiles, child);
}

void
fb_image_remove_image (FbImage *image)
{
	gint last;
	GList *t;
	FbTile *tile, *rm_tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	if (g_list_length (image->tiles) == 1) return;

	last = g_list_length (image->tiles) - 1;
	rm_tile = g_list_nth_data (image->tiles, last);
	tile = g_list_nth_data (image->tiles, last - 1);

	if (tile->area->x != rm_tile->area->x) {
		tile->area->width += rm_tile->area->width;
	}
	else if (tile->area->y != rm_tile->area->y) {
		tile->area->height += rm_tile->area->height;
	}
	tile->area->x = MIN (tile->area->x, rm_tile->area->x);
	tile->area->y = MIN (tile->area->y, rm_tile->area->y);

	image->tiles = g_list_remove (image->tiles, rm_tile);
	fb_tile_free (rm_tile);

	if (last < image->nth_tile) image->nth_tile--;

	fb_tile_redraw (tile, GTK_WIDGET (image));
}

void
fb_image_set_file (FbImage *image,
		   const gchar *path)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	if (image->nth_tile >= g_list_length (image->tiles) - 1)
		image->nth_tile = 0;
	else
		image->nth_tile++;

	tile = g_list_nth_data (image->tiles, image->nth_tile);
	g_return_if_fail (tile != NULL);

	if (tile->path) g_free (tile->path);

	tile->path = g_strdup (path);
/* 	gdk_pixbuf_get_file_info (tile->path, &tile->raw_image_w, &tile->raw_image_h); */

	fb_tile_redraw (tile, GTK_WIDGET (image));
}

gint
fb_image_set_file_at_pointer (FbImage *image,
			      const gchar *path)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (tile) {
		g_free (tile->path);
		tile->path = g_strdup (path);
		fb_tile_redraw (tile, GTK_WIDGET (image));
	}

	return g_list_index (image->tiles, tile);
}

void
fb_image_zoom (FbImage *image,
	       gfloat ratio)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile) return;

	tile->fitting = FALSE;
        tile->scale *= ratio;
	fb_tile_redraw (tile, GTK_WIDGET (image));
}

void
fb_image_zoom_at_value (FbImage *image,
			gfloat value)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile) return;

	tile->fitting = FALSE;
	tile->scale = value;
	fb_tile_redraw (tile, GTK_WIDGET (image));
}

void
fb_image_fit_window (FbImage *image)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile) return;

	tile->fitting = TRUE;

	fb_tile_redraw (tile, GTK_WIDGET (image));
}

static gboolean
rectangle_diff(GdkRectangle *src,
	       GdkRectangle *dest,
	       GdkRectangle *harea,
	       GdkRectangle *varea)
{
	GdkRectangle *common;

	common = g_new0(GdkRectangle, 1);

	if (!gdk_rectangle_intersect(src, dest, common)) return FALSE;

	if ((dest->x == common->x) && (dest->y == common->y)) {
		harea->y = dest->y + common->height;
		varea->x = dest->x + common->width;
	}
	else if ((dest->x == common->x) && (dest->y != common->y)) {
		harea->y = dest->y;
		varea->x = dest->x + common->width;
	}
	else if ((dest->x != common->x) && (dest->y == common->y)) {
		harea->y = dest->y + common->height;
		varea->x = dest->x;
	}
	else if ((dest->x != common->x) && (dest->y != common->y)) {
		harea->y = dest->y;
		varea->x = dest->x;
	}

	harea->x = dest->x;
	harea->width = dest->width;
	harea->height = dest->height - common->height;

	varea->y = dest->y;
	varea->width = dest->width - common->width;
	varea->height = dest->height;

	g_free(common);

	return TRUE;
}

static void
fb_image_magnify_init (FbImage *image,
		       gint x,
		       gint y)
{
	gint src_x, src_y, width, height;
	GdkRectangle *magnify_area;
	GdkPixbuf *src;
	GtkWidget *widget;

	g_return_if_fail (FB_IS_IMAGE (image));
	g_return_if_fail (x >= 0 && y >= 0);

	widget = GTK_WIDGET (image);

	cursor_unvisible (widget);

	src = gdk_pixbuf_get_from_drawable (NULL, widget->window,
					    NULL,
					    0,0,
/* 					    widget->allocation.x, widget->allocation.y, */
					    0,0,
/* 					    widget->allocation.x, widget->allocation.y, */
					    widget->allocation.width, widget->allocation.height);
/* 	g_print ("req w %4d h %4d\n", widget->allocation.width, widget->allocation.height); */
/* 	g_print ("src w %4d h %4d\n", gdk_pixbuf_get_width (src), gdk_pixbuf_get_height (src)); */

	width = widget->allocation.width * image->magnify_power;
	height = widget->allocation.height * image->magnify_power;

	image->magnify_pixbuf = gdk_pixbuf_scale_simple (src, width, height, GDK_INTERP_HYPER);

	g_object_unref (src);

	magnify_area = g_new0 (GdkRectangle, 1);
	magnify_area->x = CLAMP ((x - (image->magnify_w / 2)), 0, (widget->allocation.width - image->magnify_w));
	magnify_area->y = CLAMP ((y - (image->magnify_h / 2)), 0, (widget->allocation.height - image->magnify_h));
	magnify_area->width = image->magnify_w;
	magnify_area->height = image->magnify_h;

	src_x = CLAMP (((x * image->magnify_power) - (image->magnify_w / 2)), 0, (width - image->magnify_w));
	src_y = CLAMP (((y * image->magnify_power) - (image->magnify_h / 2)), 0, (height - image->magnify_h));
/* 	g_print ("draw---------------------------\n"); */
/* 	g_print ("src  x %4d y %4d w %4d h %4d\n",src_x, src_y, */
/* 		 magnify_area->width, magnify_area->height); */
/* 	g_print ("dest x %4d y %4d\n", magnify_area->x, magnify_area->y); */
/* 	g_print ("-------------------------------\n"); */
	gdk_draw_pixbuf(widget->window, 
			widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
			image->magnify_pixbuf,
			src_x, src_y,
			magnify_area->x, magnify_area->y,
			magnify_area->width, magnify_area->height,
			GDK_RGB_DITHER_NORMAL, 0, 0);

	image->prev_magnify_area = magnify_area;
}

static void
fb_image_magnify_fin (FbImage *image)
{
	GtkWidget *widget;

	g_return_if_fail (FB_IS_IMAGE (image));

	widget = GTK_WIDGET (image);

	gdk_window_invalidate_rect (widget->window, image->prev_magnify_area, TRUE);
	gdk_window_set_cursor (widget->window, NULL);

	g_object_unref (image->magnify_pixbuf);
	image->magnify_pixbuf = NULL;

	g_free (image->prev_magnify_area);
	image->prev_magnify_area = NULL;
}

static void
fb_image_magnify (FbImage *image,
		  gint x,
		  gint y)
{
	gint src_x, src_y;
	GdkRectangle *harea, *varea, *magnify_area;
	GtkWidget *widget;

	g_return_if_fail (FB_IS_IMAGE (image));
	g_return_if_fail (x >= 0 && y >= 0);

	widget  = GTK_WIDGET (image);

	magnify_area = g_new0 (GdkRectangle, 1);
	magnify_area->x = CLAMP ((x - (image->magnify_w / 2)), 0, (widget->allocation.width - image->magnify_w));
	magnify_area->y = CLAMP ((y - (image->magnify_h / 2)), 0, (widget->allocation.height - image->magnify_h));
	magnify_area->width = image->magnify_w;
	magnify_area->height = image->magnify_h;

	harea = g_new0 (GdkRectangle, 1);
	varea = g_new0 (GdkRectangle, 1);

	rectangle_diff (magnify_area, image->prev_magnify_area, harea, varea);

	gdk_window_invalidate_rect (widget->window, harea, TRUE);
	gdk_window_invalidate_rect (widget->window, varea, TRUE);

	g_free (harea);
	g_free (varea);

	src_x = CLAMP (((x * image->magnify_power) - (image->magnify_w / 2)), 0, (gdk_pixbuf_get_width (image->magnify_pixbuf) - image->magnify_w));
	src_y = CLAMP (((y * image->magnify_power) - (image->magnify_h / 2)), 0, (gdk_pixbuf_get_height (image->magnify_pixbuf) - image->magnify_h));
/* 	g_print ("draw---------------------------\n"); */
/* 	g_print ("src  x %4d y %4d w %4d h %4d\n", src_x, src_y, */
/* 		 magnify_area->width, magnify_area->height); */
/* 	g_print ("dest x %4d y %4d\n", magnify_area->x, magnify_area->y); */
/* 	g_print ("-------------------------------\n"); */
	gdk_draw_pixbuf(widget->window, 
			widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
			image->magnify_pixbuf,
			src_x, src_y,
			magnify_area->x, magnify_area->y,
			magnify_area->width, magnify_area->height,
			GDK_RGB_DITHER_NORMAL, 0, 0);

	g_free (image->prev_magnify_area);

	image->prev_magnify_area = magnify_area;
}

void
fb_image_rotate (FbImage *image,
		 gboolean counter_clockwise)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile) return;

	if (counter_clockwise && tile->rotate == ROTATE_NONE)
		tile->rotate = ROTATE_270;
	else if (! counter_clockwise && tile->rotate == ROTATE_270)
		tile->rotate = ROTATE_NONE;
	else if (counter_clockwise)
		tile->rotate = tile->rotate >> 1;
	else
		tile->rotate = tile->rotate << 1;

	fb_tile_redraw (tile, GTK_WIDGET (image));
}

void
fb_image_mirror (FbImage *image,
		 gboolean mirror,
		 gboolean flip)
{
	FbTile *tile;

	g_return_if_fail (FB_IS_IMAGE (image));

	tile = fb_image_get_tile_at_pointer (image);
	if (! tile) return;

	tile->mirroring = ! tile->mirroring;
	fb_tile_redraw (tile, GTK_WIDGET (image));
}
