/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <gtk/gtk.h>

#include "futaba.h"
#include "fb-conf.h"
#include "fb-thumbwin.h"
#include "fb-thumbwin-action.h"
#if GTK_MINOR_VERSION < 6
#include "gtkiconview.h"
#endif
#include "file-utils.h"
#include "eel-graphic-effects.h"

#define THUMBWIN_MARGIN 15
#define FRAME_LEFT_OFFSET 3
#define FRAME_RIGHT_OFFSET 6
#define FRAME_TOP_OFFSET 3
#define FRAME_BOTTOM_OFFSET 6

static void     fb_thumbwin_class_init         (FbThumbWinClass *klass);
static void     fb_thumbwin_init               (FbThumbWin      *tw);
static void     fb_thumbwin_destroy            (GtkObject       *object);
static void     fb_thumbwin_finalize           (GObject         *object);
static gboolean fb_thumbwin_focus_in_event     (GtkWidget       *widget,
						GdkEventFocus   *event);
static void     fb_thumbwin_size_allocate      (GtkWidget       *widget,
						GtkAllocation   *allocation);
static gboolean fb_thumbwin_button_release     (GtkWidget *widget,
						 GdkEventButton *event);
/* utils */
static void     store_status               (FbThumbWin      *tw);
static void     restore_status             (FbThumbWin      *tw);
/* call backs */
static gboolean cb_button_press       (GtkWidget *icon_view,
				       GdkEventButton *event,
				       gpointer data);
static void     cb_drag_begin         (GtkWidget *icon_view,
				       GdkDragContext *drag_context,
				       gpointer data);
static void     cb_drag_data_get      (GtkWidget *icon_view,
				       GdkDragContext *drag_context,
				       GtkSelectionData *selection_data,
				       guint info,
				       guint time,
				       gpointer data);
static void     cb_drag_data_received (GtkWidget *icon_view,
				       GdkDragContext *drag_context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info,
				       guint time,
				       gpointer data);
static gboolean cb_motion_notify      (GtkWidget *icon_view,
				       GdkEventMotion *event,
				       gpointer data);
/* sort funcs */
static gint     sort_name             (GtkTreeModel *model,
				       GtkTreeIter  *a,
				       GtkTreeIter  *b,
				       gpointer      data);
static gint     sort_mtime            (GtkTreeModel *model,
				       GtkTreeIter  *a,
				       GtkTreeIter  *b,
				       gpointer      data);
static gint     sort_size             (GtkTreeModel *model,
				       GtkTreeIter  *a,
				       GtkTreeIter  *b,
				       gpointer      data);
static gint     sort_area             (GtkTreeModel *model,
				       GtkTreeIter  *a,
				       GtkTreeIter  *b,
				       gpointer      data);

static GtkWindowClass *parent_class = NULL;

GtkWidget *focus_thumbwin = NULL;
GList *thumbwin_list = NULL;
static GdkPixbuf *thumbnail_frame = NULL;


GType
fb_thumbwin_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbThumbWinClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_thumbwin_class_init,
			NULL,
			NULL,
			sizeof (FbThumbWin),
			32,
			(GInstanceInitFunc) fb_thumbwin_init,
		};

		object_type = g_type_register_static (GTK_TYPE_WINDOW, "FbThumbWin",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_thumbwin_class_init (FbThumbWinClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	gobject_class->finalize = fb_thumbwin_finalize;

	object_class->destroy = fb_thumbwin_destroy;

	widget_class->focus_in_event = fb_thumbwin_focus_in_event;
	widget_class->size_allocate = fb_thumbwin_size_allocate;
}

static gint
sort_name (GtkTreeModel *model,
	   GtkTreeIter *a,
	   GtkTreeIter *b,
	   gpointer data)
{
	gint cmp;
	gchar *str1, *str2;

	gtk_tree_model_get(model, a, THUMBWIN_TEXT, &str1, -1);
	gtk_tree_model_get(model, b, THUMBWIN_TEXT, &str2, -1);

	cmp = strcmp(str1, str2);

	g_free (str1);
	g_free (str2);

	return cmp;
}

static gint
sort_mtime (GtkTreeModel *model,
	    GtkTreeIter *a,
	    GtkTreeIter *b,
	    gpointer data)
{
	struct stat s1, s2;
	gchar *p1, *p2;

	gtk_tree_model_get(model, a, THUMBWIN_PATH, &p1, -1);
	stat (p1, &s1);

	gtk_tree_model_get(model, b, THUMBWIN_PATH, &p2, -1);
	stat (p2, &s2);

	g_free (p1);
	g_free (p2);

	if (s1.st_mtime > s2.st_mtime)
		return -1;
	else if (s1.st_mtime < s2.st_mtime)
		return 1;

	return sort_name (model, a, b, data);
}

static gint
sort_size (GtkTreeModel *model,
	   GtkTreeIter  *a,
	   GtkTreeIter  *b,
	   gpointer      data)
{
	struct stat s1, s2;
	gchar *p1, *p2;

	gtk_tree_model_get(model, a, THUMBWIN_PATH, &p1, -1);
	stat (p1, &s1);

	gtk_tree_model_get(model, b, THUMBWIN_PATH, &p2, -1);
	stat (p2, &s2);

	g_free (p1);
	g_free (p2);

	if (s1.st_size > s2.st_size)
		return -1;
	else if (s1.st_size < s2.st_size)
		return 1;

	return sort_name (model, a, b, data);
}

static gint
sort_area (GtkTreeModel *model,
	   GtkTreeIter  *a,
	   GtkTreeIter  *b,
	   gpointer      data)
{
	gint a1, a2, w, h;
	gchar *path;

	gtk_tree_model_get (model, a, THUMBWIN_PATH, &path, -1);
	gdk_pixbuf_get_file_info (path, &w, &h);
	a1 = w * h;

	g_free (path);

	gtk_tree_model_get (model, b, THUMBWIN_PATH, &path, -1);
	gdk_pixbuf_get_file_info (path, &w, &h);
	a2 = w * h;

	g_free (path);

	if (a1 > a2)
		return -1;
	else if (a2 > a1)
		return 1;

	return sort_name (model, a, b, data);
}

static void
fb_thumbwin_init (FbThumbWin *tw)
{
	GtkTreeSortable *sortable;

	tw->ui_manager = fb_thumbwin_ui_manager_get (tw);

	gtk_window_set_title(GTK_WINDOW(tw), "Thumbnail - Futaba");
	gtk_window_add_accel_group(GTK_WINDOW(tw), gtk_ui_manager_get_accel_group(tw->ui_manager));

	tw->popup1 = gtk_ui_manager_get_widget (tw->ui_manager, "/OverThumbPopup");
	tw->popup2 = gtk_ui_manager_get_widget (tw->ui_manager, "/Popup");

	tw->scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(tw->scroll),
					     GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(tw->scroll),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER(tw), tw->scroll);
	gtk_widget_show (tw->scroll);

	tw->store = gtk_list_store_new (THUMBWIN_COLUMNS,
					      GDK_TYPE_PIXBUF,
					      G_TYPE_STRING,
					      G_TYPE_STRING,
					      -1);

	sortable = GTK_TREE_SORTABLE (tw->store);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_NAME,
					sort_name, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_MTIME,
					sort_mtime, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_SIZE,
					sort_size, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_AREA,
					sort_area, NULL, NULL);
	gtk_tree_sortable_set_sort_column_id(sortable, SORT_BY_NAME, GTK_SORT_ASCENDING);

	tw->icon_view = gtk_icon_view_new_with_model (GTK_TREE_MODEL (tw->store));
	gtk_icon_view_set_text_column (GTK_ICON_VIEW (tw->icon_view), THUMBWIN_TEXT);
	gtk_icon_view_set_pixbuf_column (GTK_ICON_VIEW (tw->icon_view), THUMBWIN_PIXBUF);
/* 	gtk_icon_view_set_spacing (GTK_ICON_VIEW (tw->icon_view), 2); */
	gtk_icon_view_set_row_spacing (GTK_ICON_VIEW (tw->icon_view), 10);
	gtk_icon_view_set_column_spacing (GTK_ICON_VIEW (tw->icon_view), 10);
	gtk_icon_view_set_margin (GTK_ICON_VIEW (tw->icon_view), THUMBWIN_MARGIN);
	gtk_drag_dest_set(GTK_WIDGET(tw->icon_view), 
			  GTK_DEST_DEFAULT_HIGHLIGHT |
			  GTK_DEST_DEFAULT_MOTION | 
			  GTK_DEST_DEFAULT_DROP,
                          uri_drag_types,n_uri_drag_types,
			  GDK_ACTION_MOVE | GDK_ACTION_COPY);
	g_signal_connect (G_OBJECT (tw->icon_view), "button-press-event",
			  G_CALLBACK (cb_button_press), tw);
	g_signal_connect (G_OBJECT (tw->icon_view), "drag-begin",
			  G_CALLBACK (cb_drag_begin), tw);
	g_signal_connect (G_OBJECT (tw->icon_view), "drag-data-get",
			  G_CALLBACK (cb_drag_data_get), tw);
	g_signal_connect (G_OBJECT (tw->icon_view), "drag-data-received",
			  G_CALLBACK (cb_drag_data_received), tw);
	g_signal_connect (G_OBJECT (tw->icon_view), "motion-notify-event",
			  G_CALLBACK (cb_motion_notify), tw);
	gtk_container_add (GTK_CONTAINER (tw->scroll), tw->icon_view);
	gtk_widget_show (tw->icon_view);

	tw->prev_path = NULL;
	tw->workdir = NULL;
	tw->drag_data = NULL;

	if (! thumbnail_frame)
		thumbnail_frame = gdk_pixbuf_new_from_file (ICONDIR"/thumb-frame.png", NULL);

	restore_status (tw);
}

static void
restore_status (FbThumbWin *tw)
{
	gboolean flag;
	gint val_i, val_j;
	gchar *str;

	if (focus_thumbwin) {
		val_i = focus_thumbwin->allocation.width;
		val_j = focus_thumbwin->allocation.height;
	}
	else {
		fb_conf_get_value ("thumbwin width", &val_i, FB_CONF_TYPE_INT, "550");
		fb_conf_get_value ("thumbwin height", &val_j, FB_CONF_TYPE_INT, "500");
	}
	gtk_window_set_default_size(GTK_WINDOW(tw), val_i, val_j);

	fb_conf_get_value ("thumbwin background", &str, FB_CONF_TYPE_STRING, ICONDIR"/cork.png");
	fb_thumbwin_set_background (FB_THUMBWIN (tw), str);
	g_free (str);

	fb_conf_get_value ("thumbnail width", &tw->thumb_w, FB_CONF_TYPE_INT, "72");
	fb_conf_get_value ("thumbnail height", &tw->thumb_h, FB_CONF_TYPE_INT, "72");
	fb_conf_get_value ("thumbnail spacing", &tw->thumb_spacing, FB_CONF_TYPE_INT, "20");

	fb_conf_get_value ("show thumb name", &flag, FB_CONF_TYPE_BOOLEAN, "TRUE");
	if (! flag) gtk_icon_view_set_text_column (GTK_ICON_VIEW (tw->icon_view), -1);
}

static void
store_status (FbThumbWin *tw)
{
	fb_conf_set_value ("thumbwin width", & GTK_WIDGET (tw)->allocation.width, FB_CONF_TYPE_INT);
	fb_conf_set_value ("thumbwin height", & GTK_WIDGET (tw)->allocation.height, FB_CONF_TYPE_INT);
}

static void
fb_thumbwin_destroy (GtkObject *object)
{

	store_status (FB_THUMBWIN (object));

	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
fb_thumbwin_finalize (GObject *object)
{
	gint i;
	FbThumbWin *tw = FB_THUMBWIN (object);

	i = MAX(0, g_list_index(thumbwin_list, tw) - 1);
	thumbwin_list = g_list_remove(thumbwin_list, tw);
	focus_thumbwin = g_list_nth_data(thumbwin_list, i);

	if (tw->workdir) g_free (tw->workdir);
	if (tw->drag_data) g_free (tw->drag_data);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
fb_thumbwin_focus_in_event (GtkWidget *widget,
			    GdkEventFocus *event)
{
	focus_thumbwin = widget;

	return GTK_WIDGET_CLASS (parent_class)->focus_in_event (widget, event);
}

static void
fb_thumbwin_size_allocate (GtkWidget *widget,
			   GtkAllocation *allocation)
{
	gint view_w, col_w, col_n;
	FbThumbWin *tw = FB_THUMBWIN (widget);

	view_w = tw->icon_view->allocation.width - THUMBWIN_MARGIN;
	col_w = tw->thumb_w + FRAME_LEFT_OFFSET + FRAME_RIGHT_OFFSET + (tw->thumb_spacing * 2);
	col_n = view_w / col_w;
	if (col_n != gtk_icon_view_get_columns (GTK_ICON_VIEW (tw->icon_view))) {
		gtk_icon_view_set_columns (GTK_ICON_VIEW (tw->icon_view), col_n);
		gtk_icon_view_set_item_width (GTK_ICON_VIEW (tw->icon_view), col_w);
	}

	GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
}

static gboolean
cb_button_press (GtkWidget *icon_view,
		 GdkEventButton *event,
		 gpointer data)
{
	FbThumbWin *tw = FB_THUMBWIN (data);

	switch (event->button) {
	case 1: {
	}
		break;
	case 3: {
		if (tw->prev_path) {
			gtk_menu_popup(GTK_MENU(tw->popup1), NULL, NULL, NULL, NULL,
				       event->button, event->time);
		}
		else {
			gboolean flag;
			GtkWidget *widget;

			widget = gtk_ui_manager_get_widget (tw->ui_manager, "/Popup/Paste");
			flag = (paste_path != NULL);
			gtk_widget_set_sensitive (widget, flag);

			gtk_menu_popup(GTK_MENU(tw->popup2), NULL, NULL, NULL, NULL,
				       event->button, event->time);
		}
	}
		break;
	default:
		break;
	}

	return FALSE;
}

static void
cb_drag_begin (GtkWidget *icon_view,
	       GdkDragContext *drag_context,
	       gpointer data)
{
	GtkTreeIter iter;
	FbThumbWin *tw = data;

/* 	g_print ("icon_view:drag begin\n"); */

	if (tw->prev_path && gtk_tree_model_get_iter (GTK_TREE_MODEL (tw->store), &iter, tw->prev_path)) {
		gchar *path;
		GdkPixbuf *pixbuf;

		gtk_tree_model_get (GTK_TREE_MODEL (tw->store), &iter,
				    THUMBWIN_PATH, &path,
				    THUMBWIN_PIXBUF, &pixbuf,
				    -1);

		if (tw->drag_data) g_free (tw->drag_data);
		tw->drag_data = g_strdup_printf ("file://%s\r\n", path);
		g_free (path);

		gtk_drag_set_icon_pixbuf (drag_context, pixbuf, -5, -5);
		g_object_unref (pixbuf);
	}
}

static void
cb_drag_data_get (GtkWidget *icon_view,
		  GdkDragContext *drag_context,
		  GtkSelectionData *selection_data,
		  guint info,
		  guint time,
		  gpointer data)
{
	gchar *uri;
	FbThumbWin *tw = data;
/* 	g_print ("icon_view:drag data get\n"); */

	uri = tw->drag_data;

	switch (info) {
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, uri, strlen(uri));
		break;
	default:
		break;
	}

}

static void
cb_drag_data_received (GtkWidget *icon_view,
		       GdkDragContext *drag_context,
		       gint x, gint y,
		       GtkSelectionData *selection_data,
		       guint info,
		       guint time,
		       gpointer data)
{
	gboolean success, del;
	gchar *uri;
	FbThumbWin *tw = data;

	success = del = FALSE;
/* 	g_print ("icon_view:drag data received\n"); */
	uri = g_strndup (selection_data->data, selection_data->length);
/* 	g_print ("%s", uri); */

	if (g_str_has_prefix (uri, "file:/")) {
		gint s_len, r_len;
		gchar *src, *dest, *base, *ptr;

		s_len = strlen ("file:");
		r_len = 0;
		ptr = uri + strlen (uri) - 1;
		while (*ptr == '\r' || *ptr == '\n' || *ptr == '\0') {
			r_len++;
			ptr--;
		}

		src = g_strndup (uri + s_len, strlen (uri) - s_len - r_len);
		base = g_path_get_basename (src);
		dest = g_strconcat (tw->workdir, "/", base, NULL);
		g_free (base);
/* 		g_print ("src  %s\n", src); */
/* 		g_print ("dest %s\n", dest); */

		if (! g_file_test (src, G_FILE_TEST_EXISTS) ||
		    g_file_test (dest, G_FILE_TEST_EXISTS))
			goto fin;

/* 		g_print ("%s copy %s\n", src, dest); */

		if (path_is_dir (src)) {
			success = cpdir_recursive (src, dest);
		}
		else {
			success = file_copy (src, dest);
		}

		if (drag_context->suggested_action == GDK_ACTION_MOVE && success) {
			if (path_is_dir (src)) {
				del = rmdir_recursive (src);
			}
			else {
				del = ! unlink (src);
			}
		}

		if (success && file_is_image (dest)) fb_thumbwin_add_thumbnail (tw, dest);

	fin:
		g_free (dest);
		g_free (src);
	}

	gtk_drag_finish (drag_context, success, del, time);
	g_free (uri);
}

static gboolean
cb_motion_notify(GtkWidget *icon_view,
		 GdkEventMotion *event,
		 gpointer data)
{
	GtkTreePath *tpath;
	FbThumbWin *tw = data;

	tpath = gtk_icon_view_get_path_at_pos (GTK_ICON_VIEW (icon_view), event->x, event->y);
	if (tpath && ! tw->prev_path) {
		gtk_drag_source_set(icon_view,
				    GDK_BUTTON1_MASK,
				    uri_drag_types, n_uri_drag_types,
				    GDK_ACTION_ASK  | GDK_ACTION_COPY
				    | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	}
	else if (! tpath && tw->prev_path && event->type != GDK_DRAG_MOTION) {
		gtk_drag_source_unset (icon_view);
	}

	if (tw->prev_path) gtk_tree_path_free (tw->prev_path);
	tw->prev_path = tpath;

	return FALSE;
}

GtkWidget *
fb_thumbwin_new (void)
{
	GObject *object = g_object_new (FB_TYPE_THUMBWIN, NULL);
	GtkWidget *tw = GTK_WIDGET (object);

	focus_thumbwin = tw;
	thumbwin_list = g_list_append (thumbwin_list, tw);

	return tw;
}

void
fb_thumbwin_set_background (FbThumbWin *tw,
			    const gchar *path)
{
	GdkPixbuf *pixbuf;

	g_return_if_fail (FB_IS_THUMBWIN (tw));
	g_return_if_fail (path || *path);

	pixbuf = gdk_pixbuf_new_from_file (path, NULL);
	if (pixbuf) {
		gtk_icon_view_set_background (GTK_ICON_VIEW (tw->icon_view), pixbuf);
		g_object_unref (pixbuf);
	}
}

void
fb_thumbwin_start (GList *path_list)
{
	GList *p;
	GtkWidget *tw;

	g_return_if_fail (path_list != NULL);

	tw = fb_thumbwin_new ();
	gtk_widget_show (tw);

	FB_THUMBWIN (tw)->workdir = g_path_get_dirname (path_list->data);

	for (p = path_list; p; p = p->next) {
		while (gtk_events_pending ()) gtk_main_iteration ();

		fb_thumbwin_add_thumbnail (FB_THUMBWIN (tw), p->data);
	}
}

void
fb_thumbwin_add_thumbnail (FbThumbWin *tw,
			   const gchar *path)
{
	gchar *text, *utf8;
	GdkPixbuf *pixbuf, *tmp;
	GtkTreeIter iter;

	g_return_if_fail (FB_IS_THUMBWIN (tw));
	g_return_if_fail (path || *path);

	pixbuf = gdk_pixbuf_new_from_file_at_size (path, tw->thumb_w, tw->thumb_h, NULL);
	if (! pixbuf) return;

	if (thumbnail_frame) {
		tmp = eel_embed_image_in_frame (pixbuf, thumbnail_frame,
						FRAME_LEFT_OFFSET, FRAME_TOP_OFFSET,
						FRAME_RIGHT_OFFSET, FRAME_BOTTOM_OFFSET);
		g_object_unref (pixbuf);
		pixbuf = tmp;
	}

	text = g_path_get_basename (path);
	utf8 = g_locale_to_utf8 (text, -1, NULL, NULL, NULL);
	gtk_list_store_append (tw->store, &iter);
	gtk_list_store_set (tw->store, &iter,
			    THUMBWIN_PIXBUF, pixbuf,
			    THUMBWIN_TEXT, text,
			    THUMBWIN_PATH, path,
			    -1);
	g_object_unref (pixbuf);
	g_free (text);
	g_free (utf8);
}
