/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "futaba.h"
#include "fb-viewer-action.h"
#include "fb-viewer.h"
#include "fb-image.h"

#define ZOOMIN 0.77
#define ZOOMOUT 1.3

static void
cb_quit(GtkAction *action,
	gpointer data)
{
	FbViewer *viewer = data;

	g_signal_emit_by_name(G_OBJECT(viewer), "destroy", 0);
}

static void
cb_hide(GtkAction *action,
	gpointer data)
{
	FbViewer *viewer = data;

	gtk_window_iconify(GTK_WINDOW(viewer));
}

static void
cb_first_image(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	viewer->position = 0;
	fb_viewer_next_images (viewer, 1);
}

static void
cb_next_image1(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_next_images(viewer, 1);
}

static void
cb_next_image2(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_next_images(viewer, 10);
}

static void
cb_next_image3(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_next_images(viewer, 50);
}

static void
cb_prev_image1(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_prev_images(viewer, 1);
}

static void
cb_prev_image2(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_prev_images(viewer, 10);
}

static void
cb_prev_image3(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_prev_images(viewer, 50);
}

static void
cb_last_image(GtkAction *action,
	     gpointer data)
{
	FbViewer *viewer = data;

	viewer->position = g_list_length (viewer->path_list);
	fb_viewer_prev_images(viewer, 1);
}

static void
cb_zoom_in(GtkAction *action,
	   gpointer data)
{
        FbViewer *viewer = data;

	fb_image_zoom (FB_IMAGE (viewer->image), ZOOMIN);
}

static void
cb_zoom_out(GtkAction *action,
	    gpointer data)
{
        FbViewer *viewer = data;

	fb_image_zoom (FB_IMAGE (viewer->image), ZOOMOUT);
}

static void
cb_zoom_100(GtkAction *action,
	    gpointer data)
{
        FbViewer *viewer = data;

	fb_image_zoom_at_value (FB_IMAGE (viewer->image), 1.0);
}

static void
cb_fit_image(GtkAction *action,
	     gpointer data)
{
        FbViewer *viewer = data;

	fb_image_fit_window (FB_IMAGE (viewer->image));
}

static void
cb_right_rotation(GtkAction *action,
		  gpointer data)
{
        FbViewer *viewer = data;

	fb_image_rotate (FB_IMAGE (viewer->image), FALSE);
}

static void
cb_left_rotation(GtkAction *action,
		 gpointer data)
{
        FbViewer *viewer = data;

	fb_image_rotate (FB_IMAGE (viewer->image), TRUE);
}

static void
cb_mirror_rotation(GtkAction *action,
		   gpointer data)
{
        FbViewer *viewer = data;

	fb_image_mirror (FB_IMAGE (viewer->image), TRUE, FALSE);
}

static void
cb_image_switching(GtkAction *action,
		   gpointer data)
{
        FbViewer *viewer = data;

	if (fb_image_get_n_images (FB_IMAGE (viewer->image)) == 1) {
		fb_viewer_segmentalize (viewer, GTK_ORIENTATION_HORIZONTAL);
	}
	else {
		fb_image_remove_image (FB_IMAGE (viewer->image));
		viewer->position--;
	}
}

static void
cb_max_window(GtkAction *action,
	      gpointer data)
{
        FbViewer *viewer = data;

	if (viewer->size > VIEWER_FULL) return;

	if (viewer->size > VIEWER_MAX) {
		gtk_window_unmaximize(GTK_WINDOW(viewer));
		viewer->size ^= VIEWER_MAX;
	}
	else {
		gtk_window_maximize(GTK_WINDOW(viewer));
		gtk_window_move(GTK_WINDOW(viewer), 0, 0);
		viewer->size ^= VIEWER_MAX;
	}
}

static void
cb_fullscreen(GtkAction *action,
	      gpointer data)
{
	FbViewer *viewer = data;

	if (viewer->size > VIEWER_FULL) {
		gtk_window_unfullscreen(GTK_WINDOW(viewer));
		viewer->size ^= VIEWER_FULL;
	}
	else {
		gtk_window_fullscreen(GTK_WINDOW(viewer));
		viewer->size ^= VIEWER_FULL;
	}
}

static void
cb_vsegment (GtkAction *action,
	     gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_segmentalize (viewer, GTK_ORIENTATION_VERTICAL);
}

static void
cb_hsegment (GtkAction *action,
	     gpointer data)
{
	FbViewer *viewer = data;

	fb_viewer_segmentalize (viewer, GTK_ORIENTATION_HORIZONTAL);
}

static GtkActionEntry actions[] = {
	{"Quit", "gtk-close", N_("viewer close"), "<release>q", NULL, G_CALLBACK(cb_quit)},
	{"Hide", NULL, "dummy", "<release>h", NULL, G_CALLBACK(cb_hide)},

	{"FirstPage", NULL, "dummy", "<release>Home", NULL, G_CALLBACK(cb_first_image)},
	{"NextPage1", NULL, "dummy", "<release>n", NULL, G_CALLBACK(cb_next_image1)},
	{"NextPage2", NULL, "dummy", "<release>Down", NULL, G_CALLBACK(cb_next_image2)},
	{"NextPage3", NULL, "dummy", "<release>v", NULL, G_CALLBACK(cb_next_image3)},
	{"PrevPage1", NULL, "dummy", "<release>p", NULL, G_CALLBACK(cb_prev_image1)},
	{"PrevPage2", NULL, "dummy", "<release>Up", NULL, G_CALLBACK(cb_prev_image2)},
	{"PrevPage3", NULL, "dummy", "<release>c", NULL, G_CALLBACK(cb_prev_image3)},
	{"LastPage", NULL, "dummy", "<release>e", NULL, G_CALLBACK(cb_last_image)},

	{"ZoomIn", "gtk-zoom-in", N_("zoom in"), "<release>J", NULL, G_CALLBACK(cb_zoom_in)},
	{"ZoomOut", "gtk-zoom-out", N_("zoom out"), "<release>L", NULL, G_CALLBACK(cb_zoom_out)},
	{"Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "< release>O", NULL, G_CALLBACK(cb_zoom_100)},
	{"ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<release>K", NULL, G_CALLBACK(cb_fit_image)},
	{"RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_right_rotation)},
	{"LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_left_rotation)},
	{"MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_mirror_rotation)},
	{"VSegment", NULL, N_("segmentalize vertical"), NULL, NULL, G_CALLBACK (cb_vsegment)},
	{"HSegment", NULL, N_("segmentalize horizontal"), NULL, NULL, G_CALLBACK (cb_hsegment)},
};
static guint actions_l = G_N_ELEMENTS(actions);

static GtkToggleActionEntry toggle_actions[] = {
  {"TwoWindow", NULL, N_("two pages display"), "<release>space", NULL, G_CALLBACK(cb_image_switching), FALSE},
  {"MaxWindow", NULL, N_("Max Window"), "<release>M", NULL, G_CALLBACK(cb_max_window), FALSE},
  {"FullScreen", NULL, N_("full screen"), "<release>F", NULL, G_CALLBACK(cb_fullscreen), FALSE},
};
static guint toggle_actions_l = G_N_ELEMENTS(toggle_actions);

static const gchar *viewer_ui = {
	"<ui>"
	"  <popup name=\"Popup\">"
	"    <menuitem action=\"ZoomIn\"/>"
	"    <menuitem action=\"ZoomOut\"/>"
	"    <menuitem action=\"Zoom100\"/>"
	"    <menuitem action=\"ZoomFit\"/>"
	"    <separator/>"
	"    <menuitem action=\"RightRotation\"/>"
	"    <menuitem action=\"LeftRotation\"/>"
	"    <menuitem action=\"MirrorRotation\"/>"
	"    <separator/>"
	"    <menuitem action=\"HSegment\"/>"
	"    <menuitem action=\"VSegment\"/>"
	"    <menuitem action=\"FullScreen\"/>"
	"    <separator/>"
	"    <menuitem action=\"Quit\"/>"
	"  </popup>"
	"  <popup name=\"Unvisible\">"
	"    <menuitem action=\"Hide\"/>"
	"    <menuitem action=\"TwoWindow\"/>"
	"    <menuitem action=\"MaxWindow\"/>"
	"    <menuitem action=\"FirstPage\"/>"
	"    <menuitem action=\"NextPage1\"/>"
	"    <menuitem action=\"NextPage2\"/>"
	"    <menuitem action=\"NextPage3\"/>"
	"    <menuitem action=\"PrevPage1\"/>"
	"    <menuitem action=\"PrevPage2\"/>"
	"    <menuitem action=\"PrevPage3\"/>"
	"    <menuitem action=\"LastPage\"/>"
	"  </popup>"
	"</ui>"
};

GtkUIManager *fb_viewer_ui = NULL;

GtkUIManager *
fb_viewer_ui_manager_get (gpointer data)
{
	GError *error = NULL;
	GtkActionGroup *action_group;

/* 	if (fb_viewer_ui) return fb_viewer_ui; */

	action_group = gtk_action_group_new("Futaba Viewer");

	gtk_action_group_set_translation_domain(action_group, NULL);

	gtk_action_group_add_actions(action_group,
				     actions, actions_l,
				     data);

	gtk_action_group_add_toggle_actions(action_group,
					    toggle_actions, toggle_actions_l,
					    data);

	fb_viewer_ui = gtk_ui_manager_new();

	gtk_ui_manager_insert_action_group(fb_viewer_ui, action_group, 0);

	if (!gtk_ui_manager_add_ui_from_string(fb_viewer_ui, viewer_ui, -1, &error)) {
		g_print("Failed viewer popup init: %s\n", error->message);
		g_error_free(error);

		return NULL;
	}

	return fb_viewer_ui;
}
