/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "futaba.h"
#include "fb-window.h"
#include "fb-conf.h"
#include "fb-actions.h"
#include "fb-dirview.h"
#include "fb-preview.h"
#include "fb-statusbar.h"
#include "fb-locationbar.h"
#include "fb-prefs.h"
#include "fb-dialog.h"
#include "file-utils.h"
#include "fb-viewer.h"
#include "fb-thumbwin.h"

static void
cb_show_file_menu(GtkAction *action,
		  gpointer data)
{
	gboolean flag;
	FbWindow *fb = data;

	flag = (FB_DIRVIEW (fb->dirview)->image_num > 0);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/FileMenu/OpenViewer", flag);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/FileMenu/OpenThumb", flag);

	flag = (focus_viewer != NULL);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/FileMenu/CloseViewer", flag);

	flag = (focus_thumbwin != NULL);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/FileMenu/CloseThumb", flag);
}

static void
cb_show_edit_menu(GtkAction *action,
		  gpointer data)
{
	gboolean flag;
	gchar *path;
	FbDirview *dv;
	FbWindow *fb = data;

	dv = FB_DIRVIEW (fb->dirview);

	flag = (dv->writable && (paste_list != NULL));
	ui_set_sensitive(fb->ui_manager, "/MenuBar/EditMenu/Paste", flag);

	flag = fb_dirview_row_get (dv,
				   COLUMN_DISPLAY_PATH, &path,
				   -1);
	if (flag && (path[0] == '.') && ((path[1] == '\0') || (path[1] == '.')))
		flag = FALSE;

	ui_set_sensitive(fb->ui_manager, "/MenuBar/EditMenu/Copy", flag);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/EditMenu/Cut", flag);
	ui_set_sensitive(fb->ui_manager, "/MenuBar/EditMenu/Delete", flag);
}

static void
cb_show_view_menu(GtkAction *action,
		  gpointer data)
{
	FbWindow *fb = data;


}

static void
cb_show_move_menu(GtkAction *action,
		  gpointer data)
{
	gboolean flag;
	const gchar *workdir;
	GList *history;
	FbDirview *dv;
	FbWindow *fb = data;

	dv = FB_DIRVIEW (fb->dirview);

	workdir = fb_dirview_get_workdir (dv);
	flag = g_strcasecmp(workdir, "/");
	ui_set_sensitive (fb->ui_manager, "/MenuBar/MoveMenu/UpDir", flag);

	history = gtk_container_get_children (GTK_CONTAINER (dv->prev_history));
	flag = (history != NULL);
	ui_set_sensitive (fb->ui_manager, "/MenuBar/MoveMenu/BackHistory", flag);

	history = gtk_container_get_children (GTK_CONTAINER (dv->next_history));
	flag = (history != NULL);
	ui_set_sensitive (fb->ui_manager, "/MenuBar/MoveMenu/NextHistory", flag);

}

static void
cb_open(GtkAction *action,
	gpointer data)
{
	gint type;
	gchar *path;
	FbWindow *fb = data;

	fb_dirview_row_get (FB_DIRVIEW (fb->dirview),
			    COLUMN_TYPE, &type,
			    COLUMN_PATH, &path,
			    -1);
	g_signal_emit_by_name(G_OBJECT(fb->dirview), "path-activated", path, type);
	g_free (path);
}

static void
cb_open_viewer(GtkAction *action,
	       gpointer data)
{
	GList *path_list;
	FbDirview *dv;
	FbWindow *fb = data;
	
	dv = FB_DIRVIEW (fb->dirview);

	if (! dv->image_num) {
		fb_statusbar_push("%s", _("No image !!"));
		return;
	}

	path_list = fb_dirview_get_path_list(dv, IMAGE);
	fb_viewer_start(path_list, NULL);
	path_list_free(path_list);
}

static void
cb_open_thumb(GtkAction *action,
	      gpointer data)
{
	GList *path_list;
	FbDirview *dv;
	FbWindow *fb = data;

	dv = FB_DIRVIEW (fb->dirview);

	if (! dv->image_num) {
		fb_statusbar_push ("%s", _("No image !!"));
		return;
	}
	
	path_list = fb_dirview_get_path_list (dv, IMAGE);
	fb_thumbwin_start (path_list);
	path_list_free (path_list);
}

static void
cb_close_viewer(GtkAction *action,
		gpointer data)
{
	FbWindow *fb = data;

	if (! focus_viewer) return;

	gtk_widget_destroy (focus_viewer);
}

static void
cb_close_thumb(GtkAction *action,
	       gpointer data)
{
	FbWindow *fb = data;

	if (!focus_thumbwin) return;

	gtk_widget_destroy (focus_thumbwin);
}

static void
cb_quit(GtkAction *action,
	gpointer data)
{
	FbWindow *fb = data;

	fb_window_store_status (fb);
	g_signal_emit_by_name (G_OBJECT (fb), "destroy", 0);
}

static void
cb_clear_cache (GtkAction *action,
		gpointer data)
{
	gchar *ext_dir;
	FbWindow *fb = data;

	ext_dir = g_strconcat (g_get_home_dir (),
			       "/.futaba/tmp", NULL);
	rmdir_recursive (ext_dir);
	g_free (ext_dir);

}

static void
cb_update_archive (GtkAction *action,
		   gpointer data)
{
	gchar *ext_dir, *set_path;
	const gchar *path;
	FbWindow *fb = data;

	path = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	ext_dir = g_strconcat (g_get_home_dir (),
			       "/.futaba/tmp", NULL);
	if (! g_str_has_prefix (path, ext_dir)) {
		g_free (ext_dir);
		return;
	}

	if (file_is_archive (path)) {
		rmdir_recursive (path);
		/* FIX ME (Don't manipulate dirview) */ 
	}

	set_path = g_strdup (path + strlen (ext_dir));
	g_signal_emit_by_name (G_OBJECT (fb->dirview), "path-activated",
			       set_path, ARCHIVE);

	g_free (ext_dir);
	g_free (set_path);
}

static void
cb_copy(GtkAction *action,
	gpointer data)
{
	gchar *path;
	FbWindow *fb = data;

	if (paste_list) {
		g_list_foreach (paste_list, (GFunc) g_free, NULL);
		g_list_free (paste_list);
		paste_list = NULL;
	}

	fb_dirview_row_get (FB_DIRVIEW (fb->dirview),
			    COLUMN_PATH, &path,
			    -1);
	paste_list = g_list_append (paste_list, path);
	is_copy = TRUE;

	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Paste", path);
	ui_set_message(fb->ui_manager, "/DirViewPopup/Paste", path);
}

static void
cb_cut(GtkAction *action,
       gpointer data)
{
	gchar *path;
	FbWindow *fb = data;

	if (paste_list) {
		g_list_foreach (paste_list, (GFunc) g_free, NULL);
		g_list_free (paste_list);
		paste_list = NULL;
	}

	fb_dirview_row_get (FB_DIRVIEW (fb->dirview),
			    COLUMN_PATH, &path,
			    -1);
	paste_list = g_list_append (paste_list, path);
	is_copy = FALSE;

	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Paste", path);
	ui_set_message(fb->ui_manager, "/DirViewPopup/Paste", path);
}

static void
cb_paste(GtkAction *action,
	 gpointer data)
{
	gboolean result;
	gint step, len;
	gchar *tmp, *src, *dest;
	const gchar *workdir;
	GList *pathl;
	FbWindow *fb = data;

	if (! paste_list) return;

	fb_statusbar_pop ();

	workdir = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));

	/* 	fb_statusbar_progress_start (FB_STATUSBAR (fb->status_bar)); */
	step = 1;
	len = g_list_length (paste_list);
	for (pathl = paste_list; pathl; pathl = pathl->next) {
		while (gtk_events_pending ()) gtk_main_iteration ();

		src = pathl->data;
		result  = FALSE;

		tmp = g_path_get_basename (src);
		dest = g_strconcat (workdir, "/", tmp, NULL);
		g_free (tmp);

		result = FALSE;
		if (path_is_dir (src)) {
			result = cpdir_recursive (src, dest);
			if (! is_copy && result)
				result = rmdir_recursive (src);
		}
		else if (path_is_file (src)) {
			result = file_copy (src, dest);
			if (! is_copy && result)
				result = ! unlink (src);
		}
		fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), (gdouble) step / (gdouble) len);
		/* 	fb_statusbar_progress_end (FB_STATUSBAR (fb->status_bar)); */
		step++;

		if (result) {
			fb_dirview_add_path (FB_DIRVIEW (fb->dirview), dest);
		}
		else {
			GtkWidget *dlg;

			dlg = gtk_message_dialog_new (GTK_WINDOW (fb), GTK_DIALOG_MODAL,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						      _("Failed paste"));
			g_signal_connect_swapped (G_OBJECT (dlg), "response",
						  G_CALLBACK (gtk_widget_destroy), dlg);
			gtk_widget_show (dlg);
		}
		g_free (dest);
	}

	if (! is_copy) {
		g_list_foreach (paste_list, (GFunc) g_free, NULL);
		g_list_free (paste_list);
		paste_list = NULL;
	}
	fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), 0.0);
}

static void
cb_delete(GtkAction *action,
	  gpointer data)
{
	gint res;
	gchar *path;
	GtkWidget *dlg;
	FbWindow *fb = data;

	fb_statusbar_pop ();

	fb_dirview_row_get(FB_DIRVIEW (fb->dirview), 
			   COLUMN_PATH, &path,
			   -1);

	dlg = gtk_message_dialog_new (GTK_WINDOW (fb), GTK_DIALOG_MODAL,
				      GTK_MESSAGE_QUESTION, GTK_BUTTONS_OK_CANCEL,
				      _("Delete \"%s\" ?"), path);
	res = gtk_dialog_run (GTK_DIALOG (dlg));
	gtk_widget_destroy (dlg);

	if (res == GTK_RESPONSE_CANCEL) {
		g_free (path);
		return;
	}

/* 	fb_statusbar_progress_start (FB_STATUSBAR (fb->status_bar)); */
	if (path_is_dir (path)) {
		gboolean is_delete;
		gint len, step;
		GList *node, *path_list;

		path_list = get_path_list (path, TRUE);
		step = 1;
		len = g_list_length (path_list);
		res = GTK_RESPONSE_NONE;
		for (node = path_list; node; node = node->next) {
			while ( gtk_events_pending ()) gtk_main_iteration ();

			if (path_is_dir (node->data))
				is_delete = rmdir (node->data);
			else
				is_delete = unlink (node->data);

			if (is_delete && (res != GTK_RESPONSE_YES)) {
				dlg = gtk_message_dialog_new (GTK_WINDOW (fb), GTK_DIALOG_MODAL,
							      GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
							      _("Couldn't delete \"%s\"\nDo you continue?"), node->data);
				res = gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);

				if (res == GTK_RESPONSE_NO) {
					path_list_free (path_list);
					return;
				}
			}
			fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), (gdouble) step/ (gdouble) len);
			step++;
		}
		path_list_free (path_list);
	}
	else if (path_is_file (path)) {
		if (unlink (path)) {
			dlg = gtk_message_dialog_new (GTK_WINDOW (fb), GTK_DIALOG_MODAL,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						      _("Couldn't delete"));
			g_signal_connect_swapped (G_OBJECT (dlg), "response",
						  G_CALLBACK (gtk_widget_destroy), dlg);
			gtk_widget_show (dlg);
		}
	}
/* 	fb_statusbar_progress_end (FB_STATUSBAR (fb->status_bar)); */
	fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), 0.0);

	fb_dirview_row_remove (FB_DIRVIEW (fb->dirview));

	g_free(path);
}

static void
cb_make_dir(GtkAction *action,
	    gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_dirview_add_new_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_rename(GtkAction *action,
	  gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_dirview_row_edit(FB_DIRVIEW (fb->dirview));
}

static void
cb_change_bulk(GtkAction *action,
	       gpointer data)
{
	GtkWidget *dialog;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

/* 	dialog = fb_bulk_change_dialog_new(fb); */
/* 	gtk_widget_show(dialog); */
}

static void
cb_property(GtkAction *action,
	    gpointer data)
{
	GtkWidget *dialog;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

/* 	dialog = fb_property_dialog_new(fb); */
/* 	gtk_widget_show(dialog); */
}

static void
cb_preferences(GtkAction *action,
	       gpointer data)
{
	GtkWidget *win;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	win = fb_prefs_win_new();
	gtk_widget_show(win);
}

static void
cb_add_bookmark(GtkAction *action,
		gpointer data)
{
	const gchar *path;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	path = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	fb_window_add_bookmark(fb, path);
}

static void
cb_edit_bookmark(GtkAction *action,
		 gpointer data)
{
	GtkWidget *editer;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	editer = fb_bookmark_editer_new(fb);
	gtk_widget_show_all(editer);
}

static void
cb_about(GtkAction *action,
	 gpointer data)
{
	GtkWidget *dialog;
	FbWindow *fb = data;

	dialog = fb_about_dialog_new(fb);
	gtk_widget_show_all(dialog);
}

static void
cb_back_history(GtkAction *action,
		gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	fb_dirview_prev_history (FB_DIRVIEW (fb->dirview));
}

static void
cb_next_history(GtkAction *action,
		gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	fb_dirview_next_history (FB_DIRVIEW (fb->dirview));
}

static void
set_position (GtkMenu *menu,
	      gint *x,
	      gint *y,
	      gboolean *push_in,
	      gpointer data)
{
	gint top_x, top_y;
	GtkWidget *item = data;

	gdk_window_get_position (item->window, &top_x, &top_y);

	*x = top_x + item->allocation.x;
	*y = top_y + item->allocation.y + item->allocation.height;
	*push_in = FALSE;
}

static void
cb_show_back_history(GtkAction *action,
		     gpointer data)
{
	GtkWidget *item;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	item = gtk_ui_manager_get_widget (fb->ui_manager, "/ToolBar/ShowBackHistory");
	gtk_menu_popup(GTK_MENU(FB_DIRVIEW (fb->dirview)->prev_history),
		       NULL, NULL,
		       (GtkMenuPositionFunc) set_position, item,
		       0, gtk_get_current_event_time ());
}

static void
cb_show_next_history(GtkAction *action,
		     gpointer data)
{
	GtkWidget *item;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	item = gtk_ui_manager_get_widget (fb->ui_manager, "/ToolBar/ShowNextHistory");
	gtk_menu_popup (GTK_MENU (FB_DIRVIEW (fb->dirview)->next_history),
			NULL, NULL,
			(GtkMenuPositionFunc) set_position, item,
			0, gtk_get_current_event_time ());
}

static void
cb_clear_history(GtkAction *action,
		 gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_dirview_clear_history (FB_DIRVIEW (fb->dirview));
}

static void
cb_up_dir(GtkAction *action,
	  gpointer data)
{
	gchar *path;
	const gchar *workdir;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	workdir = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	if (!g_ascii_strcasecmp (workdir, "/")) {
		fb_statusbar_push ("%s",  _("Top directory !!"));
		return;
	}

	path = g_path_get_dirname (workdir);
	if (fb->archive_opened) {
		gchar *ext_dir, *real_path;

		ext_dir = g_strconcat (g_get_home_dir (),
				       "/.futaba/tmp", NULL);
		real_path = g_strdup (path + strlen (ext_dir));
		if (path_is_dir (real_path)) {
			g_free (path);
			path = real_path;
			fb->archive_opened = FALSE;
		}
	}
	fb_dirview_set_dir (FB_DIRVIEW (fb->dirview), path);
	g_free (path);
}

static void
cb_go_home(GtkAction *action,
	   gpointer data)
{
	const gchar *workdir, *home;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	workdir = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	fb_conf_get_value ("home page", &home, FB_CONF_TYPE_STRING, g_get_home_dir ());

	if (strcmp (workdir, home))
		fb_dirview_set_dir (FB_DIRVIEW (fb->dirview), home);
}

static void
cb_refresh_dir(GtkAction *action,
	       gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	fb_dirview_update_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_change_column_type(GtkAction *action,
		      gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

}

static void
cb_switch_file_display(GtkAction *action,
		       gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	is_active = fb_dirview_get_otherfilter (FB_DIRVIEW (fb->dirview));
	fb_dirview_set_otherfilter (FB_DIRVIEW (fb->dirview), ! is_active);
	fb_dirview_update_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_switch_dot_file(GtkAction *action,
		   gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	is_active = fb_dirview_get_dotfilter (FB_DIRVIEW (fb->dirview));
	fb_dirview_set_dotfilter (FB_DIRVIEW (fb->dirview), ! is_active);
	fb_dirview_update_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_switch_preview(GtkAction *action,
		  gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	if (GTK_WIDGET_VISIBLE (fb->preview)) {
		gtk_widget_hide (fb->preview);
	}
	else {
		gtk_widget_show (fb->preview);
	}
}

static void
cb_show_tbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(fb->tool_bar);
	else
		gtk_widget_hide(fb->tool_bar);

}
      
static void
cb_show_lbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(fb->location_bar);
	else
		gtk_widget_hide(fb->location_bar);

}
      
static void
cb_show_sbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(fb->status_bar);
	else
		gtk_widget_hide(fb->status_bar);

}

static void
cb_set_reverse(GtkAction *action,
	       gpointer data)
{
	gint sort_id;
	GtkSortType order;
	GtkTreeSortable *sortable;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	sortable = GTK_TREE_SORTABLE (FB_DIRVIEW (fb->dirview)->store);
	gtk_tree_sortable_get_sort_column_id(sortable, &sort_id, NULL);

	if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))
		order = GTK_SORT_DESCENDING;
	else
		order = GTK_SORT_ASCENDING;

	gtk_tree_sortable_set_sort_column_id(sortable, sort_id, order);
}      
      
static void
cb_show_col1(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(FB_DIRVIEW (fb->dirview)->column1, is_active);

	fb_conf_set_value ("show col1", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_show_col2(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(FB_DIRVIEW (fb->dirview)->column2, is_active);

	fb_conf_set_value ("show col2", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_show_col3(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(FB_DIRVIEW (fb->dirview)->column3, is_active);

	fb_conf_set_value ("show col3", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_show_col4(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(FB_DIRVIEW (fb->dirview)->column4, is_active);

	fb_conf_set_value ("show col4", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_show_col5(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(FB_DIRVIEW (fb->dirview)->column5, is_active);

	fb_conf_set_value ("show col5", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_show_coltitle(GtkAction *action,
		 gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(fb->dirview), is_active);

	fb_conf_set_value ("show title", &is_active, FB_CONF_TYPE_BOOLEAN);
}

static void
cb_set_dot_filter(GtkAction *action,
		  gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	fb_dirview_set_dotfilter (FB_DIRVIEW (fb->dirview), is_active);
	fb_dirview_update_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_set_other_filter(GtkAction *action,
		    gpointer data)
{
	gboolean is_active;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	fb_statusbar_pop ();

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	fb_dirview_set_otherfilter (FB_DIRVIEW (fb->dirview), is_active);
	fb_dirview_update_dir(FB_DIRVIEW (fb->dirview));
}

static void
cb_sort_mode(GtkRadioAction *action,
	     GtkRadioAction *current,
	     gpointer data)
{
	gint mode;
	FbWindow *fb = data;

	g_return_if_fail (FB_WINDOW (fb));

	mode = gtk_radio_action_get_current_value(GTK_RADIO_ACTION(current));
	fb_dirview_set_sort_mode(FB_DIRVIEW (fb->dirview), mode);
}

/* GtkActionEntry {
   const gchar *name,
   const gchar *stock_id,
   const gchar *label,
   const gchar *accelerator,
   const gchar *tooltip,
   GCallback func} */
static GtkActionEntry actions[] = {
  {"FileMenu", NULL, N_("File"), NULL, NULL, G_CALLBACK(cb_show_file_menu)},
  {"EditMenu", NULL, N_("Edit"), NULL, NULL, G_CALLBACK(cb_show_edit_menu)},
  {"ViewMenu", NULL, N_("View"), NULL, NULL, G_CALLBACK(cb_show_view_menu)},
  {"MoveMenu", NULL, N_("Move"), NULL, NULL, G_CALLBACK(cb_show_move_menu)},
  {"BookmarkMenu", NULL, N_("Bookmark")},
  {"HelpMenu", NULL, N_("Help")},
  /* File Menu */
  {"Open", "gtk-open", N_("Open"), NULL, N_("Open"), G_CALLBACK(cb_open)},
  {"OpenViewer", NULL, N_("Open Viewer"), "<control>V", N_("Open Image Viewer"), G_CALLBACK(cb_open_viewer)},
  {"OpenThumb", NULL, N_("Open Thumbnail"), "<control>T", N_("Open Thumbnail Window"), G_CALLBACK(cb_open_thumb)},
  {"CloseViewer", "gtk-close", N_("Close Viewer"), "<control>W", N_("Close Image Viewer"), G_CALLBACK(cb_close_viewer)},
  {"CloseThumb", "gtk-close", N_("Close Thumbnail"), "<shift><control>W", N_("Close Thumbnail Window"), G_CALLBACK(cb_close_thumb)},
  {"Quit", "gtk-quit", N_("Quit"), "<control>Q", N_("Quit"), G_CALLBACK(cb_quit)},
  /* Edit Menu */
  {"ClearCache", NULL, N_("Clear Cache"), NULL, N_("Clear Cache"), G_CALLBACK (cb_clear_cache)},
  {"UpdateArchive", "gtk-refresh", N_("Update Archive"), NULL, NULL, G_CALLBACK (cb_update_archive)},
  {"MakeDir", NULL, N_("Create new dir"), NULL, N_("Create new directory"), G_CALLBACK(cb_make_dir)},
  {"ChangeBulk", NULL, N_("Change bulk"), NULL, N_("Change file in bulk"), G_CALLBACK(cb_change_bulk)},
  {"Rename", NULL, N_("Rename"), NULL, N_("Rename the selected item"), G_CALLBACK(cb_rename)},
  {"Property", NULL, N_("Property"), NULL, N_("Display the properties of selected item"), G_CALLBACK(cb_property)},
  {"Copy", "gtk-copy", N_("Copy"), "<control>C", N_("Copy"), G_CALLBACK(cb_copy)},
  {"Cut", "gtk-cut", N_("Cut"), "<control>U", N_("Cut"), G_CALLBACK(cb_cut)},
  {"Paste", "gtk-paste", N_("Paste"), "<control>P", N_("Paste"), G_CALLBACK(cb_paste)},
  {"Delete", NULL, N_("Delete"), "<control>D", N_("Delete"), G_CALLBACK(cb_delete)},
  {"Preferences", "gtk-preferences", N_("Preferences..."), "<alt>P", NULL, G_CALLBACK(cb_preferences)},
  /* Visible Menu */
  {"Sort", NULL, N_("Sort")},
  {"Column", NULL, N_("Column")},
  {"Filter", NULL, N_("Filter")},

  {"SwitchFileDisplay", "fb-other-file", N_("File Display"), NULL, N_("Show/Hide another file"), G_CALLBACK(cb_switch_file_display)},
  {"SwitchDotFile", "fb-dot", N_("Dot file"), NULL, N_("Show/Hide dot file"), G_CALLBACK(cb_switch_dot_file)},
  {"ChangeColumnType", "fb-list", N_("fileinfo"), NULL, NULL, G_CALLBACK(cb_change_column_type)},

  {"SwitchPreview", "fb-preview", N_("preview"), "<alt>P", N_("Show/Hide Preview"), G_CALLBACK(cb_switch_preview)},
  /* Move Menu */
  {"BackHistory", "gtk-go-back", N_("Back"), "<alt>Left", N_("Go to the previous visited directory"), G_CALLBACK(cb_back_history)},
  {"NextHistory", "gtk-go-forward", N_("Forward"), "<alt>Right", N_("Go to the next visited directory"), G_CALLBACK(cb_next_history)},
  {"ShowBackHistory", NULL, N_("Back"), NULL, N_("Go back a few pages"), G_CALLBACK(cb_show_back_history)},
  {"ShowNextHistory", NULL, N_("Next"), NULL, N_("Go forward a number of pages"), G_CALLBACK(cb_show_next_history)},
  {"ClearHistory", NULL, N_("Clear History"), NULL, N_("Clear Back/Forward lists"), G_CALLBACK(cb_clear_history)},

  {"UpDir", "gtk-go-up", N_("Up"), "<alt>Up", N_("Go up one level"), G_CALLBACK(cb_up_dir)},
  {"GoToHome", "gtk-home", N_("Home"), "<alt>Home", N_("Go to the home"), G_CALLBACK(cb_go_home)},
  {"RefreshDir", "gtk-refresh", N_("Refresh"), "<control>R", N_("Refresh directory"), G_CALLBACK(cb_refresh_dir)},
  /* Bookmark Menu */
  {"AddBookmark", "gtk-add", N_("Add bookmark"), "<control>B", NULL, G_CALLBACK(cb_add_bookmark)},
  {"EditBookmark", NULL, N_("Edit bookmark"), NULL, NULL, G_CALLBACK(cb_edit_bookmark)},
  /* Help Menu */
  {"About", NULL, N_("About"), NULL, N_("Show credits"), G_CALLBACK(cb_about)},
};
static guint actions_len = G_N_ELEMENTS(actions);

static GtkToggleActionEntry toggle_actions[] = {
  {"TBarShow", NULL, N_("Toolbar"), NULL, NULL, G_CALLBACK(cb_show_tbar), TRUE},
  {"LBarShow", NULL, N_("Locationbar"), NULL, NULL, G_CALLBACK(cb_show_lbar), TRUE},
  {"SBarShow", NULL, N_("Statusbar"), NULL, NULL, G_CALLBACK(cb_show_sbar), TRUE},

  {"Reverse", NULL, N_("Reversed order"), NULL, NULL, G_CALLBACK(cb_set_reverse), FALSE},

  {"Col1", NULL, N_("Permission"), NULL, NULL, G_CALLBACK(cb_show_col1), FALSE},
  {"Col2", NULL, N_("Owner"), NULL, NULL, G_CALLBACK(cb_show_col2), FALSE},
  {"Col3", NULL, N_("Group"), NULL, NULL, G_CALLBACK(cb_show_col3), FALSE},
  {"Col4", NULL, N_("Size"), NULL, NULL, G_CALLBACK(cb_show_col4), FALSE},
  {"Col5", NULL, N_("Mtime"), NULL, NULL, G_CALLBACK(cb_show_col5), FALSE},
  {"ColTitle", NULL, N_("Title"), NULL, NULL, G_CALLBACK(cb_show_coltitle), FALSE},

  {"FilterDot", NULL, N_("Dot file"), NULL, NULL, G_CALLBACK(cb_set_dot_filter), TRUE},
  {"FilterOther", NULL, N_("Other file"), NULL, NULL, G_CALLBACK(cb_set_other_filter), TRUE},
};
static guint toggle_actions_len = G_N_ELEMENTS(toggle_actions);

static GtkRadioActionEntry radio_actions[] = {
  {"SortName", NULL, N_("Sort by name"), NULL, NULL, SORT_BY_NAME},
  {"SortSize", NULL, N_("Sort by size"), NULL, NULL, SORT_BY_SIZE},
  {"SortMtime", NULL, N_("Sort by mtime"), NULL, NULL, SORT_BY_MTIME},
  {"SortType", NULL, N_("Sort by type"), NULL, NULL, SORT_BY_TYPE}
};
static guint radio_actions_len = G_N_ELEMENTS(radio_actions);

/*
 *------------------------------------------------------------
 *       Public 
 *------------------------------------------------------------
 */
void
action_connect_accelerator(GtkActionGroup *action_group,
			   const gchar *path)
{
	GtkAction *action;

	action = gtk_action_group_get_action(action_group, path);
	gtk_action_connect_accelerator(action);
}

GtkActionGroup *
fb_actions_new(FbWindow *fb)
{
	GtkActionGroup *action_group;

	action_group = gtk_action_group_new("Futaba");

	gtk_action_group_set_translation_domain(action_group, NULL);

	gtk_action_group_add_actions(action_group,
				     actions, actions_len,
				     fb);

	gtk_action_group_add_toggle_actions(action_group,
					    toggle_actions, toggle_actions_len,
					    fb);

	gtk_action_group_add_radio_actions(action_group,
					   radio_actions, radio_actions_len,
					   SORT_BY_TYPE,
					   G_CALLBACK(cb_sort_mode), fb);

	return action_group;
}
