/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "fb-preview.h"

static void     fb_preview_class_init      (FbPreviewClass *klass);
static void     fb_preview_init            (FbPreview      *klass);
static void     fb_preview_finalize        (GObject        *object);
static void     fb_preview_size_allocate   (GtkWidget      *widget,
					    GtkAllocation  *allocation);

static GtkVBoxClass *parent_class = NULL;

GType
fb_preview_get_type (void)
{
	static GType object_type = 0;

	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbPreviewClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_preview_class_init,
			NULL,
			NULL,
			sizeof (FbPreview),
			32,
			(GInstanceInitFunc) fb_preview_init,
		};

		object_type = g_type_register_static (GTK_TYPE_VBOX, "FbPreview",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_preview_class_init (FbPreviewClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	object_class = (GObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	object_class->finalize = fb_preview_finalize;

	widget_class->size_allocate = fb_preview_size_allocate;
}

static void
fb_preview_init (FbPreview *pv)
{
	GtkWidget *align;

	pv->frame = gtk_frame_new(_("PreView"));
	pv->image = gtk_image_new ();
	pv->label = gtk_label_new (NULL);
	pv->path = NULL;

	align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 20, 0, 20, 20);
	gtk_box_pack_start(GTK_BOX(pv), align, TRUE, TRUE, 0);
	gtk_widget_show(align);

	gtk_container_add(GTK_CONTAINER(align), pv->frame);
	gtk_widget_show(pv->frame);

	align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 1, 1, 1, 1);
	gtk_container_add(GTK_CONTAINER(pv->frame), align);
	gtk_widget_show(align);

	gtk_container_add(GTK_CONTAINER(align), pv->image);
	gtk_widget_show(pv->image);

	align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(align), 5, 5, 0, 0);
	gtk_box_pack_start(GTK_BOX(pv), align, FALSE, FALSE, 0);
	gtk_widget_show(align);

	gtk_container_add(GTK_CONTAINER(align), pv->label);
	gtk_widget_show(pv->label);

}

static void
fb_preview_size_allocate (GtkWidget *widget,
			  GtkAllocation *allocation)
{
	gint frame_w, frame_h, image_w, image_h;
	GdkPixbuf *pixbuf;
	FbPreview *pv = FB_PREVIEW (widget);

	GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);

	g_return_if_fail ((allocation->width > 0) || (allocation->height > 0));

	if ((widget->allocation.width == allocation->width) &&
	    (widget->allocation.height == allocation->height)) return;
	if (! pv->path) return;

	frame_w = allocation->width - 40;
	frame_h = allocation->height - 50;

	gdk_pixbuf_get_file_info (pv->path, &image_w, &image_h);

	if ((frame_w >= image_w) && (frame_h >= image_h)) return;

	pixbuf = gdk_pixbuf_new_from_file_at_size (pv->path, frame_w, frame_h, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (pv->image), pixbuf);
	gdk_pixbuf_unref (pixbuf);
}

static void
fb_preview_finalize (GObject *object)
{
	FbPreview *pv = FB_PREVIEW (object);

	if (pv->path) g_free (pv->path);

	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
fb_preview_new (void)
{
	GObject *object = g_object_new (FB_TYPE_PREVIEW, NULL);

	return GTK_WIDGET (object);
}

void
fb_preview_set_image (FbPreview *pv,
		      const gchar *path)
{
	gint width, height, image_w, image_h;
	gchar *label;
	GdkPixbuf *pixbuf;

	g_return_if_fail (FB_IS_PREVIEW (pv));

	if (! path) {
		gtk_frame_set_label (GTK_FRAME (pv->frame), NULL);
		gtk_image_set_from_file (GTK_IMAGE (pv->image), NULL);
		gtk_label_set_text (GTK_LABEL (pv->label), NULL);
		if (pv->path) g_free (pv->path);

		return;
	}

	g_return_if_fail (path_is_file (path));

	width = pv->frame->allocation.width;
	height = pv->frame->allocation.height;

	gdk_pixbuf_get_file_info (path, &image_w, &image_h);

	if ((width >= image_w) && (height >= image_h)) {
		width = image_w;
		height = image_h;
	}

	pixbuf = gdk_pixbuf_new_from_file_at_size (path, width, height, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (pv->image), pixbuf);
	gdk_pixbuf_unref (pixbuf);

	label = g_path_get_basename (path);
	gtk_frame_set_label (GTK_FRAME (pv->frame), label);
	g_free (label);

	label = g_strdup_printf ("pixel: %4d x %4d", image_w, image_h);
	gtk_label_set_text (GTK_LABEL (pv->label), label);
	g_free (label);

	pv->path = g_strdup (path);
}
