/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "fb-window.h"
#include "futaba.h"
#include "fb-conf.h"
#include "fb-actions.h"
#include "fb-dirview.h"
#include "fb-preview.h"
#include "fb-statusbar.h"
#include "fb-locationbar.h"
#include "fb-pathmenuitem.h"
#include "fb-viewer.h"
#include "fr-archive.h"

GList *paste_list = NULL;
gboolean is_copy = TRUE;
/* futaba window */
static void     fb_window_class_init     (FbWindowClass *klass);
static void     fb_window_init           (FbWindow      *fb);
static gboolean fb_window_delete_event   (GtkWidget     *widget,
					  GdkEventAny   *event);
static void     fb_window_destroy        (GtkObject     *object);
static void     fb_window_finalize       (GObject       *object);
/* callback */
static void     cb_locate_path           (FbLocationbar *lb,
					  gchar         *path,
					  gpointer       data);
static gboolean cb_show_popup            (GtkWidget      *wg,
					  GdkEventButton *event,
					  gpointer        gpointer);
static void     cb_change_preview        (GtkTreeView    *tview,
					  gpointer        data);
static void     cb_set_path              (FbDirview      *dv,
					  gchar          *path,
					  gint            type,
					  gpointer        data);
static void     cb_set_message           (FbDirview      *dv,
					  gpointer        data);
/* utils */
static void     ui_init                  (FbWindow       *fb);
static void     restore_status           (FbWindow       *fb);
/* callback for utils func */
static gboolean cb_show_message          (GtkWidget        *wg,
					  GdkEventCrossing *event,
					  gpointer          data);
static gboolean cb_remove_message        (GtkWidget        *wg,
					  GdkEventCrossing *event,
					  gpointer          data);

static GtkWindowClass *parent_class = NULL;

GType
fb_window_get_type (void)
{
	static GType object_type = 0;

	if (! object_type) {
		static const GTypeInfo object_info = {
			sizeof (FbWindowClass),
			NULL,
			NULL,
			(GClassInitFunc) fb_window_class_init,
			NULL,
			NULL,
			sizeof (FbWindow),
			32,
			(GInstanceInitFunc) fb_window_init,
		};

		object_type = g_type_register_static (GTK_TYPE_WINDOW, "FbWindow",
						      &object_info, 0);
	}

	return object_type;
}

static void
fb_window_class_init (FbWindowClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	gobject_class->finalize = fb_window_finalize;

	object_class->destroy = fb_window_destroy;

	widget_class->delete_event = fb_window_delete_event;
}

static const gchar *fb_ui = {
	"<ui>"
	"  <menubar name=\"MenuBar\">"
	"    <menu action=\"FileMenu\">"
	"      <menuitem action=\"OpenViewer\"/>"
	"      <menuitem action=\"OpenThumb\"/>"
	"      <separator/>"
	"      <menuitem action=\"CloseViewer\"/>"
	"      <menuitem action=\"CloseThumb\"/>"
	"      <menuitem action=\"Quit\"/>"
	"    </menu>"
	"    <menu action=\"EditMenu\">"
/* 	"      <menuitem action=\"ClearCache\"/>" */
/* 	"      <separator/>" */
	"      <menuitem action=\"Copy\"/>"
	"      <menuitem action=\"Cut\"/>"
	"      <menuitem action=\"Paste\"/>"
	"      <menuitem action=\"Delete\"/>"
	"      <separator/>"
	"      <menuitem action=\"Preferences\"/>"
	"    </menu>"
	"    <menu action=\"ViewMenu\">"
	"      <menuitem action=\"RefreshDir\"/>"
	"      <separator/>"
	"      <menuitem action=\"TBarShow\"/>"
	"      <menuitem action=\"LBarShow\"/>"
	"      <menuitem action=\"SBarShow\"/>"
	"      <separator/>"
	"      <menu action=\"Sort\">"
	"        <menuitem action=\"SortName\"/>"
	"        <menuitem action=\"SortSize\"/>"
	"        <menuitem action=\"SortMtime\"/>"
	"        <menuitem action=\"SortType\"/>"
	"        <separator/>"
	"        <menuitem action=\"Reverse\"/>"
	"      </menu>"
	"      <menu action=\"Column\">"
	"        <menuitem action=\"Col1\"/>"
	"        <menuitem action=\"Col2\"/>"
	"        <menuitem action=\"Col3\"/>"
	"        <menuitem action=\"Col4\"/>"
	"        <menuitem action=\"Col5\"/>"
	"        <separator/>"
	"        <menuitem action=\"ColTitle\"/>"
	"      </menu>"
	"      <menu action=\"Filter\">"
	"        <menuitem action=\"FilterDot\"/>"
	"        <menuitem action=\"FilterOther\"/>"
	"      </menu>"
	"    </menu>"
	"    <menu action=\"MoveMenu\">"
	"      <menuitem action=\"UpDir\"/>"
	"      <menuitem action=\"BackHistory\"/>"
	"      <menuitem action=\"NextHistory\"/>"
	"      <separator/>"
	"      <menuitem action=\"GoToHome\"/>"
	"      <separator/>"
	"      <menuitem action=\"ClearHistory\"/>"
	"    </menu>"
	"    <menu action=\"BookmarkMenu\">"
	"      <menuitem action=\"AddBookmark\"/>"
	"      <menuitem action=\"EditBookmark\"/>"
	"    </menu>"
	"    <menu action=\"HelpMenu\">"
	"      <menuitem action=\"About\"/>"
	"    </menu>"
	"  </menubar>"
	"  <toolbar name=\"ToolBar\">"
	"    <toolitem action=\"BackHistory\"/>"
	"    <toolitem action=\"ShowBackHistory\"/>"
	"    <toolitem action=\"NextHistory\"/>"
	"    <toolitem action=\"ShowNextHistory\"/>"
	"    <toolitem action=\"UpDir\"/>"
	"    <toolitem action=\"GoToHome\"/>"
	"    <toolitem action=\"RefreshDir\"/>"
	"    <separator/>"
	"    <toolitem action=\"SwitchDotFile\"/>"
	"    <toolitem action=\"SwitchFileDisplay\"/>"
	/*     "    <toolitem action=\"ChangeColumnType\"/>" */
	"    <separator/>"
	"    <toolitem action=\"SwitchPreview\"/>"
	"  </toolbar>"
	"  <popup name=\"DirViewPopup\">"
	"    <menuitem action=\"Open\"/>"
	"    <menuitem action=\"OpenViewer\"/>"
	"    <menuitem action=\"OpenThumb\"/>"
	"    <separator/>"
	"    <menuitem action=\"Rename\"/>"
	"    <menuitem action=\"Copy\"/>"
	"    <menuitem action=\"Cut\"/>"
	"    <menuitem action=\"Paste\"/>"
	"    <menuitem action=\"Delete\"/>"
	"    <separator/>"
	"    <menuitem action=\"MakeDir\"/>"
	"    <menuitem action=\"UpdateArchive\"/>"
	/*   "    <menuitem action=\"ChangeBulk\"/>" */
/* 	"    <menuitem action=\"Property\"/>" */
	"  </popup>"
	"</ui>"
};

static void
ui_init (FbWindow *fb)
{
	GtkWidget *item, *arrow;

	/* File Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/FileMenu/OpenViewer",
		       _("Open viewer"));
	ui_set_message(fb->ui_manager, "/MenuBar/FileMenu/OpenThumb",
		       _("Open thumbnail window"));
	ui_set_message(fb->ui_manager, "/MenuBar/FileMenu/CloseViewer",
		       _("Close viewer"));
	ui_set_message(fb->ui_manager, "/MenuBar/FileMenu/CloseThumb",
		       _("Close Thumbnail Window"));
	ui_set_message(fb->ui_manager, "/MenuBar/FileMenu/Quit", 
		       _("Quit futaba"));
	/* Edit Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Copy",
		       _("Copy the selected item"));
	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Cut",
		       _("Cut the selected item"));
	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Paste",
		       _("Paste the stocked item"));
	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Delete",
		       _("Delete  the selected item"));
	ui_set_message(fb->ui_manager, "/MenuBar/EditMenu/Preferences", 
		       _("Edit preferences"));
	/* View Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/RefreshDir",
		       _("Refresh Directory"));

	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/TBarShow",
		       _("Change the visibility of toolbar"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/LBarShow",
		       _("Change the visibility of locationbar"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/SBarShow",
		       _("Change the visibility of statusbar"));

	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Sort/SortName",
		       _("Keep items sorted by name"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Sort/SortSize",
		       _("Keep items sorted by size"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Sort/SortMtime",
		       _("Keep items sorted by modification data"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Sort/SortType",
		       _("Keep items sorted by type"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Sort/Reverse",
		       _("Display items in the opposite order"));

	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/Col1",
		       _("Change the visibility of Permission column"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/Col2",
		       _("Change the visibility of Owner column"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/Col3",
		       _("Change the visibility of Group column"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/Col4",
		       _("Change the visibility of Size column"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/Col5",
		       _("Change the visibility of Mtime column"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Column/ColTitle",
		       _("Change the visibility of Column title"));

	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Filter/FilterDot",
		       _("Change the visibility of Dot file"));
	ui_set_message(fb->ui_manager, "/MenuBar/ViewMenu/Filter/FilterOther",
		       _("Change the visibility of Other file"));

	/* Move Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/MoveMenu/UpDir",
		       _("Go up one level"));
	ui_set_message(fb->ui_manager, "/MenuBar/MoveMenu/BackHistory",
		       _("Go to the previous visited directory"));
	ui_set_message(fb->ui_manager, "/MenuBar/MoveMenu/NextHistory",
		       _("Go to the next visited directory"));
	ui_set_message(fb->ui_manager, "/MenuBar/MoveMenu/GoToHome",
		       _("Go to the home directory"));
	ui_set_message(fb->ui_manager, "/MenuBar/MoveMenu/ClearHistory",
		       _("Clear Back/Forward lists"));
	/* Bookmark Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/BookmarkMenu/AddBookmark",
		       _("Add a bookmark for the current directory to this menu"));
	ui_set_message(fb->ui_manager, "/MenuBar/BookmarkMenu/EditBookmark",
		       _("Display a window editing the bookmarks"));
	/* Help Menu */
	ui_set_message(fb->ui_manager, "/MenuBar/HelpMenu/About",
		       _("Display credits"));

	/* DirView Popup */
	ui_set_message(fb->ui_manager, "/DirViewPopup/Open",
		       _("Open the selected item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/OpenViewer",
		       _("Open the selected item with viewer"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/OpenThumb",
		       _("Open the current directory from thumbnail window"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/Rename",
		       _("Rename the selected item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/Copy",
		       _("Copy the selected item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/Cut",
		       _("Cut the selected item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/Paste",
		       _("Paste the stocked item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/Delete",
		       _("Delete the selected item"));
	ui_set_message(fb->ui_manager, "/DirViewPopup/MakeDir",
		       _("Create a new directory"));
	/*   ui_set_message(fb->ui_manager, "/DirViewPopup/ChangeBulk", */
	/* 		    _("Change a lot of file name")); */
/* 	ui_set_message(fb->ui_manager, "/DirViewPopup/Property", */
/* 		       _("Display the properties of selected item")); */


	/* toolbar */
	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_widget_show (arrow);

	item = gtk_ui_manager_get_widget (fb->ui_manager, "/ToolBar/ShowBackHistory");
	gtk_tool_item_set_homogeneous(GTK_TOOL_ITEM(item), FALSE);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(item), arrow);

	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_widget_show(arrow);

	item = gtk_ui_manager_get_widget(fb->ui_manager, "/ToolBar/ShowNextHistory");
	gtk_tool_item_set_homogeneous(GTK_TOOL_ITEM(item), FALSE);
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(item), arrow);

}

void
ui_set_active (GtkUIManager *ui_manager,
	       const gchar *path,
	       gboolean is_active)
{
	GtkAction *action;

	action = gtk_ui_manager_get_action (ui_manager, path);
	if (action)
		gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), is_active);
}

void
ui_set_sensitive (GtkUIManager *ui_manager,
		  const gchar *path,
		  gboolean is_sensitive)
{
	GtkWidget *item;

	item = gtk_ui_manager_get_widget (ui_manager, path);
	if (item)
		gtk_widget_set_sensitive (item, is_sensitive);
}

static gboolean
cb_show_message(GtkWidget *wg,
		GdkEventCrossing *event,
		gpointer data)
{
	gchar *message = data;

	fb_statusbar_push ("%s", message);

	return FALSE;
}

static gboolean
cb_remove_message(GtkWidget *wg,
		  GdkEventCrossing *evnet,
		  gpointer data)
{
	fb_statusbar_pop ();

	return FALSE;
}

void
ui_set_message (GtkUIManager *ui_manager,
		const gchar *path,
		const gchar *message)
{
	GtkWidget *item;

	g_return_if_fail (path != NULL);

	item = gtk_ui_manager_get_widget (ui_manager, path);
	g_return_if_fail (item != NULL);

	g_signal_connect (G_OBJECT (item), "enter-notify-event",
			  G_CALLBACK (cb_show_message), (gpointer) message);
	g_signal_connect (G_OBJECT (item), "leave-notify-event",
			  G_CALLBACK (cb_remove_message), NULL);
}

void
widget_set_message (GtkWidget *wg,
		    const gchar *message)
{
	g_signal_connect (G_OBJECT (wg), "enter-notify-event",
			  G_CALLBACK (cb_show_message), (gpointer) message);
	g_signal_connect (G_OBJECT (wg), "leave-notify-event",
			  G_CALLBACK (cb_remove_message), NULL);
}

static void
cb_locate_path (FbLocationbar *lb,
		gchar *path,
		gpointer data)
{
	gchar *set_path;
	FbWindow *fb = data;

	set_path = g_strdup (path);
	if (fb->archive_opened) {
		gchar *ext_path;

		ext_path = g_strconcat (g_get_home_dir (),
					"/.futaba/tmp",
					path, NULL);
		if (path_is_exists (ext_path)) {
			g_free (set_path);
			set_path = ext_path;
		}
		else {
			g_free (ext_path);
			fb->archive_opened = FALSE;
		}
	}

	fb_dirview_set_dir (FB_DIRVIEW (fb->dirview), set_path);
	g_free (set_path);
}

static gboolean
cb_show_popup (GtkWidget *wg,
	       GdkEventButton *event,
	       gpointer data)
{
	gboolean flag;
	gchar *path;
	GtkTreePath *tpath;
	GtkTreeViewColumn *column;
	GtkWidget *menu;
	FbDirview *dv = FB_DIRVIEW (wg);
	FbWindow *fb = data;

	if (event->button != 3) return FALSE;

	flag = (dv->image_num > 0);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/OpenViewer", flag);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/OpenThumb", flag);

	/* check write parmission */
	flag = dv->writable;
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Rename", flag);
	/*       ui_set_sensitive(fb->ui_manager, "/DirViewPopup/ChangeBulk", flag); */
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/MakeDir", flag);
	flag = (flag && (paste_list != NULL));
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Paste", flag);


	flag = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (dv),
					      event->x, event->y,
					      &tpath, &column,
					      NULL, NULL);
	if (flag) {
		gtk_tree_view_set_cursor(GTK_TREE_VIEW(dv), tpath, column, FALSE);
		gtk_tree_path_free (tpath);

		fb_dirview_row_get (dv,
				    COLUMN_DISPLAY_PATH, &path,
				    -1);
	}

	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Open", flag);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Property", flag);
 
	if (flag) {
		if ((path[0] == '.') && ((path[1] == '\0') || (path[1] == '.')))
			flag = FALSE;
		g_free (path);
	}

	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Copy", flag);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Cut", flag);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Delete", flag);
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/Rename", flag);

	flag = file_is_archive (fb_dirview_get_workdir (dv));
	ui_set_sensitive(fb->ui_manager, "/DirViewPopup/UpdateArchive", flag);

	menu = gtk_ui_manager_get_widget(fb->ui_manager, "/DirViewPopup");
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		       event->button, event->time);

	return TRUE;
}

static void
cb_change_preview (GtkTreeView *tview,
		   gpointer data)
{
	gint type;
	gchar *path, *set_path, *size, *message;
	FbWindow *fb = data;

	fb_statusbar_pop ();

	fb_dirview_row_get (FB_DIRVIEW (fb->dirview),
			    COLUMN_TYPE, &type,
			    COLUMN_PATH, &path,
			    COLUMN_DISPLAY_SIZE, &size,
			    -1);

	if (fb->archive_opened) {
		gchar *rm_path;

		rm_path = g_strconcat (g_get_home_dir (),
				       "/.futaba/tmp", NULL);
		set_path = g_strdup (path + strlen (rm_path));
		g_free (rm_path);
	}
	else
		set_path = g_strdup (path);

	if (type <= DIRECTORY)
		fb_statusbar_push(_("\"%s\" selected (containing %s)"), set_path, size);
	else
		fb_statusbar_push(_("\"%s\" selected (size %s)"), set_path, size);
	g_free (set_path);
	g_free (size);

	if (type != IMAGE)
		goto fin;

	if (! GTK_WIDGET_VISIBLE (fb->preview)) return;

	set_path = FB_PREVIEW (fb->preview)->path;
	if (set_path && ! strcmp (set_path, path))
		goto fin;

	while (gtk_events_pending()) gtk_main_iteration();

	fb_preview_set_image(FB_PREVIEW (fb->preview), path);

 fin:
	g_free (path);
}

static void
cb_archive_progress (FRArchive *archive,
		     double fraction,
		     gpointer data)
{
	FbWindow *fb = data;

	fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), CLAMP (fraction, 0, 1));
}

static void
cb_set_path (FbDirview *dv,
	     gchar *path,
	     gint type,
	     gpointer data)
{
	FbWindow *fb = data;

	g_return_if_fail (FB_IS_WINDOW (fb));

	if (fb->archive_opened) {
		gchar *ext_dir;

		ext_dir = g_strconcat (g_get_home_dir (),
				       "/.futaba/tmp",
				       NULL);
		fb->archive_opened = g_str_has_prefix (path, ext_dir);
		g_free (ext_dir);
	}

	switch (type) {
	case WORKDIR:
	case PARENTDIR:
	case DIRECTORY: {
		fb_dirview_set_dir (dv, path);
	}
		break;
	case IMAGE: {
		GList *path_list;

		path_list = fb_dirview_get_path_list (dv, IMAGE);
		fb_viewer_start (path_list, path);
		path_list_free (path_list);
	}
		break;
	case ARCHIVE: {
		gchar *extract_path;
		FRArchive *archive;
		
		fb->archive_opened = TRUE;

		extract_path = g_strconcat (g_get_home_dir (),
					    "/.futaba/tmp", path,
					    NULL);

		if (path_is_exists (extract_path)) {
			fb_dirview_set_dir (dv, extract_path);

			g_free (extract_path);
			return;
		}

       		archive = fr_archive_new ();
		g_signal_connect (G_OBJECT(archive), "done",
				  G_CALLBACK(gtk_main_quit), NULL);
		g_signal_connect(G_OBJECT (archive), "progress",
				 G_CALLBACK (cb_archive_progress), fb);
		
		if (fr_archive_load (archive, path, NULL)) {

	     		gtk_main (); /* wait loading file list */

			if (ensure_dir_exists (extract_path, 0755)) {

				fr_archive_extract (archive, NULL, extract_path, NULL, 
						    FALSE, TRUE, FALSE, NULL); 
				gtk_main (); /* wait extracting archive */

				fb_dirview_set_dir (dv, extract_path);
			}
		}

		fb_statusbar_set_progress (FB_STATUSBAR (fb->status_bar), 0.0);
		g_object_unref (archive);

		g_free (extract_path);
	}
		break;
	default:
		break;
	}
}

static void
toolbar_set_sensitive (FbWindow *fb)
{
	gboolean flag;
	const gchar *workdir;
	GList *history;

	history = gtk_container_get_children (GTK_CONTAINER (FB_DIRVIEW (fb->dirview)->prev_history));
	flag = (history != NULL);
	ui_set_sensitive (fb->ui_manager, "/ToolBar/BackHistory", flag);
	ui_set_sensitive (fb->ui_manager, "/ToolBar/ShowBackHistory", flag);

	history = gtk_container_get_children (GTK_CONTAINER (FB_DIRVIEW (fb->dirview)->next_history));
	flag = (history != NULL);
	ui_set_sensitive (fb->ui_manager, "/ToolBar/NextHistory", flag);
	ui_set_sensitive (fb->ui_manager, "/ToolBar/ShowNextHistory", flag);

	workdir = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	flag = g_ascii_strcasecmp (workdir, "/");
	ui_set_sensitive (fb->ui_manager, "/ToolBar/UpDir", flag);
}

static void
cb_set_message (FbDirview *dv,
		gpointer data)
{
	gint num;
	const gchar *workdir;
	gchar *set_path;
	FbWindow *fb = data;

	g_return_if_fail (FB_IS_WINDOW (fb));

	toolbar_set_sensitive (fb);

	workdir = fb_dirview_get_workdir (dv);
	if (fb->archive_opened) {
		gchar *rm_path;
		rm_path = g_strconcat (g_get_home_dir (),
				       "/.futaba/tmp", NULL);
		set_path = g_strdup (workdir + strlen (rm_path));
		g_free (rm_path);
	}
	else 
		set_path = g_strdup (workdir);

	fb_locationbar_set_text (FB_LOCATIONBAR (fb->location_bar), set_path);

	num = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (dv->store), NULL);
	if (! pdir_filter) num--;
	if (! cdir_filter) num--;

	if (num == 0)
		fb_statusbar_swap(_("\"%s\" open, No item"), set_path);
	else if (num == 1)
		fb_statusbar_swap(_("\"%s\" open, 1 item exist"), set_path);
	else if (num > 1)
		fb_statusbar_swap(_("\"%s\" open, %d items exist"), set_path, num);


	g_free (set_path);
}

static void
cb_judge_archive (FbDirview *dv,
		  gchar *path,
		  gpointer data)
{
	gchar *ext_dir;
	FbWindow *fb = data;

	/* FIXME (dirview perform manipulating archive) */
	ext_dir = g_strconcat (g_get_home_dir (),
			       "/.futaba/tmp", NULL);
	fb->archive_opened = g_str_has_prefix (path, ext_dir);
	g_free (ext_dir);
}

static void
fb_window_init (FbWindow *fb)
{
	GtkActionGroup *action_group;
	GtkWidget *menu, *vbox, *pane, *scroll;
	GError *error = NULL;

	fb->ui_manager = gtk_ui_manager_new ();
	action_group = fb_actions_new (fb);

	gtk_ui_manager_insert_action_group (fb->ui_manager, action_group, 0);
	if (! gtk_ui_manager_add_ui_from_string (fb->ui_manager, fb_ui, -1, &error)) {
		g_print ("UI init error: %s\n", error->message);
		g_error_free (error);
	}
	gtk_window_add_accel_group (GTK_WINDOW (fb), gtk_ui_manager_get_accel_group (fb->ui_manager));

	fb->tool_bar = gtk_ui_manager_get_widget (fb->ui_manager, "/ToolBar");
	fb->location_bar = fb_locationbar_new ();
	fb->dirview = fb_dirview_new ();
	fb->preview = fb_preview_new ();
	fb->status_bar = fb_statusbar_new ();

	fb->archive_opened = FALSE;

/* 	gtk_window_set_icon (GTK_WINDOW (fb), fb_icon); */
	gtk_window_set_title(GTK_WINDOW(fb), "Futaba");
	gtk_window_set_default_size (GTK_WINDOW (fb), 600, 500);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(fb), vbox);
	gtk_widget_show(vbox);

	menu = gtk_ui_manager_get_widget (fb->ui_manager, "/MenuBar");
	gtk_box_pack_start (GTK_BOX (vbox), menu, FALSE, FALSE, 0);
	gtk_widget_show (GTK_WIDGET (menu));

	gtk_toolbar_set_style (GTK_TOOLBAR (fb->tool_bar), GTK_TOOLBAR_ICONS);
	gtk_box_pack_start (GTK_BOX (vbox), fb->tool_bar, FALSE, FALSE, 0);
	gtk_widget_show (GTK_WIDGET (fb->tool_bar));

	/* location bar */
	g_signal_connect (G_OBJECT (fb->location_bar), "location-activate",
			  G_CALLBACK (cb_locate_path), fb);
	gtk_box_pack_start (GTK_BOX (vbox), fb->location_bar, FALSE, FALSE, 0);
	gtk_widget_show (fb->location_bar);

	pane = gtk_hpaned_new ();
	gtk_paned_set_position (GTK_PANED (pane), 300);
	gtk_box_pack_start (GTK_BOX (vbox), pane, TRUE, TRUE, 0);
	gtk_widget_show (pane);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
					    GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_paned_pack1(GTK_PANED(pane), scroll, TRUE, TRUE);
	gtk_widget_show (scroll);

	/* dirview */
	g_signal_connect (G_OBJECT (fb->dirview), "button-press-event",
			  G_CALLBACK (cb_show_popup), fb);
	g_signal_connect (G_OBJECT (fb->dirview), "cursor-changed",
			  G_CALLBACK (cb_change_preview), fb);
	g_signal_connect (G_OBJECT (fb->dirview), "path-activated",
			  G_CALLBACK (cb_set_path), fb);
	g_signal_connect (G_OBJECT (fb->dirview), "path-setted",
			  G_CALLBACK (cb_set_message), fb);
	g_signal_connect (G_OBJECT (fb->dirview), "history-move",
			  G_CALLBACK (cb_judge_archive), fb);
	gtk_container_add (GTK_CONTAINER (scroll), fb->dirview);
	gtk_widget_show(fb->dirview);

	/* preview */
	gtk_paned_pack2(GTK_PANED(pane), fb->preview, TRUE, TRUE);
	gtk_widget_show(fb->preview);

	/* status bar */
	gtk_box_pack_start(GTK_BOX(vbox), fb->status_bar, FALSE, FALSE, 0);
	gtk_widget_show(fb->status_bar);

	ui_init (fb);
	restore_status (fb);
}

static void
restore_status (FbWindow *fb)
{
	FILE *fptr;
	gboolean flag;
	gint width, height;
	gchar *path, *tmp, buf[1024];

	g_return_if_fail (FB_IS_WINDOW (fb));

	fb_conf_get_value ("fb_w", &width, FB_CONF_TYPE_INT, "600");
	fb_conf_get_value ("fb_h", &height, FB_CONF_TYPE_INT, "500");
	gtk_window_set_default_size (GTK_WINDOW (fb), width, height);

	fb_conf_get_value ("show title", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (fb->dirview), flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/ColTitle", flag);

	fb_conf_get_value ("show col1", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_column_set_visible (FB_DIRVIEW (fb->dirview)->column1, flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/Col1", flag);

	fb_conf_get_value ("show col2", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_column_set_visible (FB_DIRVIEW (fb->dirview)->column2, flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/Col2", flag);

	fb_conf_get_value ("show col3", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_column_set_visible (FB_DIRVIEW (fb->dirview)->column3, flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/Col3", flag);

	fb_conf_get_value ("show col4", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_column_set_visible (FB_DIRVIEW (fb->dirview)->column4, flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/Col4", flag);

	fb_conf_get_value ("show col5", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	gtk_tree_view_column_set_visible (FB_DIRVIEW (fb->dirview)->column5, flag);
	ui_set_active (fb->ui_manager, "/MenuBar/ViewMenu/Column/Col5", flag);

	fb_conf_get_value ("show preview", &flag, FB_CONF_TYPE_BOOLEAN, "FALSE");
	if (! flag) gtk_widget_hide (fb->preview);

	/* load bookmark */
	path = g_strconcat (g_get_home_dir (),
			    "/.futaba/bookmarks",
			    NULL);

	fptr = fopen (path, "r");
	while (fgets (buf, 1024, fptr)) {
		if (buf[0] == '#' || buf[0] == '\n') continue;

		tmp = g_strndup (buf, strlen (buf) - 1);
		fb_window_add_bookmark (fb, tmp);
		g_free (tmp);
	}
	fclose (fptr);
	g_free (path);

	/* load accel */
	path = g_strconcat (g_get_home_dir (),
			    "/.futaba/keyaccelrc",
			    NULL);

	gtk_accel_map_load (path);

	g_free (path);
}

void
fb_window_store_status (FbWindow *fb)
{
	FILE *fptr;
	gboolean flag;
	gchar *path;
	const gchar *tmp;
	GList *child;
	GtkWidget *menu, *item;

	g_return_if_fail (FB_IS_WINDOW (fb));

	fb_conf_set_value ("fb_w", &GTK_WIDGET (fb)->allocation.width, FB_CONF_TYPE_INT);
	fb_conf_set_value ("fb_h", &GTK_WIDGET (fb)->allocation.height, FB_CONF_TYPE_INT);

	tmp = fb_dirview_get_workdir (FB_DIRVIEW (fb->dirview));
	fb_conf_set_value ("last page", tmp, FB_CONF_TYPE_STRING);

	flag = GTK_WIDGET_VISIBLE (fb->preview);
	fb_conf_set_value ("show preview", &flag , FB_CONF_TYPE_BOOLEAN);

	/* save bookmark */
	item = gtk_ui_manager_get_widget (fb->ui_manager, "/MenuBar/BookmarkMenu");
	menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (item));
	child = gtk_container_get_children (GTK_CONTAINER (menu));

	path = g_strconcat (g_get_home_dir (),
			    "/.futaba/bookmarks",
			    NULL);

	fptr = fopen (path, "w");
	g_free (path);
	g_fprintf (fptr, "# futaba generated last bookmark\n");
	for (; child; child = child->next) {
		if (! FB_IS_PATH_MENU_ITEM (child->data)) continue;

		tmp = fb_path_menu_item_get_path (FB_PATH_MENU_ITEM (child->data));
		g_fprintf (fptr, "%s\n", tmp);
	}
	fclose (fptr);

	/* save accel */
	path = g_strconcat (g_get_home_dir (),
			    "/.futaba/keyaccelrc",
			    NULL);

	gtk_accel_map_save (path);

	g_free (path);
}

static gboolean
fb_window_delete_event (GtkWidget *widget,
			GdkEventAny *event)
{
	FbWindow *fb = FB_WINDOW (widget);

	fb_window_store_status (fb);

	return FALSE;
}

static void
fb_window_destroy (GtkObject *object)
{
	FbWindow *fb = FB_WINDOW (object);

/* 	fb_window_store_status (fb); */

	GTK_OBJECT_CLASS (parent_class)->destroy (object);

	gtk_main_quit ();
}

static void
fb_window_finalize (GObject *object)
{
	FbWindow *fb = FB_WINDOW (object);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
fb_window_new (const gchar *path)
{
	GObject *object = g_object_new (FB_TYPE_WINDOW, NULL);
	FbWindow *fb = FB_WINDOW (object);

	if (path) fb_dirview_set_dir (FB_DIRVIEW (fb->dirview), path);

	return GTK_WIDGET (object);
}

static void
cb_set_bookmark (GtkMenuItem *item,
		 gpointer data)
{
	const gchar *path;
	FbWindow *fb = data;

	path = fb_path_menu_item_get_path (FB_PATH_MENU_ITEM (item));
	fb_dirview_set_dir (FB_DIRVIEW (fb->dirview), path);
}

void
fb_window_add_bookmark (FbWindow *fb,
			const gchar *path)
{
	GList *child;
	GtkWidget *bmark, *item, *menu;

	g_return_if_fail (FB_IS_WINDOW (fb));

	if (! path_is_exists (path)) return;

	item = gtk_ui_manager_get_widget (fb->ui_manager, "/MenuBar/BookmarkMenu");
	menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (item));

	child = gtk_container_get_children (GTK_CONTAINER (menu));
	if (g_list_length (child) == 4) {
		GtkWidget *sepa;

		sepa = gtk_separator_menu_item_new ();
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), sepa);
		gtk_widget_show (sepa);
	}

	bmark = fb_path_menu_item_new (path);
	g_signal_connect (G_OBJECT (bmark), "activate",
			  G_CALLBACK (cb_set_bookmark), fb);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), bmark);
	gtk_widget_show (bmark);
}

void
fb_window_remove_bookmark (FbWindow *fb,
			   gint nth)
{
	GList *child;
	GtkWidget *item, *menu, *bookmark;

	g_return_if_fail (FB_IS_WINDOW (fb));

	item = gtk_ui_manager_get_widget (fb->ui_manager, "/MenuBar/BookmarkMenu");
	menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (item));
	child = gtk_container_get_children (GTK_CONTAINER (menu));

	bookmark = g_list_nth_data (child, nth + 5);
	if (bookmark) gtk_widget_destroy (bookmark);
}
