#include "stdafx.hpp"

#include "FWatchApp.hpp"
#include "resource.h"

#include "Utility.hpp"
#include "tstringUty.hpp"

#include "WatchTask.hpp"

#include <assert.h>

#include <string>
#include <vector>


WatchTask::WatchTask( const CSettingInfo& v_settingInfo )
	: settingInfo_( v_settingInfo )
	, dirChangeNotification_(
		v_settingInfo.getWatchDir(),
		v_settingInfo.getMaxDepth() > 0,
		v_settingInfo.getDirNotificationAPIRetryInterval() * 1000,
		v_settingInfo.getDirNotificationAPIExpirySpan() *1000, 
		v_settingInfo.isUsingDirNotificationAPI()
		)
	, enumerationSignal_( v_settingInfo.getDelaySignalSpan() )
	, forceEnumerationTimer_(
		v_settingInfo.getForceInterval() * 1000,
		::GetTickCount(),
		v_settingInfo.getForceInterval() > 0
		)
	, garbageCollectTimer_( 60 * 1000, 0, true )
	, fileTimeComparator_( v_settingInfo.getTolerance() )
	, nameFilter_( false )
{
	const ActionInvokerFactory actionInvokerFactory( settingInfo_ );
	pActionInvoker_ = actionInvokerFactory.create();

	std::vector<tstring> matches;
	split( settingInfo_.getWatchFile(), ';', std::back_insert_iterator<std::vector<tstring> >( matches ) );
	for( std::vector<tstring>::iterator ite_matches = matches.begin();
		ite_matches != matches.end();
		++ite_matches
		)
	{
		const tstring nameMatch = tstringuty::trimSpaceBoth( *ite_matches );
		if( ! nameMatch.empty() ) {
			nameFilter_.addPattern( SimplePatternMatch( nameMatch ) );
		}
	}
}


WatchTask::~WatchTask()
{
	//Stop(); // ̃NXj鎞_(eNXł͂Ȃ)ŃXbh͒~Ȃ΂Ȃ܂B
}

const CSettingInfo& WatchTask::getSettingInfo() const
{
	return settingInfo_;
}

bool WatchTask::getFileSize( const tstring& v_path, DWORD* v_pSizeLow, DWORD* v_pSizeHi )
{
	//required:
	assert( ! v_path.empty() && "t@Cɋ󕶎͎wł܂B" );
	
	//do:
	HANDLE fileHandle = ::CreateFile( v_path.c_str(), GENERIC_READ, FILE_SHARE_READ|FILE_SHARE_WRITE,NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if( fileHandle != INVALID_HANDLE_VALUE ) {
		if( v_pSizeLow != NULL ) {
			*v_pSizeLow = ::GetFileSize( fileHandle, v_pSizeHi );
		}
		::CloseHandle( fileHandle );
		return true;
	}
	return false;
}

void WatchTask::EnumFile( const bool v_enableEvent )
{
	// 폜ۗ̊؂ACȅ
	sweepDeletedEntries();

	// t@C̑
	if( EnumFileRecursive( settingInfo_.getWatchDir(), 0, v_enableEvent ) ) {
		// 폜ΏۃACe̐ݒ
		// JnfBNg̃t@C񋓂ꍇ̂ݍ폜ҋ@XgZbg܂B
		prepareDeleteEntries();
	}
}

void WatchTask::prepareDeleteEntries()
{
	const DWORD now = ::GetTickCount();
	deletePendingList_.clear();
	for( FileInfoMap::iterator ite_fileInfo = fileInfoMap_.begin();
		ite_fileInfo != fileInfoMap_.end(); )
	{
		const tstring& absolutePath = ite_fileInfo->first;
		FileInfo& fileInfo = ite_fileInfo->second;
		if( ! fileInfo.isExist() ) {
			if( settingInfo_.getDeletePending() > 0 ) {
				// 폜ҋ@
				fileInfo.setDeletePending( true );
				fileInfo.setDeleteBeginTickCount( now );
				deletePendingList_.push_back( absolutePath );
			}
			else {
				// 폜ҋ@Ȃ
				ite_fileInfo = fileInfoMap_.erase( ite_fileInfo );
				continue;
			}
		}
		++ite_fileInfo;
	}
}

void WatchTask::sweepDeletedEntries()
{
	unsigned int pendingCount = 0;

	const DWORD now = ::GetTickCount();
	const DWORD pendingLimit = (DWORD) ( settingInfo_.getDeletePending() * 100 );

	for( FileNameList::iterator ite_pendingList = deletePendingList_.begin();
		ite_pendingList != deletePendingList_.end(); )
	{
		const tstring& absoluteName = *ite_pendingList;
		FileInfoMap::iterator ite_fileInfo = fileInfoMap_.find( absoluteName );
		
		bool removed = false;
		if( ite_fileInfo == fileInfoMap_.end() ) {
			assert( false && "폜ҋ@XgɁAt@CGg}bvɑ݂Ȃt@Cw肳Ă܂B" );
			removed = true;
		}
		else {
			FileInfo& fileInfo = ite_fileInfo->second;
			if( fileInfo.isExist() ) {
				// t@Ĉ݂ō폜ҋ@Xg珜
				removed = true;
			}
			else if( fileInfo.isDeletePending() ) {
				if( ( settingInfo_.getDeletePending() <= 0 ) ||
					( ( now - fileInfo.getDeleteBeginTickCount() ) > pendingLimit )	)
				{
					// 폜ҋ@ΏۂłA폜Aw莞Ԃo߂Ă̂
					// t@CGg}bvƁA폜ҋ@Xg̑o폜B
					fileInfoMap_.erase( ite_fileInfo );
					removed = true;
				}
			}
		}

		if( removed ) {
			ite_pendingList = deletePendingList_.erase( ite_pendingList );
			continue;
		}
		++ite_pendingList;
	}
}

bool WatchTask::EnumFileRecursive( const tstring& v_baseDir, const int v_depth, const bool v_enableEvent )
{
	// ΏۃpX𐶐
	const tstring baseDir = tstringuty::ensureEndsWith( v_baseDir, tstring( _TEXT("\\") ) );
	
	// ύXƎvt@C
	WIN32_FIND_DATA finddata = { 0 };
	HANDLE findHandle = ::FindFirstFile( ( baseDir + _TEXT("*.*") ).c_str(), &finddata );
	if( findHandle == INVALID_HANDLE_VALUE ) {
		// ΏۂɃANZXs\ȂΉȂŖ߂B
		return false;
	}

	if( v_depth == 0 ) {
		// ݂Ȃt@C폜邽߁AׂẴt@C݃tOZbgB
		// (lbg[N̈ꎞIȐؒf̉\̂߁AΏۂɃANZXłꍇ̂ݎsB)
		for( FileInfoMap::iterator ite = fileInfoMap_.begin();
			ite != fileInfoMap_.end();
			++ite)
		{
			FileInfo& fileInfo = ite->second;
			fileInfo.setExist( false );
		}
	}

	const DWORD now = ::GetTickCount();
	for(;;) {
		const tstring absolutePath = baseDir + finddata.cFileName;
		if( finddata.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) {
			if( ! IsMatch( finddata.cFileName, _TEXT(".") ) && ! IsMatch( finddata.cFileName, _TEXT("..") ) ) {
				if( v_depth < settingInfo_.getMaxDepth() ) {
					EnumFileRecursive( absolutePath, v_depth + 1, v_enableEvent ); // TufBNǧs̗L͕s
				}
			}
		}
		else {
			if( nameFilter_.isMatch( finddata.cFileName ) ) {
				FileInfoMap::iterator ite = fileInfoMap_.find( absolutePath );
				if( ite != fileInfoMap_.end() ) {
					FileInfo& fileInfo = ite->second;
					if( v_enableEvent && ! fileTimeComparator_( finddata.ftLastWriteTime, fileInfo.getLastModified() ) ) {
						fileInfo.setDetectModified( true );
						fileInfo.setCountdownTickCount( now );
					}
					fileInfo.update( finddata );
					fileInfo.setExist( true );
					fileInfo.setDeletePending( false );
				}
				else {
					FileInfo fileInfo( finddata, v_enableEvent, true );
					fileInfo.setDeletePending( false );
					fileInfo.setCountdownTickCount( now );
					fileInfoMap_.insert( FileInfoMap::value_type( absolutePath, fileInfo ) );
				}
			}
		}
		if( ! ::FindNextFile( findHandle, &finddata ) ) {
			::FindClose( findHandle );
			break;
		}
	}

	return true;
}


void WatchTask::onThreadBinded()
{
	// Xg̏(ʒmȂ)
	EnumFile( false );
}

void WatchTask::onThreadUnbinded()
{
	// Xg̃NA
	fileInfoMap_.clear();
}

void WatchTask::onThreadSignal()
{
	enumerationSignal_.setSignalDelay( ::GetTickCount() );
	::OutputDebugString( _TEXT("signal: dirchange notification\n") );
	dirChangeNotification_.next();
}

void WatchTask::onThreadTick()
{
	// ݎ̎擾
	const DWORD now = ::GetTickCount();

	// q[v̈k(s)
	if( garbageCollectTimer_.isSignaled( now ) ) {
		garbageCollectTimer_.reset( now );
		sweepDeletedEntries();
		_heapmin();
	}

	// ANV̎s
	pActionInvoker_->sweepTerminatedProcess();
	const DWORD writeWaitLimit = (DWORD)( settingInfo_.getWaitWrite() * 100 );
	for( FileInfoMap::iterator ite = fileInfoMap_.begin();
		ite != fileInfoMap_.end();
		++ite)
	{
		if( settingInfo_.getMaxProcess() != 0 && pActionInvoker_->sweepTerminatedProcess() >= settingInfo_.getMaxProcess() ) {
			// vZXEɒB̂ŌssȂB
			break;
		}
		const tstring& absolutePath = ite->first;
		FileInfo& fileInfo = ite->second;
		if( fileInfo.isDetectModified() ) {
			const DWORD span = now - fileInfo.getCountdownTickCount();
			if( span >= writeWaitLimit ) {
				DWORD nFileSizeLow;
				if( ! getFileSize( absolutePath, &nFileSizeLow ) ||	( fileInfo.getSize() != nFileSizeLow ) ) {
					// t@CJȂAt@CTCYύXĂ΃JEĝ蒼
					fileInfo.setSize( nFileSizeLow );
					fileInfo.setCountdownTickCount( now );
				}
				else{
					fileInfo.setDetectModified( false );
					::OutputDebugString( _TEXT("do action\n") );
					pActionInvoker_->createProcess( absolutePath, fileInfo );
				}
			}
		}
	}

	// t@C̃gK

	if( forceEnumerationTimer_.isSignaled( now ) ) {
		// XVw肳ĂAŌɌĂw莞Ԃo߂Ă邽ߋsB
		forceEnumerationTimer_.reset( now );
		enumerationSignal_.setSignalImmediate();
		::OutputDebugString( _TEXT("signal: force interval\n") );
	}

	if( enumerationSignal_.isSignaled( now ) ) {
		enumerationSignal_.clear();
		forceEnumerationTimer_.reset( now );
		::OutputDebugString( _TEXT("begin Enumeration\n") );
		EnumFile(true);
	}
}

HANDLE WatchTask::getWaitableHandle()
{
	return dirChangeNotification_.getHandle();
}
