#include "stdafx.hpp"

#include "FileInfo.hpp"

#include <memory>

#include "tstringUty.hpp"
#include "XMLWriter.hpp"
#include "SAXParser.hpp"
#include "DateFormatter.hpp"

namespace 
{
	class PersistentSAXHandler : public SAXContentHandler
	{
	private:
		FileInfoMap& fileInfoMap_;
		int level_;
		bool acceptDocument_;
		bool acceptFileInfo_;

		tstring path_;
		FileInfo fileInfo_;

		static CFileTime convertStringToFileTime( const tstring& v_tmp ) throw()
		{
			SYSTEMTIME dtTmp = { 0 };
			_stscanf( v_tmp.c_str(), _TEXT("%hd/%hd/%hd %hd:%hd:%hd.%hd"),
				&dtTmp.wYear,
				&dtTmp.wMonth,
				&dtTmp.wDay,
				&dtTmp.wHour,
				&dtTmp.wMinute,
				&dtTmp.wSecond,
				&dtTmp.wMilliseconds
				);
			FILETIME lft = { 0 };
			FILETIME ft = { 0 };
			SystemTimeToFileTime( &dtTmp, &lft );
			LocalFileTimeToFileTime( &lft, &ft );
			return CFileTime( ft );
		}

	public:
		PersistentSAXHandler( FileInfoMap& v_fileInfoMap )
			: fileInfoMap_( v_fileInfoMap )
			, level_( 0 )
			, acceptDocument_( false )
			, acceptFileInfo_( false )
		{
		}

		virtual ~PersistentSAXHandler()
		{
		}

		virtual void startDocument()
		{
			fileInfoMap_.clear();
		}

		virtual void endDocument()
		{
		}

		virtual void startPrefixMapping(const tstring&, const tstring& )
		{
		}

		virtual void endPrefixMapping(const tstring& )
		{
		}

		virtual void startElement(
			const tstring& v_namespace,
			const tstring& v_localName,
			const tstring& v_QName,
			const SAXAttributes& attr
			)
		{
			switch( level_++ )
			{
			case 0:
				if( v_QName == _TEXT("watch-files") ) {
					acceptDocument_ = true;
				}
				break;
			case 1:
				if( acceptDocument_ && v_QName == _TEXT("file") ) {
					const tstring path = attr.getValueFromQName( _TEXT("path") );
					const unsigned long long size = tstringuty::atoi64( attr.getValueFromQName( _TEXT("size") ) );
					const CFileTime lastModified( convertStringToFileTime( attr.getValueFromQName( _TEXT("lastModified") ) ) );

					path_ = path;
					fileInfo_ = FileInfo();
					fileInfo_.setSize( size );
					fileInfo_.setLastModified( lastModified );
					fileInfo_.setExist( true );
					acceptFileInfo_ = true;
				}
				break;
			case 2:
				if( acceptDocument_ && v_QName == _TEXT("detectModified") ) {
					const CFileTime actionPendingStartTime( convertStringToFileTime( attr.getValueFromQName( _TEXT("beginTime") ) ) );
					fileInfo_.setDetectModified( true );
					fileInfo_.setActionPendingStartTime( actionPendingStartTime );
				}
				else if( acceptDocument_ && v_QName == _TEXT("deletePending") ) {
					const CFileTime deletePendingStartTime( convertStringToFileTime( attr.getValueFromQName( _TEXT("beginTime") ) ) );
					fileInfo_.setDeletePending( true );
					fileInfo_.setDeletePendingStartTime( deletePendingStartTime );
				}
				break;
			default:
				break;
			}
		}

		virtual void endElement(
			const tstring& v_namespace,
			const tstring& v_localName,
			const tstring& v_QName
			)
		{
			switch( --level_ )
			{
			case 0:
				break;
			case 1:
				if( acceptDocument_ && v_QName == _TEXT("file") ) {
					if( acceptFileInfo_ ) {
						fileInfoMap_.insert( FileInfoMap::value_type( path_, fileInfo_ ) );
						acceptFileInfo_ = false;
					}
				}
				break;
			case 2:
				break;
			case 3:
				break;
			default:
				break;
			}
		}

		virtual void characters(const tstring& v_chars )
		{
		}

		virtual void ignorableWhitespace(const tstring& v_chars )
		{
		}

		virtual void processingInstruction( const tstring& , const tstring& )
		{
		}

		virtual void skippedEntity( const tstring& )
		{
		}
	};
}

FileInfo::FileInfo()
	: writetime_( 0 )
	, size_( 0 )
	, bExist_( false )
	, detectModified_( false )
	, actionPendingStartTime_( 0 )
	, deletePending_( false )
	, deletePendingStartTime_( 0 )
{
}

FileInfo::FileInfo( const FileInfo& v_other )
	: writetime_( v_other.writetime_ )
	, size_( v_other.size_ )
	, bExist_( v_other.bExist_ )
	, detectModified_( v_other.detectModified_ )
	, actionPendingStartTime_( v_other.actionPendingStartTime_ )
	, deletePending_( v_other.deletePending_ )
	, deletePendingStartTime_( v_other.deletePendingStartTime_ )
{
}

FileInfo& FileInfo::operator=( const FileInfo& v_other )
{
	if( this == &v_other ) {
		return *this;
	}

	writetime_ = v_other.writetime_;
	size_ = v_other.size_;
	bExist_ = v_other.bExist_;
	detectModified_ = v_other.detectModified_;
	actionPendingStartTime_ = v_other.actionPendingStartTime_;
	deletePending_ = v_other.deletePending_;
	deletePendingStartTime_ = v_other.deletePendingStartTime_;

	return *this;
}

FileInfo::~FileInfo()
{
}

void FileInfo::update( const WIN32_FIND_DATA& v_finddata )
{
	writetime_ = v_finddata.ftLastWriteTime;
	size_ = ( v_finddata.nFileSizeHigh << 32 ) | v_finddata.nFileSizeLow;
}

void FileInfo::setExist( bool v_exist )
{
	bExist_ = v_exist;
}

void FileInfo::setDeletePending( bool v_deletePending ) 
{
	deletePending_ = v_deletePending;
}

void FileInfo::setDeletePendingStartTime( const CFileTime& v_deletePendingStartTime )
{
	deletePendingStartTime_ = v_deletePendingStartTime;
}

void FileInfo::setDetectModified( bool v_detectModified )
{
	detectModified_ = v_detectModified;
}

void FileInfo::setLastModified( const CFileTime& v_filetime )
{
	writetime_ = v_filetime;
}

void FileInfo::setActionPendingStartTime( const CFileTime& v_actionPendingStartTime )
{
	actionPendingStartTime_ = v_actionPendingStartTime;
}

void FileInfo::setSize( unsigned long long v_size )
{
	size_ = v_size;
}

////

FileInfoMap::FileInfoMap()
{
}

FileInfoMap::~FileInfoMap()
{
}

bool FileInfoMap::loadPersistence( const tstring& v_path )
{
	if( v_path.empty() ) {
		return false;
	}

	// t@CɃANZXłȂꍇA܂݂͑ȂꍇfalseԂB
	const DWORD fileAttr = ::GetFileAttributes( v_path.c_str() );
	if( fileAttr == (DWORD) -1 ) {
		return false;
	}

	SAXParserFactory parserFactory;
	SAXParserPtr pParser = parserFactory.create();
	try{
		pParser->parse( v_path, PersistentSAXHandler( *this ) );
		return true;
	}
	catch( const std::exception& ) {
		//TODO: G[O
		return false;
	}
}

bool FileInfoMap::savePersistence( const tstring& v_path )
{
	if( v_path.empty() ) {
		return true;
	}

	const ConfigurableDateFormatter dateFormatter( _TEXT("yyyy/MM/dd HH:mm:ss.SSS") );

	try{
		XMLWriterFactory writerFactory;
		XMLWriterPtr writer = writerFactory.create( v_path );
		SAXAttributesHolderPtr pAttr = writer->createAttributesHolder();

		writer->startDocument();

		pAttr->clear();
		writer->startElement( _TEXT(""), _TEXT("watch-files"), _TEXT("watch-files"), *pAttr );
		for( const_iterator ite_fileInfoMap = begin();
			ite_fileInfoMap != end();
			++ite_fileInfoMap
			)
		{
			const tstring& fileName = ite_fileInfoMap->first;
			const FileInfo& fileInfo = ite_fileInfoMap->second;

			if( fileInfo.isExist() ) {
				pAttr->clear();
				pAttr->addAttribute( _TEXT(""), _TEXT("path"), _TEXT("path"), _TEXT("ENTITY"), fileName );

				SYSTEMTIME dtTmp = { 0 };
				::FileTimeToSystemTime( &(FILETIME)fileInfo.getLastModified().UTCToLocal(), &dtTmp );
				pAttr->addAttribute( _TEXT(""), _TEXT("lastModified"), _TEXT("lastModified"), _TEXT("ENTITY"), dateFormatter.format( dtTmp ) );
				pAttr->addAttribute( _TEXT(""), _TEXT("size"), _TEXT("size"), _TEXT("ENTITY"), tstringuty::ToString( _TEXT("%I64d"), fileInfo.getSize() ) );
				writer->startElement( _TEXT(""), _TEXT("file"), _TEXT("file"), *pAttr );

				if( fileInfo.isDetectModified() ) {
					pAttr->clear();
					::FileTimeToSystemTime( &(FILETIME)fileInfo.getActionPendingStartTime().UTCToLocal(), &dtTmp );
					pAttr->addAttribute( _TEXT(""), _TEXT("beginTime"), _TEXT("beginTime"), _TEXT("ENTITY"), dateFormatter.format( dtTmp ) );
					writer->startElement( _TEXT(""), _TEXT("detectModified"), _TEXT("detectModified"), *pAttr );
					writer->endElement( _TEXT(""), _TEXT("detectModified"), _TEXT("detectModified") );
				}
				if( fileInfo.isDeletePending() ) {
					pAttr->clear();
					::FileTimeToSystemTime( &(FILETIME)fileInfo.getDeletePendingStartTime().UTCToLocal(), &dtTmp );
					pAttr->addAttribute( _TEXT(""), _TEXT("beginTime"), _TEXT("beginTime"), _TEXT("ENTITY"), dateFormatter.format( dtTmp ) );
					writer->startElement( _TEXT(""), _TEXT("deletePending"), _TEXT("deletePending"), *pAttr );
					writer->endElement( _TEXT(""), _TEXT("deletePending"), _TEXT("deletePending") );
				}

				writer->endElement( _TEXT(""), _TEXT("file"), _TEXT("file") );
			}
		}
		writer->endElement( _TEXT(""), _TEXT("watch-files"), _TEXT("watch-files") );

		writer->endDocument();
		writer.reset();

		return true;
	}
	catch( const std::exception& ) {
		//TODO: G[O
		return false;
	}
}
