#include "stdafx.hpp"
#include "resource.h"

#include "FWatchApp.hpp"

#include "tstringUty.hpp"

#include "SettingInfoFactory.hpp"
#include "WatchTaskGroup.hpp"

#include <assert.h>

#include <fstream>
#include <exception>

//////////////////////////////////////////////

WatchTaskGroup::WatchTaskGroup( const bool v_dryrun )
	: modified_( false )
	, dryrun_( v_dryrun )
	, multiTask_( ThreadGroupProfile() )
{
}

void WatchTaskGroup::clear()
{
	multiTask_.resetSchedule();

	// ׂẴ^XNjB
	for( WatchTaskList::iterator ite = taskList_.begin();
		ite != taskList_.end();
		++ite )
	{
		WatchTask* pWatchTask = *ite;
		delete pWatchTask;
	}
	taskList_.clear();

	modified_ = false;
}

WatchTaskGroup::~WatchTaskGroup()
{
	clear();
}

bool WatchTaskGroup::isDryRun() const
{
	return dryrun_;
}

namespace
{
	class SettingInfoFactoryAdaptor
	{
	private:
		CSettingInfoFactory& factory_;
	public:
		SettingInfoFactoryAdaptor( CSettingInfoFactory& v_factory )
			: factory_( v_factory )
		{
		}

		void setWatchDir( const tstring& arg ) {
			factory_.setWatchDir( arg );
		}

		void setWatchFile( const tstring& arg ) {
			factory_.setWatchFile( arg );
		}

		void setAction( const tstring& arg ) {
			factory_.setAction( arg );
		}

		void setAppName( const tstring& arg ) {
			factory_.setAppName( arg );
		}

		void setParam( const tstring& arg ) {
			factory_.setParam( arg );
		}

		void setMaxProcess( const tstring& arg ) {
			factory_.setMaxProcess( tstringuty::atoi( arg ) );
		}

		void setShowWindow( const tstring& arg ) {
			factory_.setShowWindow( ShowWindowType::valueOf( tstringuty::atoi( arg ) ) );
		}

		void setWaitWrite( const tstring& arg ) {
			factory_.setWaitWrite( tstringuty::atoi( arg ) );
		}

		void setTolerance( const tstring& arg ) {
			factory_.setTolerance( tstringuty::atoi( arg ) );
		}

		void setLog( const tstring& arg ) {
			factory_.setLog( arg );
		}

		void setMaxDepth( const tstring& arg ) {
			factory_.setMaxDepth( tstringuty::atoi( arg ) );
		}

		void setUsingDirNotificationAPI( const tstring& arg ) {
			factory_.setUsingDirNotificationAPI( tstringuty::atoi( arg ) != 0 );
		}

		void setForceInterval( const tstring& arg ) {
			factory_.setForceInterval( tstringuty::atoi( arg ) );
		}

		void setDeletePending( const tstring& arg ) {
			factory_.setDeletePending( tstringuty::atoi( arg ) );
		}

		void setDirNotificationAPIExpirySpan( const tstring& arg ) {
			factory_.setDirNotificationAPIExpirySpan( tstringuty::atoi( arg ) );
		}

		void setDirNotificationAPIRetryInterval( const tstring& arg ) {
			factory_.setDirNotificationAPIRetryInterval( tstringuty::atoi( arg ) );
		}

		void setDelaySignalSpan( const tstring& arg ) {
			factory_.setDelaySignalSpan( tstringuty::atoi( arg ) );
		}

		void setAppCurrentDir( const tstring& arg ) {
			factory_.setAppCurrentDir( arg );
		}

		void setPersistentFile( const tstring& arg ) {
			factory_.setPersistentFile( arg );
		}

		void setPersistenceDelaySpan( const tstring& arg ) {
			factory_.setPersistenceDelaySpan( tstringuty::atoi( arg ) );
		}

		typedef void (SettingInfoFactoryAdaptor::*adaptorFunc)( const tstring& );

		static const SettingInfoFactoryAdaptor::adaptorFunc inputListFunctions[]; 
	};


	const SettingInfoFactoryAdaptor::adaptorFunc SettingInfoFactoryAdaptor::inputListFunctions[] = {
		SettingInfoFactoryAdaptor::setWatchDir,
		SettingInfoFactoryAdaptor::setWatchFile,
		SettingInfoFactoryAdaptor::setAction,
		SettingInfoFactoryAdaptor::setAppName,
		SettingInfoFactoryAdaptor::setParam,
		SettingInfoFactoryAdaptor::setMaxProcess,
		SettingInfoFactoryAdaptor::setShowWindow,
		SettingInfoFactoryAdaptor::setWaitWrite,
		SettingInfoFactoryAdaptor::setTolerance,
		SettingInfoFactoryAdaptor::setDeletePending,
		SettingInfoFactoryAdaptor::setLog,
		SettingInfoFactoryAdaptor::setMaxDepth,
		SettingInfoFactoryAdaptor::setUsingDirNotificationAPI,
		SettingInfoFactoryAdaptor::setForceInterval,
		SettingInfoFactoryAdaptor::setDirNotificationAPIExpirySpan,
		SettingInfoFactoryAdaptor::setDirNotificationAPIRetryInterval,
		SettingInfoFactoryAdaptor::setDelaySignalSpan,
		SettingInfoFactoryAdaptor::setAppCurrentDir,
		SettingInfoFactoryAdaptor::setPersistentFile,
		SettingInfoFactoryAdaptor::setPersistenceDelaySpan,
		NULL
	};
}

void WatchTaskGroup::startWatch()
{
	//required:
	assert( ! multiTask_.isScheduling() && "Xbh͂łɎsłB" );

	//do:
	if( ! dryrun_ ) {
		multiTask_.startSchedule();
	}
}

void WatchTaskGroup::stopWatch()
{
	if( ! dryrun_ ) {
		multiTask_.stopSchedule();
	}
}

bool WatchTaskGroup::LoadInfData( const tstring& InfPath )
{
	//required:
	assert( ! multiTask_.isScheduling() && "XbhsłB" );

	//do:
	clear();
	if( InfPath.empty() ) {
		return false;
	}

	bool succeeded = false;
	try{
		std::ifstream fin;
		std::string infPath( tstringuty::getStdString( InfPath ) );
		fin.open( infPath.c_str() );
		if( ! fin.is_open() ) {
			return false;
		}

		const CSettingInfo defaultParameter;

		std::string line;
		while( std::getline( fin, line ) ) {
			::OutputDebugString( tstringuty::getTString( line ).c_str() );

			CSettingInfoFactory factory( defaultParameter );
			SettingInfoFactoryAdaptor adaptor( factory );

			int col = 0;
			std::string::size_type lastpos = 0;
			for(;;) {
				const std::string::size_type pos = line.find( '\t', lastpos );
				std::string arg;
				if( pos != tstring::npos ) {
					arg = line.substr( lastpos, pos - lastpos );
				}
				else {
					arg = line.substr( lastpos );
				}
				
				SettingInfoFactoryAdaptor::adaptorFunc func = SettingInfoFactoryAdaptor::inputListFunctions[ col ];
				if( func == NULL ) {
					break;
				}

				(adaptor.*func)( tstringuty::getTString( arg ) );

				col++;
				lastpos = pos + 1;
				if( pos == tstring::npos ) {
					break;
				}
			}

			append( factory.create() );
		}
		fin.close();
		succeeded = true;
	}
	catch( const std::exception& v_exception ) {
		pApp->showMessageBox( MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_INFLOADFAILED, tstringuty::getTString( v_exception.what() ).c_str() );
	}

	modified_ = false;
	return succeeded;
}

bool WatchTaskGroup::isModified() const
{
	return modified_;
}

bool WatchTaskGroup::SaveInfData( const tstring& InfPath, bool v_bForce ) const
{
	if( InfPath.empty() ) {
		return false;
	}

	if( ! isModified() && ! v_bForce ) {
		return true;
	}

	bool succeeded = false;
	try{
		std::ofstream fout;
		fout.exceptions( std::ios::failbit | std::ios::badbit );

		std::string infPath( tstringuty::getStdString( InfPath ) );
		fout.open( infPath.c_str() );
		if( ! fout.is_open() ) {
			return false;
		}
		for( WatchTaskList::const_iterator ite = taskList_.begin();
			ite != taskList_.end();
			++ite )
		{
			const WatchTask* pWatchThread = *ite;
			const CSettingInfo& settingInfo = pWatchThread->getSettingInfo();

			fout << tstringuty::getStdString( settingInfo.getWatchDir()	)		<< "\t";
			fout << tstringuty::getStdString( settingInfo.getWatchFile() )		<< "\t";
			fout << tstringuty::getStdString( settingInfo.getAction() )			<< "\t";
			fout << tstringuty::getStdString( settingInfo.getAppName() )		<< "\t";
			fout << tstringuty::getStdString( settingInfo.getParam() )			<< "\t";
			fout << settingInfo.getMaxProcess()									<< "\t";
			fout << settingInfo.getShowWindow()									<< "\t";
			fout << settingInfo.getWaitWrite().GetTimeSpan()					<< "\t";
			fout << settingInfo.getTolerance().GetTimeSpan()					<< "\t";
			fout << settingInfo.getDeletePending().GetTimeSpan()				<< "\t";
			fout << tstringuty::getStdString( settingInfo.getLog() )			<< "\t";
			fout << settingInfo.getMaxDepth()									<< "\t";
			fout << ( settingInfo.isUsingDirNotificationAPI() ? 1 : 0 )			<< "\t";
			fout << settingInfo.getForceInterval().GetTimeSpan()				<< "\t";
			fout << settingInfo.getDirNotificationAPIExpirySpan().GetTimeSpan()		<< "\t";
			fout << settingInfo.getDirNotificationAPIRetryInterval().GetTimeSpan()	<< "\t";
			fout << settingInfo.getDelaySignalSpan().GetTimeSpan()				<< "\t";
			fout << tstringuty::getStdString( settingInfo.getAppCurrentDir() )	<< "\t";
			fout << tstringuty::getStdString( settingInfo.getPersistentFile() )	<< "\t";
			fout << settingInfo.getPersistenceDelaySpan().GetTimeSpan()			<< std::endl;
		}

		modified_ = false;
		succeeded = true;
	}
	catch( const std::exception& v_exception ) {
		pApp->showMessageBox( MB_ICONERROR | MB_OK, NULL, IDS_MAINPROC_INFSAVEFAILED, tstringuty::getTString( v_exception.what() ).c_str() );
	}

	return succeeded;
}

unsigned int WatchTaskGroup::size() const
{
	return (unsigned int) taskList_.size();
}

const CSettingInfo& WatchTaskGroup::getInfo( unsigned int idx ) const
{
	//required:
	assert( idx >= 0 && idx < taskList_.size() && "݂ȂACeɂ̓ANZXł܂B" );
	
	//do:
	return taskList_[ idx ]->getSettingInfo();
}

bool WatchTaskGroup::append( const CSettingInfo& v_settingInfo )
{
	WatchTask* pWatchTask = new WatchTask( v_settingInfo );
	taskList_.push_back( pWatchTask );
	bool result = true;
	if( ! dryrun_ ) {
		result = multiTask_.appendTask( pWatchTask, false );
	}
	modified_ = true;
	assert( result && "^XN̒ǉɎs܂B" );
	return result;
}

bool WatchTaskGroup::replace( unsigned int idx, const CSettingInfo& v_settingInfo )
{
	//required:
	assert( idx >= 0 && idx < taskList_.size() && "݂ȂACeɂ̓ANZXł܂B" );
	
	//do:

	WatchTask* pOldWatchTask = taskList_[ idx ];
	multiTask_.removeTask( pOldWatchTask );
	delete pOldWatchTask;

	WatchTask *const pWatchTask = new WatchTask( v_settingInfo );
	taskList_[ idx ] = pWatchTask;

	bool result = true;
	if( ! dryrun_ ) {
		result = multiTask_.appendTask( pWatchTask, false );
	}
	modified_ = true;
	assert( result && "^XN̍ւɎs܂B" );
	return result;
}

bool WatchTaskGroup::erase(unsigned int idx)
{
	//required:
	assert( idx >= 0 && idx < taskList_.size() && "͈͊OłB" );

	//do:
	WatchTaskList::iterator ite = taskList_.begin() + idx;
	WatchTask* pWatchTask = *ite;
	const bool result = multiTask_.removeTask( pWatchTask );
	delete pWatchTask;
	taskList_.erase( ite );

	modified_ = true;
	assert( result && "^XN̍폜Ɏs܂B" );
	return result;
}

bool WatchTaskGroup::isRunning() const
{
	return multiTask_.isScheduling();
}
