#include "stdafx.hpp"

#include "DateFormatter.hpp"

#include <assert.h>

LiteralDateFormatter::LiteralDateFormatter( const tstring& v_literal )
	: literal_( v_literal )
{
}

LiteralDateFormatter::~LiteralDateFormatter() throw()
{
	
}

tstring LiteralDateFormatter::format( const SYSTEMTIME& v_systime ) const throw()
{
	return literal_;
}

// 

PartDateFormatter::PartDateFormatter( DATEFORMATITEM v_formatType, int v_length )
	: formatType_( v_formatType )
	, length_( v_length )
{
}

PartDateFormatter::~PartDateFormatter() throw()
{
}

tstring PartDateFormatter::format( const SYSTEMTIME& v_systime ) const throw()
{
	static const TCHAR* const WEEKS[] =
	{
		_TEXT("SUN"),
		_TEXT("MON"),
		_TEXT("TUE"),
		_TEXT("WED"),
		_TEXT("THU"),
		_TEXT("FRI"),
		_TEXT("SAT")
	};

	TCHAR tmp[ 64 ];
	switch( formatType_ )
	{
	case DATEFORMATITEM_YEAR:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wYear );
		break;
	case DATEFORMATITEM_MONTH:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wMonth );
		break;
	case DATEFORMATITEM_DAY:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wDay );
		break;
	case DATEFORMATITEM_HOUR24:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wHour );
		break;
	case DATEFORMATITEM_HOUR12:
		_stprintf( tmp, _TEXT("%0*d"), length_, ( v_systime.wHour >= 12 ) ? ( v_systime.wHour - 12 ) : v_systime.wHour );
		break;
	case DATEFORMATITEM_AMPM:
		_tcscpy( tmp, ( v_systime.wHour >= 12 ) ? _TEXT("PM") : _TEXT("AM") );
		break;
	case DATEFORMATITEM_MINUTE:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wMinute );
		break;
	case DATEFORMATITEM_SECOND:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wSecond );
		break;
	case DATEFORMATITEM_MILLSECOND:
		_stprintf( tmp, _TEXT("%0*d"), length_, v_systime.wMilliseconds );
		break;
	case DATEFORMATITEM_DAYOFWEEK:
		assert( v_systime.wDayOfWeek >= 0 && v_systime.wDayOfWeek <= sizeof( WEEKS ) && "j͈̔͂0`6ł͂܂B" );
		_tcscpy(tmp, WEEKS[ v_systime.wDayOfWeek ] );
		break;
	default:
		_tcscpy( tmp, _TEXT("*") );
		break;
	}

	return tmp;
}

//

ConfigurableDateFormatter::ConfigurableDateFormatter( const tstring& v_format )
	: containDateFormatter_( false )
{
	try{
		parse( v_format );
	}
	catch(...){
		clearFormat();
		throw;
	}
}

ConfigurableDateFormatter::~ConfigurableDateFormatter() throw()
{
	clearFormat();
}

void ConfigurableDateFormatter::clearFormat()
{
	for( FormatItemList::iterator ite_items = itemList_.begin();
		ite_items != itemList_.end();
		++ite_items)
	{
		DateFormatter* pFormatter = *ite_items;
		delete pFormatter;
	}
	itemList_.clear();
	containDateFormatter_ = false;
}

tstring ConfigurableDateFormatter::format( const SYSTEMTIME& v_systime ) const throw()
{
	tstring result;
	for( FormatItemList::const_iterator ite_items = itemList_.begin();
		ite_items != itemList_.end();
		++ite_items)
	{
		DateFormatter* pFormatter = *ite_items;
		result += pFormatter->format( v_systime );
	}
	return result;
}

void ConfigurableDateFormatter::parse( const tstring& v_format ) throw()
{
	tstring buf;
	TCHAR ctlch = 0;
	int ctlcnt = 0;

	enum STATE {
		STATE_NORMAL = 0,
		STATE_MULTIBYTE = 1,
		STATE_CONTROL = 2,
		STATE_ESCAPE_HEAD = 3,
		STATE_ESCAPE = 4,
		STATE_ESCAPE_MULTIBYTE = 5,
		STATE_FINISH = 6
	} state = STATE_NORMAL;

	tstring::const_iterator ite_format = v_format.begin();
	for(;;) {
		TCHAR ch = 0;
		if( ite_format != v_format.end() ) {
			ch = *ite_format;
		}

		if( state == STATE_NORMAL ) {
			bool changeMode = false;
			
			if( ch == 0 ) {
				state = STATE_FINISH;
				changeMode = true;
			}
			else if( ch == 'y' || ch == 'M' || ch == 'd' || ch == 'H' || ch == 'a' || ch == 'h' || ch == 'm' || ch == 's' || ch == 'S' || ch == 'E' ) {
				ctlch = ch;
				ctlcnt = 0;
				containDateFormatter_ = true;
				state = STATE_CONTROL;
				changeMode = true;
			}
			else if( ch == '\'' ) {
				state = STATE_ESCAPE_HEAD;
			}
			else {
				buf += ch;
#ifndef _UNICODE
				if( isleadbyte( (int) ch & 0xff ) ) {
					state = STATE_MULTIBYTE;
				}
#endif
			}

			if( changeMode ) {
				if( buf.size() > 0 ) {
					itemList_.push_back( new LiteralDateFormatter( buf ) );
				}
				buf.clear();
			}
		}
		else if( state == STATE_MULTIBYTE ) {
			buf += ch;
			state = STATE_NORMAL;
		}
		else if( state == STATE_CONTROL ) {
			ctlcnt++;
			if( ch != ctlch ) {
				PartDateFormatter::DATEFORMATITEM formatType = PartDateFormatter::DATEFORMATITEM_YEAR;
				switch( ctlch )
				{
				case 'y':
					formatType = PartDateFormatter::DATEFORMATITEM_YEAR;
					break;
				case 'M':
					formatType = PartDateFormatter::DATEFORMATITEM_MONTH;
					break;
				case 'd':
					formatType = PartDateFormatter::DATEFORMATITEM_DAY;
					break;
				case 'a':
					formatType = PartDateFormatter::DATEFORMATITEM_AMPM;
					break;
				case 'h':
					formatType = PartDateFormatter::DATEFORMATITEM_HOUR12;
					break;
				case 'H':
					formatType = PartDateFormatter::DATEFORMATITEM_HOUR24;
					break;
				case 'm':
					formatType = PartDateFormatter::DATEFORMATITEM_MINUTE;
					break;
				case 's':
					formatType = PartDateFormatter::DATEFORMATITEM_SECOND;
					break;
				case 'S':
					formatType = PartDateFormatter::DATEFORMATITEM_MILLSECOND;
					break;
				case 'E':
					formatType = PartDateFormatter::DATEFORMATITEM_DAYOFWEEK;
					break;
				}
				itemList_.push_back( new PartDateFormatter( formatType, ctlcnt ) );

				state = STATE_NORMAL;
				continue;
			}
		}
		else if( state == STATE_ESCAPE_HEAD ) { 
			buf += ch;
			if( ch == 0 || ch == '\'' ) {
				state = STATE_NORMAL;
			}
			else {
				state = STATE_ESCAPE;
			}
		}
		else if( state == STATE_ESCAPE ) {
			if( ch == 0 || ch == '\'' ) {
				state = STATE_NORMAL;
			}
			else {
				buf += ch;
#ifndef _UNICODE
				if( isleadbyte( (int) ch & 0xff ) ) {
					state = STATE_ESCAPE_MULTIBYTE;
				}
#endif
			}
		}
		else if( state == STATE_ESCAPE_MULTIBYTE ) {
			if( ch == 0 ) {
				state = STATE_NORMAL;
			}
			else {
				buf += ch;
				state = STATE_ESCAPE;
			}
		}
		else if( state == STATE_FINISH ) {
			break;
		}

		if( ite_format != v_format.end() ) {
			++ite_format;
		}
	}
}

bool ConfigurableDateFormatter::isContainDateFormat() const throw()
{
	return containDateFormatter_;
}
