#include "stdafx.hpp"

#include "DirWatchObject.hpp"

#include "tstringUty.hpp"

#include <assert.h>

DirWatchObject::DirWatchObject(
	const tstring& v_watchDir,
	bool v_enableSubDir,
	const CFileTimeSpan& v_retrySpan,
	const CFileTimeSpan& v_expirySpan,
	bool v_enableAPI
	)
	: watchDir_( tstringuty::ensureEndsWith( v_watchDir, _TEXT("\\") ) )
	, enableSubDir_( v_enableSubDir )
	, enableAPI_( v_enableAPI )
	, hWatchHandle_( NULL )
	, expiryTimer_( v_expirySpan, 0, v_expirySpan > 0 )
	, retryTimer_( v_retrySpan, 0, false )
{
}

DirWatchObject::~DirWatchObject()
{
	if( hWatchHandle_ != NULL ) {
		::FindCloseChangeNotification( hWatchHandle_ );
	}
}

HANDLE DirWatchObject::getHandle()
{
	if( enableAPI_ ) {
		const CFileTime now = CFileTime::GetCurrentTime();
		if( hWatchHandle_ == NULL || expiryTimer_.isSignaled( now ) ) {
			if( ! retryTimer_.isEnabled() || retryTimer_.isSignaled( now ) ) {
				notifyEvent( DirWatchObjectListener::DIRWATCHOBJECT_CREATING );
				HANDLE watchHandle = ::FindFirstChangeNotification(
					watchDir_.c_str(),
					enableSubDir_,
					FILE_NOTIFY_CHANGE_FILE_NAME |
					FILE_NOTIFY_CHANGE_DIR_NAME |
					FILE_NOTIFY_CHANGE_LAST_WRITE |
					FILE_NOTIFY_CHANGE_SIZE |
					FILE_NOTIFY_CHANGE_ATTRIBUTES
					);
				if( watchHandle != INVALID_HANDLE_VALUE ) {
					notifyEvent( DirWatchObjectListener::DIRWATCHOBJECT_CREATED);
					if( hWatchHandle_ != NULL ) {
						// ÂnhjB
						::FindCloseChangeNotification( hWatchHandle_ );
					}
					hWatchHandle_ = watchHandle;
					expiryTimer_.reset( now );
					retryTimer_.setEnable( false );
				}
				else {
					notifyEvent( DirWatchObjectListener::DIRWATCHOBJECT_CREATE_FAILED);
					retryTimer_.reset( now );
					retryTimer_.setEnable( true );
				}
			}
		}
	}
	return hWatchHandle_;
}

void DirWatchObject::next()
{
	if( hWatchHandle_ != NULL ) {
		if( ! ::FindNextChangeNotification( hWatchHandle_ ) ) {
			// ̑ҋ@Ɏs = nh̉\邽߁Ač쐬ׂłB
			::FindCloseChangeNotification( hWatchHandle_ );
			hWatchHandle_ = NULL;
		}
		else {
			const CFileTime now = CFileTime::GetCurrentTime();
			expiryTimer_.reset( now );
		}
	}
}

void DirWatchObject::expireCreationRetrySpan()
{
	retryTimer_.setSignaled();
}

void DirWatchObject::appendListener( DirWatchObjectListener* const v_listener )
{
	//required:
	assert( v_listener != NULL );

	//do:
	listeners_.push_back( v_listener );
}

void DirWatchObject::removeListener( DirWatchObjectListener* const v_listener )
{
	//required:
	assert( v_listener != NULL );

	//do:
	for( DirWatchObjectListenerList::iterator ite = listeners_.begin();
		ite != listeners_.end(); )
	{
		DirWatchObjectListener* const listener = *ite;
		assert( listener != NULL );
		if( listener == v_listener ) {
			ite = listeners_.erase( ite );
			continue;
		}
		++ite;
	}
}

void DirWatchObject::notifyEvent( DirWatchObjectListener::DirWatchObjectEventType v_event )
{
	for( DirWatchObjectListenerList::iterator ite = listeners_.begin(), last = listeners_.end();
		ite != last;
		++ite )
	{
		DirWatchObjectListener* const listener = *ite;
		assert( listener != NULL );
		listener->notifyDirWatchObjectEvent( v_event );
	}
}


