#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gehl_types.h"
#include "gehl_bmp_def.h"
#include "debug_print.h"
#include "endian.h"


static int
GEHL_BMP_Load_8(const char *fname,
                GEHL_IMG *img,
                GEHL_BMP_FH *bfh,
                GEHL_BMP_IH *bih);

static int
GEHL_BMP_Load_24(const char *fname,
                 GEHL_IMG *img,
                 GEHL_BMP_FH *bfh,
                 GEHL_BMP_IH *bih);

static int
GetRowbytes(int bits, int width);

static int
SetParameters(GEHL_BMP_FH *bfh, GEHL_BMP_IH *bih, GEHL_IMG *img);

#ifdef __BIG_ENDIAN__
static int
ReverseEndian(GEHL_BMP_FH *bfh, GEHL_BMP_IH *bih);
#endif /* __BIG_ENDIAN__ */


/*
 * BMP ν
 */
int
GEHL_BMP_GetInfo(const char *fname, GEHL_IMG *img)
{
    FILE *fp;
    GEHL_BMP_FH bfh;
    GEHL_BMP_IH bih;
    int ret;

    memset(&bfh, 0, sizeof(GEHL_BMP_FH));
    memset(&bih, 0, sizeof(GEHL_BMP_IH));

    fp = fopen(fname, "rb");
    if (NULL==fp){
        return -1;
    }

    fread(&(bfh.bfType),  1, sizeof(GEHL_BMP_FH)-2, fp);
    fread(&bih, 1, sizeof(GEHL_BMP_IH), fp);
    fclose(fp);

    ret = SetParameters(&bfh, &bih, img);

    return 0;
}

/*
 * BMP ɤ
 */
int
GEHL_BMP_Load(const char *fname, GEHL_IMG *img)
{
    FILE *fp;
    GEHL_BMP_FH bfh;
    GEHL_BMP_IH bih;
    int ret = -1;

    memset(&bfh, 0, sizeof(GEHL_BMP_FH));
    memset(&bih, 0, sizeof(GEHL_BMP_IH));

    fp = fopen(fname, "rb");
    if (NULL==fp){
        return -1;
    }

    fread(&(bfh.bfType), 1, sizeof(GEHL_BMP_FH)-2, fp);
    fread(&bih, 1, sizeof(GEHL_BMP_IH), fp);

    ret = SetParameters(&bfh, &bih, img);
    fclose(fp);
    if (ret != 0)
        return ret;

    /* ̤ˤб */
    if (bih.biCompression != 0)
        return -1;

    switch (img->bits) {

    case 1:
    case 8:
        ret = GEHL_BMP_Load_8(fname, img, &bfh, &bih);
        break;

    case 24:
        ret = GEHL_BMP_Load_24(fname, img, &bfh, &bih);
        break;

    default:
        break;
    }

    return ret;
}

/*
 * 8bitʲBMPɤ߹ߴؿ
 * (palette)
 * :
 *   fname --- filename
 *   bfh --- Bitmap File Header
 *   bih --- Bitmap Info Header
 * :
 *   img --- GEHL᡼
 */
static int
GEHL_BMP_Load_8(const char *fname,
                GEHL_IMG *img,
                GEHL_BMP_FH *bfh,
                GEHL_BMP_IH *bih)
{
    FILE *fp;
    int i, x, y;
    int npal, rowbytes;
    unsigned char *pdata, swp;
    unsigned char *dest, *buf;
    GEHL_PAL *ppal;
    const unsigned char bitmask[] = {
        0x80, 0x40, 0x20, 0x10,
        0x8, 0x4, 0x2, 0x1 };

    /* img ɬפʥѥ᡼򥻥åȤ */
    img->width = img->rowbytes = bih->biWidth;
    img->height = bih->biHeight;
    img->dpi = (int)( (double)bih->biXPixPerMeter / 0.0254 );
    img->bits = bih->biBitCount;
    img->black = 0;

    //rowbytes = bih->biSizeImage / bih->biHeight;
    rowbytes = GetRowbytes(img->bits, img->width);
    /* ե륪ץ */
    fp = fopen(fname, "rb");
    if (NULL == fp)
        return -1;

    /* ѥåȤ򥻥åȤ */
    for (i = 0, npal = 1; i < img->bits; i++)
        npal *= 2;
    ppal = (GEHL_PAL *)malloc(sizeof(GEHL_BMP_RGBQUAD) * npal);
    if (NULL == ppal) {
        fclose(fp);
        return -1;
    }

    fseek(fp, sizeof(GEHL_BMP_FH)-2 + sizeof(GEHL_BMP_IH), SEEK_SET);
    for (i = 0; i < npal; i++) {
        fread(ppal+i, sizeof(GEHL_BMP_RGBQUAD), 1, fp);
        swp = ppal[i].blue;
        ppal[i].blue = ppal[i].red;
        ppal[i].red = swp;
    }
    img->ppalette = ppal;

    /* ǡѤΥݤ */
    pdata = (unsigned char *)malloc(img->width * img->height);
    if (NULL == pdata) {
        free(ppal);
        fclose(fp);
        return -1;
    }
    img->pdata = img->pred = img->pgreen = img->pblue = pdata;

    /* ǡɤ߹ */
    switch (img->bits) {

    case 1:
        buf = (unsigned char *)malloc(rowbytes);
        if (NULL == buf) {
            free(ppal);
            free(pdata);
            fclose(fp);
            return -1;
        }

        for (y = img->height - 1; y >= 0; y--) {
            dest = pdata + img->width * y;
            fread(buf, rowbytes, 1, fp);
            for (x = 0; x < img->width; x++) {
                dest[x] = ((buf[x/8] & bitmask[x%8]) == bitmask[x%8]) ? 1 : 0;
            }
        }
        free(buf);
        break;

    case 8:
        for (y = img->height - 1; y >= 0; y--) {
            dest = pdata + img->width * y;
            fread(dest, img->width, 1, fp);
            fseek(fp, rowbytes-img->width, SEEK_CUR);
        }
        break;

    default:
        free(ppal);
        free(pdata);
        fclose(fp);
        return -1;
        break;
    }

    fclose(fp);
    return 0;
}

/*
 * 9bitʾBMPɤ߹ߴؿ
 * (palette̵)
 * :
 *   fname --- filename
 *   bfh --- Bitmap File Header
 *   bih --- Bitmap Info Header
 * :
 *   img --- GEHL᡼
 */
static int
GEHL_BMP_Load_24(const char *fname,
                 GEHL_IMG *img,
                 GEHL_BMP_FH *bfh,
                 GEHL_BMP_IH *bih)
{
    FILE *fp;
    int x, y;
    int rowbytes;
    unsigned char *pdata;
    unsigned char *pr, *pg, *pb, *buf;


    /* img ɬפʥѥ᡼򥻥åȤ */
    img->width = img->rowbytes = bih->biWidth;
    img->height = bih->biHeight;
    img->dpi = (int)( (double)bih->biXPixPerMeter / 0.0254 );
    img->bits = bih->biBitCount;
    img->black = 0;
    img->ppalette = NULL;
    img->pdata = NULL;

    //rowbytes = bih->biSizeImage / bih->biHeight;
    rowbytes = GetRowbytes(img->bits, img->width);

    /* ե륪ץ */
    fp = fopen(fname, "rb");
    if (NULL == fp)
        return -1;
    fseek(fp, bfh->bfOffBits, SEEK_SET);

    /* ǡѤΥݤ */
    pdata = (unsigned char *)malloc(3 * img->height * img->width);
    if (NULL == pdata) {
        fclose(fp);
        return -1;
    }
    img->pdata = img->pred = pdata;
    img->pgreen = img->pred + img->width * img->height;
    img->pblue = img->pgreen + img->width * img->height;

    /* ǡɤ߹ */
    switch (img->bits) {

    case 24:
        buf = (unsigned char *)malloc(rowbytes);
        if (NULL == buf) {
            free(pdata);
            fclose(fp);
            return -1;
        }

        for (y = img->height - 1; y >= 0; y--) {
            pr = img->pred + img->width * y;
            pg = img->pgreen + img->width * y;
            pb = img->pblue + img->width * y;

            fread(buf, rowbytes, 1, fp);

            for (x = 0; x < img->width; x++) {
                pb[x] = buf[x * 3];
                pg[x] = buf[x * 3 + 1];
                pr[x] = buf[x * 3 + 2];
            }
        }

        free(buf);
        break;

    default:
        free(pdata);
        fclose(fp);
        return -1;
        break;
    }

    fclose(fp);
    return 0;
}

/*
 * BMP ¸
 */
int
GEHL_BMP_Save(const char *fname, const GEHL_IMG *img)
{
    GEHL_BMP_FH bfh;
    GEHL_BMP_IH bih;
    FILE *fp;
    int ret;
    int fhsize;
    int rowbytes, npad;
    unsigned char pad[] = {0, 0, 0};
    int i, x, y;
    char *src;
    int pal, palsize;
    GEHL_PAL *palette, tmp;
    int isMakePalette = 0;
    unsigned char *buf, swp;
    unsigned char *pr, *pg, *pb;
    const unsigned char bitmask[] = {
        0x80, 0x40, 0x20, 0x10,
        0x8, 0x4, 0x2, 0x1 };

    if ((NULL==fname)||(NULL==img)){
        return -1;
    }

    fhsize = sizeof(GEHL_BMP_FH)-2;

    pal = 0; palsize = 0;
    
    /* set Palette */
    switch (img->bits) {

    case 1:
        pal = 2;
        palsize = pal * sizeof(GEHL_PAL);
        
        if (img->ppalette == NULL) {
            palette = (GEHL_PAL *)malloc(palsize);
            if (NULL == palette) return -1;
            memset(palette, 0, palsize);
            if (img->black == 0) {
                palette[1].blue = 255;
                palette[1].green = 255;
                palette[1].red = 255;
            } else {
                palette[0].blue = 255;
                palette[0].green = 255;
                palette[0].red = 255;
            }
            isMakePalette = 1;
        } else {
            palette = img->ppalette;
        }
        break;

    case 8:
        pal = 256; palsize = pal * sizeof(GEHL_PAL);
        if (NULL == img->ppalette) {
            palette = (GEHL_PAL *)malloc(palsize);
            if (NULL == palette) return -1;
            memset(palette, 0, palsize);
            if (img->black == 0) {
                for(i = 0; i < pal; i++) {
                    palette[i].blue = i;
                    palette[i].green = i;
                    palette[i].red = i;
                }
            } else {
                for(i = 0; i < pal; i++) {
                    palette[i].blue = 255-i;
                    palette[i].green = 255-i;
                    palette[i].red = 255-i;
                }
            }
            isMakePalette = 1;
        } else {
            palette = img->ppalette;
        }
        break;

    default:
        break;
    }

    memset(&bfh, 0, sizeof(GEHL_BMP_FH));
    memset(&bih, 0, sizeof(GEHL_BMP_IH));

    /* set Bitmap File Header */
    rowbytes = GetRowbytes(img->bits, img->width);
    bfh.bfType = GEHL_BMP_TYPE;
    bfh.bfSize = fhsize + sizeof(GEHL_BMP_IH)
        + palsize + img->height * rowbytes;
    bfh.bfOffBits = fhsize + sizeof(GEHL_BMP_IH) + palsize;

    /* set Bitmap Info Header */
    bih.biSize = sizeof(GEHL_BMP_IH);
    bih.biWidth = img->width;
    bih.biHeight = img->height;
    bih.biPlanes = 1;
    bih.biBitCount = img->bits;
    bih.biSizeImage = rowbytes * img->height;
    bih.biXPixPerMeter = (long)((double)img->dpi*0.0254);
    bih.biYPixPerMeter = (long)((double)img->dpi*0.0254);

    fp = fopen(fname, "wb");
    if (NULL==fp){
        return -1;
    }

#ifdef __BIG_ENDIAN__
    ReverseEndian(&bfh, &bih);
#endif  

    ret = fwrite(&(bfh.bfType), fhsize, 1, fp);
    ret = fwrite(&bih, sizeof(bih), 1, fp);
    if (pal != 0){
        for (i = 0; i < pal; i++) {
            tmp = palette[i];
            swp = tmp.blue;
            tmp.blue = tmp.red;
            tmp.red = swp;
            fwrite(&tmp, sizeof(GEHL_PAL), 1, fp);
        }
    }

    switch (img->bits) {

    case 1:
        buf = (unsigned char *)malloc(rowbytes);
        if (NULL == buf)
            break;
        memset(buf, 0, rowbytes);

        for (y = img->height-1; y >= 0; y--) {
            src = img->pdata + img->width * y;
            for (x = 0; x < img->width; x++) {
                buf[x/8] |= (src[x]) ? bitmask[x%8] : 0;
            }
            fwrite(buf, rowbytes, 1, fp);
            memset(buf, 0, rowbytes);
        }

        free(buf);
        break;

    case 8:
        npad = rowbytes - img->width;
        for(y = img->height - 1; y >= 0; y--){
            src = img->pdata + img->width * y;
            ret = fwrite(src, img->width, 1, fp);
            ret = fwrite(pad, npad, 1, fp);
        }
        break;

    case 24:
        buf = (unsigned char *)malloc(rowbytes);
        if (NULL == buf)
            break;
        memset(buf, 0, rowbytes);

        for (y = img->height - 1; y >= 0; y--) {
            pr = img->pred + img->width * y;
            pg = img->pgreen + img->width * y;
            pb = img->pblue + img->width * y;
            for (x = 0; x < img->width; x++) {
                buf[x*3] = pb[x];
                buf[x*3+1] = pg[x];
                buf[x*3+2] = pr[x];
            }
            fwrite(buf, rowbytes, 1, fp);
        }
        free(buf);
        break;

    default:
        break;
    }

    if (isMakePalette) {
        free(palette);
    }
    fclose(fp);

    fprintf(stderr, "written w:%d h:%d rb:%d bit:%d total:%d\n",
            img->width,
            img->height,
            rowbytes,
            img->bits,
            img->height * rowbytes);

    return 0;
}


/*
 * BMP 
 * (ޤ̣ʤ)
 */
int
GEHL_BMP_Free(GEHL_IMG *img)
{
    if (NULL==img){
        return -1;
    }

    if (NULL!=img->ppalette){
        free(img->ppalette);
    }

    if (NULL!=img->pdata){
        free(img->pdata);
    } else {
        return -1;
    }

    return 0;
}


/*
 * BMP ΣԤΥХȿ׻
 *
 * :
 *   bits --- color depth
 *   width --- width (pixel)
 * :
 *   rowbytes
 */
static int
GetRowbytes(int bits, int width)
{
    int rowbytes;

    switch (bits){

    case 1:
        rowbytes = ((width+31)/32)*4;
        break;

    case 8:
        rowbytes = ((width+3)/4)*4;
        break;

    case 24:
        rowbytes = ((width*3+3)/4)*4;
        break;

    default:
        rowbytes = -1;
    }

    return rowbytes;
}


/*
 * إåѥ᡼
 * ¤Τ˥åȤ
 */

/* 28, July 2003 ɵ
   إåʥǡޤޤƤθ
   ʤ٤饤֥ȼ˥׻褦
   ɬפ */
static int
SetParameters(GEHL_BMP_FH *bfh, GEHL_BMP_IH *bih, GEHL_IMG *img)
{
    if ((NULL==img)||(NULL==bfh)||(NULL==bih)){
        return -1;
    }

#ifdef __BIG_ENDIAN__    
    ReverseEndian(bfh, bih);
#endif

    if (bfh->bfType != GEHL_BMP_TYPE){
        return -1;
    }
    
    img->width = bih->biWidth;
    /* biSizeImage׻ */
    bih->biSizeImage = bfh->bfSize - bfh->bfOffBits;
    img->rowbytes = bih->biSizeImage/bih->biHeight;
    img->height = bih->biHeight;
    img->dpi = 72; /* 塹ȵ褦ˤͽ */
    img->bits = bih->biBitCount;
    img->black = 0;
    img->ppalette = NULL;
    img->pdata = NULL;

    if ((img->bits != 1)&&(img->bits != 8)&&(img->bits != 24)){
        return -1;
    }

    return 0;
}


/*
 * ¤ΤΥǥž
 * Big Endian ĶΤ߻
 */
#ifdef __BIG_ENDIAN__

static int
ReverseEndian(GEHL_BMP_FH *bfh, GEHL_BMP_IH *bih)
{
    Reverse16(&(bfh->dummy));
    Reverse16(&(bfh->bfType));
    Reverse32(&(bfh->bfSize));
    Reverse16(&(bfh->bfReserved1));
    Reverse16(&(bfh->bfReserved2));
    Reverse32(&(bfh->bfOffBits));

    Reverse32(&(bih->biSize));
    Reverse32(&(bih->biWidth));
    Reverse32(&(bih->biHeight));
    Reverse16(&(bih->biPlanes));
    Reverse16(&(bih->biBitCount));
    Reverse32(&(bih->biCompression));
    Reverse32(&(bih->biSizeImage));
    Reverse32(&(bih->biXPixPerMeter));
    Reverse32(&(bih->biYPixPerMeter));
    Reverse32(&(bih->biClrUsed));
    Reverse32(&(bih->biClrImportant));

    return 0;
}

#endif /* __BIG_ENDIAN__ */
