/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/* 
 * Copyright (C) 2001-2002 the xine project
 * Copyright (C) 2002 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gimv_xine_0_9.c,v 1.1.2.1 2003/04/05 18:01:40 makeinu Exp $
 *
 * the xine engine in a widget - implementation
 */

#include "gimv_xine.h"

#ifdef ENABLE_XINE

#include <xine.h>

#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION <= 13)

#include "gimv_xine_priv.h"


enum {
   PLAY_SIGNAL,
   STOP_SIGNAL,
   PLAYBACK_FINISHED_SIGNAL,
   NEED_NEXT_MRL_SIGNAL,
   BRANCHED_SIGNAL,
   LAST_SIGNAL
};


/* missing stuff from X includes */
#ifndef XShmGetEventBase
extern int XShmGetEventBase(Display *);
#endif

static void gimv_xine_class_init    (GimvXineClass   *klass);
static void gimv_xine_init          (GimvXine        *gxine);

/* object class methods */
static void gimv_xine_destroy       (GtkObject      *object);

/* widget class methods */
static void gimv_xine_realize       (GtkWidget      *widget);
static void gimv_xine_unrealize     (GtkWidget      *widget);
static gint gimv_xine_expose        (GtkWidget      *widget,
                                     GdkEventExpose *event);
static void gimv_xine_size_allocate (GtkWidget      *widget, 
                                     GtkAllocation  *allocation);

static GtkWidgetClass *parent_class = NULL;
static gint gimv_xine_signals[LAST_SIGNAL] = {0};


GtkType
gimv_xine_get_type (void)
{
   static GtkType gimv_xine_type = 0;

#if (GTK_MAJOR_VERSION >= 2)
   if (!gimv_xine_type) {
      static const GTypeInfo gimv_xine_info = {
         sizeof (GimvXineClass),
         NULL,               /* base_init */
         NULL,               /* base_finalize */
         (GClassInitFunc)    gimv_xine_class_init,
         NULL,               /* class_finalize */
         NULL,               /* class_data */
         sizeof (GimvXine),
         0,                  /* n_preallocs */
         (GInstanceInitFunc) gimv_xine_init,
      };

      gimv_xine_type = g_type_register_static (GTK_TYPE_WIDGET,
                                               "GimvXine",
                                               &gimv_xine_info,
                                               0);
   }
#else /* (GTK_MAJOR_VERSION >= 2) */
   if (!gimv_xine_type) {
      static const GtkTypeInfo gimv_xine_info = {
         "GimvXine",
         sizeof (GimvXine),
         sizeof (GimvXineClass),
         (GtkClassInitFunc) gimv_xine_class_init,
         (GtkObjectInitFunc) gimv_xine_init,
         /* reserved_1 */ NULL,
         /* reserved_2 */ NULL,
         (GtkClassInitFunc) NULL,
      };
    
      gimv_xine_type = gtk_type_unique (gtk_widget_get_type (), &gimv_xine_info);
   }
#endif /* (GTK_MAJOR_VERSION >= 2) */

   return gimv_xine_type;
}


static void
gimv_xine_class_init (GimvXineClass *class)
{
   GtkObjectClass *object_class;
   GtkWidgetClass *widget_class;

   object_class = (GtkObjectClass *) class;
   widget_class = (GtkWidgetClass *) class;

   parent_class = gtk_type_class (gtk_widget_get_type ());

#if (defined USE_GTK2) && (defined GTK_DISABLE_DEPRECATED)
   gimv_xine_signals[PLAY_SIGNAL]
      = g_signal_new ("play",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GimvXineClass, play),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

   gimv_xine_signals[STOP_SIGNAL]
      = g_signal_new ("stop",
                      G_SIGNAL_RUN_FIRST,
                      G_TYPE_FROM_CLASS (object_class),
                      G_STRUCT_OFFSET (GimvXineClass, stop),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

   gimv_xine_signals[PLAYBACK_FINISHED_SIGNAL]
      = g_signal_new ("playback_finished",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GimvXineClass, playback_finished),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

   gimv_xine_signals[NEED_NEXT_MRL_SIGNAL]
      = g_signal_new ("need_next_mrl",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GimvXineClass, need_next_mrl),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__POINTER,
                      G_TYPE_NONE, 1, G_TYPE_POINTER);

   gimv_xine_signals[BRANCHED_SIGNAL]
      = g_signal_new ("branched",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GimvXineClass, branched),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);
#else /* (defined USE_GTK2) && (defined GTK_DISABLE_DEPRECATED) */
   gimv_xine_signals[PLAY_SIGNAL]
      = gtk_signal_new ("play",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (GimvXineClass, play),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   gimv_xine_signals[STOP_SIGNAL]
      = gtk_signal_new ("stop",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (GimvXineClass, stop),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   gimv_xine_signals[PLAYBACK_FINISHED_SIGNAL]
      = gtk_signal_new ("playback_finished",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (GimvXineClass, playback_finished),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   gimv_xine_signals[NEED_NEXT_MRL_SIGNAL]
      = gtk_signal_new ("need_next_mrl",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (GimvXineClass, need_next_mrl),
                        gtk_marshal_NONE__POINTER,
                        GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

   gimv_xine_signals[BRANCHED_SIGNAL]
      = gtk_signal_new ("branched",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (GimvXineClass, branched),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   gtk_object_class_add_signals (object_class, gimv_xine_signals, LAST_SIGNAL);
#endif /* (defined USE_GTK2) && (defined GTK_DISABLE_DEPRECATED) */

   object_class->destroy       = gimv_xine_destroy;

   widget_class->realize = gimv_xine_realize;
   widget_class->unrealize = gimv_xine_unrealize;

   widget_class->size_allocate = gimv_xine_size_allocate;
  
   widget_class->expose_event = gimv_xine_expose;

   class->play                = NULL;
   class->stop                = NULL;
   class->playback_finished   = NULL;
   class->need_next_mrl       = NULL;
   class->branched            = NULL;
}


static void
gimv_xine_init (GimvXine *gxine)
{
   GimvXinePrivate *priv;

   gxine->widget.requisition.width  = 8;
   gxine->widget.requisition.height = 8;

   priv = gxine->private = g_new0 (GimvXinePrivate, 1);

   priv->mrl             = NULL;
   priv->config          = NULL;
   priv->xine            = NULL;
   priv->vo_driver       = NULL;
   priv->ao_driver       = NULL;
   priv->exit            = FALSE;
}


static void
gimv_xine_destroy (GtkObject *object)
{
   GimvXine *gtx = GIMV_XINE (object);
   GimvXinePrivate *priv;

   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;

   if (priv) {
      g_free (priv->mrl);
      g_free (gtx->private);
      gtx->private = NULL;
   }

   if (GTK_OBJECT_CLASS (parent_class)->destroy)
      GTK_OBJECT_CLASS (parent_class)->destroy (object);
}


static void
dest_size_cb (void *gxine_gen,
              int video_width, int video_height,
              int *dest_width, int *dest_height)
{
   GimvXine *gxine = (GimvXine *) gxine_gen;
   GimvXinePrivate *priv;

   g_return_if_fail (GIMV_IS_XINE (gxine));
   priv = gxine->private;

   *dest_width  = gxine->widget.allocation.width;
   *dest_height = gxine->widget.allocation.height;
}


static void
frame_output_cb (void *gxine_gen,
                 int video_width, int video_height, 
                 int *dest_x,     int *dest_y,
                 int *dest_width, int *dest_height
#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   )
#else
                 ,int *win_x,      int *win_y)
#endif
{
   GimvXine   *gxine = (GimvXine *) gxine_gen;
   GimvXinePrivate *priv;

   g_return_if_fail (GIMV_IS_XINE (gxine));
   priv = gxine->private;

   *dest_x = 0;
   *dest_y = 0;

#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   *dest_width  = gxine->widget.allocation.width;
   *dest_height = gxine->widget.allocation.height;
#else
   if (GTK_WIDGET_TOPLEVEL (&gxine->widget)) {
      gdk_window_get_position (gxine->widget.window, win_x, win_y);
   } else {
      GdkWindow *window;

      if (GTK_WIDGET_NO_WINDOW (&gxine->widget)) {
         window = gxine->widget.window;
     } else {
         window = gdk_window_get_parent (gxine->widget.window);
      }

      gdk_window_get_position (window, win_x, win_y);

      *win_x += gxine->widget.allocation.x;
      *win_y += gxine->widget.allocation.y;
   }

   *dest_width  = gxine->widget.allocation.width;
   *dest_height = gxine->widget.allocation.height;
#endif
}


static xine_vo_driver_t *
load_video_out_driver(GimvXine *this)
{
   GimvXinePrivate *priv;
   double res_h, res_v;
   x11_visual_t vis;
   char **driver_ids;
   int i;
   char *video_driver_id;
   xine_vo_driver_t *vo_driver;

   g_return_val_if_fail (GIMV_IS_XINE (this), NULL);
   priv = this->private;

   vis.display       = priv->display;
   vis.screen        = priv->screen;
   vis.d             = priv->video_window;
   res_h             = (DisplayWidth  (priv->display, priv->screen)*1000 
                          / DisplayWidthMM (priv->display, priv->screen));
   res_v             = (DisplayHeight (priv->display, priv->screen)*1000
                          / DisplayHeightMM (priv->display, priv->screen));
   vis.display_ratio = res_h / res_v;

   if (fabs(vis.display_ratio - 1.0) < 0.01) {
      vis.display_ratio   = 1.0;
   }

#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   vis.calc_dest_size     = dest_size_cb;
   vis.request_dest_size  = frame_output_cb;
#else
   vis.dest_size_cb      = dest_size_cb;
   vis.frame_output_cb   = frame_output_cb;
#endif
   vis.user_data          = this;
  
   /* video output driver auto-probing */
   driver_ids = xine_list_video_output_plugins (VISUAL_TYPE_X11);

   /* try to init video with stored information */
   if (priv->video_driver_id)
      video_driver_id = priv->video_driver_id;
   else
      video_driver_id = priv->config->register_string (priv->config, 
                                                       "video.driver", "auto",
                                                       "video driver to use",
                                                       NULL, NULL, NULL);
   if (!strcmp (video_driver_id, "auto")) {
      vo_driver = xine_load_video_output_plugin(priv->config, 
                                                video_driver_id,
                                                VISUAL_TYPE_X11,
                                                (void *) &vis);
      if (vo_driver) {
         free(driver_ids);
         return vo_driver;
      }
   }
    
   i = 0;
   while (driver_ids[i]) {	
      video_driver_id = driver_ids[i];
	
      vo_driver = xine_load_video_output_plugin(priv->config, 
                                                video_driver_id,
                                                VISUAL_TYPE_X11, 
                                                (void *) &vis);
      if (vo_driver) {
         priv->config->update_string (priv->config,
                                      "video.driver",
                                      video_driver_id);
         return vo_driver;
      }
	
      i++;
   }
  
   return NULL;
}


static xine_ao_driver_t *
load_audio_out_driver(GimvXine *this)
{
   GimvXinePrivate *priv;
   xine_ao_driver_t *ao_driver=NULL;
   char        *audio_driver_id;
   char       **driver_ids = xine_list_audio_output_plugins ();
   int          i = 0;

   g_return_val_if_fail (GIMV_IS_XINE (this), NULL);
   priv = this->private;

   /* try to init audio with stored information */
   if (priv->audio_driver_id)
      audio_driver_id = priv->video_driver_id;
   else
      audio_driver_id = priv->config->register_string (priv->config,
                                                       "audio.driver", "auto",
                                                       "audio driver to use",
                                                       NULL, NULL, NULL);
  
   if (strcmp (audio_driver_id, "auto")) 
      return xine_load_audio_output_plugin(priv->config, 
                                           audio_driver_id);


   while (driver_ids[i]) {
      audio_driver_id = driver_ids[i];
    
      ao_driver = xine_load_audio_output_plugin(priv->config, 
                                                driver_ids[i]);

      if (ao_driver) {
         printf ("main: ...worked, using '%s' audio driver.\n",
                 driver_ids[i]);
      
         priv->config->update_string (priv->config, "audio.driver", 
                                      audio_driver_id);

         return ao_driver;
      }
      i++;
   }

   return ao_driver;
}


static void *
xine_thread (void *this_gen)
{
   GimvXine    *this   = (GimvXine *) this_gen;
   GimvXinePrivate *priv;

   g_return_val_if_fail (GIMV_IS_XINE (this), NULL);
   priv = this->private;

   while (1) {	
      XEvent xevent;
      gint status;

      if (priv->exit)
         break;

      status = xine_get_status (priv->xine);
      if (status != XINE_PLAY) {
         usleep (1000);
         continue;
      }

      XNextEvent (priv->display, &xevent);
 
      /* printf ("gtkxine: got an event (%d)\n", event.type);  */

      switch (xevent.type) {
      case Expose:
         if (xevent.xexpose.count != 0)
            break;

         priv->vo_driver->gui_data_exchange (priv->vo_driver, 
                                             GUI_DATA_EX_EXPOSE_EVENT, 
                                             &xevent);
         break;

#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
      case ConfigureNotify:
      {
         XConfigureEvent *ev = (XConfigureEvent *) &xevent;
         x11_rectangle_t area;
         area.x = 0;
         area.y = 0;
         area.w = ev->width;
         area.h = ev->height;
         priv->vo_driver->gui_data_exchange(priv->vo_driver,
                                            GUI_DATA_EX_DEST_POS_SIZE_CHANGED,
                                            &area);
         break;
      }
#endif

      case FocusOut:
         break;
      }

      if (xevent.type == priv->completion_event) {
         priv->vo_driver->gui_data_exchange (priv->vo_driver, 
                                             GUI_DATA_EX_COMPLETION_EVENT, 
                                             &xevent); 
         /* printf ("gtkxine: completion event\n"); */
      }
   }

   pthread_exit(NULL);
   return NULL;
}


void
event_listener (void *data, xine_event_t *event)
{
   GimvXine *gtx = GIMV_XINE (data);

   g_return_if_fail (GIMV_IS_XINE (gtx));
   g_return_if_fail (event);

   switch (event->type) {
   case XINE_EVENT_MOUSE_BUTTON:
      break;
   case XINE_EVENT_MOUSE_MOVE:
      break;
   case XINE_EVENT_SPU_BUTTON:
      break;
   case XINE_EVENT_SPU_CLUT:
      break;
   case XINE_EVENT_UI_CHANNELS_CHANGED:
      break;
   case XINE_EVENT_UI_SET_TITLE:
      g_print ("\"XINE_EVENT_UI_SET_TITLE\" signal\n");
      break;
   case XINE_EVENT_INPUT_MENU1:
      break;
   case XINE_EVENT_INPUT_MENU2:
      break;
   case XINE_EVENT_INPUT_MENU3:
      break;
   case XINE_EVENT_INPUT_UP:
      break;
   case XINE_EVENT_INPUT_DOWN:
      break;
   case XINE_EVENT_INPUT_LEFT:
      break;
   case XINE_EVENT_INPUT_RIGHT:
      break;
   case XINE_EVENT_INPUT_SELECT:
      break;
   case XINE_EVENT_PLAYBACK_FINISHED:
#ifdef USE_GTK2
      g_signal_emit (G_OBJECT(gtx),
                     gimv_xine_signals[PLAYBACK_FINISHED_SIGNAL], 0);
#else /* USE_GTK2 */
      gtk_signal_emit (GTK_OBJECT(gtx),
                       gimv_xine_signals[PLAYBACK_FINISHED_SIGNAL]);
#endif /* USE_GTK2 */
      break;
   case XINE_EVENT_BRANCHED:
      /* g_print ("\"XINE_EVENT_BRANCHED\" signal\n"); */
#ifdef USE_GTK2
      g_signal_emit (G_OBJECT(gtx),
                     gimv_xine_signals[BRANCHED_SIGNAL], 0);
#else /* USE_GTK2 */
      gtk_signal_emit (GTK_OBJECT(gtx),
                       gimv_xine_signals[BRANCHED_SIGNAL]);
#endif /* USE_GTK2 */
       break;
   case XINE_EVENT_NEED_NEXT_MRL:
   {
      xine_next_mrl_event_t *uevent = (xine_next_mrl_event_t *) event;
      gchar *next_mrl;

#ifdef USE_GTK2
      g_signal_emit (G_OBJECT(gtx),
                     gimv_xine_signals[NEED_NEXT_MRL_SIGNAL],
                     0,
                     &next_mrl);
#else /* USE_GTK2 */
      gtk_signal_emit (GTK_OBJECT(gtx),
                       gimv_xine_signals[NEED_NEXT_MRL_SIGNAL],
                       &next_mrl);
#endif /* USE_GTK2 */

      if (next_mrl && *next_mrl) {
         uevent->handled = 1;
         uevent->mrl = next_mrl;
      }
      break;
   }
   case XINE_EVENT_INPUT_NEXT:
      break;
   case XINE_EVENT_INPUT_PREVIOUS:
      break;
   case XINE_EVENT_INPUT_ANGLE_NEXT:
      break;
   case XINE_EVENT_INPUT_ANGLE_PREVIOUS:
      break;
   case XINE_EVENT_SPU_FORCEDISPLAY:
      break;
   case XINE_EVENT_FRAME_CHANGE:
      break;
   case XINE_EVENT_CLOSED_CAPTION:
      break;
#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   case XINE_EVENT_INPUT_BUTTON_FORCE:
      break;
#endif

   default:
      break;
   }
}


static void
gimv_xine_realize (GtkWidget *widget)
{
   GimvXine      *this;
   GimvXinePrivate *priv;
   Window        parent;
   GdkWindow    *toplevel;

   g_return_if_fail(widget);
   g_return_if_fail(GIMV_IS_XINE(widget));

   this = GIMV_XINE(widget);
   priv = this->private;

   /* set realized flag */

   GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

   /* create our own video window */
   parent = GDK_WINDOW_XWINDOW (gtk_widget_get_parent_window(widget));
   priv->video_window
      = XCreateSimpleWindow (gdk_display, 
                             parent,
                             0, 0, 
                             widget->allocation.width,
                             widget->allocation.height, 1,
                             BlackPixel(gdk_display, DefaultScreen(gdk_display)),
                             BlackPixel(gdk_display, DefaultScreen(gdk_display)));

   widget->window = gdk_window_foreign_new (priv->video_window);
   /*
   gdk_window_add_filter (widget->window, gimv_xine_event_filter, this);
   */

   toplevel = gdk_window_get_toplevel (gtk_widget_get_parent_window(widget));
   priv->toplevel = toplevel;

   if (!XInitThreads ()) {
      printf ("XInitThreads failed - looks like you don't have a thread-safe xlib.\n");
      return;
   }

   printf ("xine_thread: open display\n");

   priv->display = XOpenDisplay (NULL);

   if (!priv->display) {
      printf ("XOpenDisplay failed!\n");
      return;
   }

   XLockDisplay (priv->display);

   priv->screen = DefaultScreen (priv->display);

   if (XShmQueryExtension (priv->display) == True) {
      priv->completion_event = XShmGetEventBase (priv->display) + ShmCompletion;
   } else {
      priv->completion_event = -1;
   }

#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   XSelectInput (priv->display, priv->video_window,
                 /* StructureNotifyMask | */ ExposureMask
                 /* | ButtonPressMask | PointerMotionMask */);
#else
   XSelectInput (priv->display, priv->video_window,
                 StructureNotifyMask | ExposureMask
                 /* | ButtonPressMask | PointerMotionMask */);
#endif

   /* generate and init a config "object" */
   g_snprintf (priv->configfile, 255, "%s/.xine/config", getenv ("HOME"));
#if (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION < 9)
   priv->config = config_file_init (priv->configfile);
#else
   priv->config = xine_config_file_init (priv->configfile);
#endif

   /* load audio, video drivers */
   priv->vo_driver = load_video_out_driver(this);

   if (!priv->vo_driver) {
      printf ("couldn't open video driver\n");
      return ;
   }

   priv->ao_driver = load_audio_out_driver(this);

   /* init xine engine */
   priv->xine = xine_init (priv->vo_driver, priv->ao_driver,
                           priv->config);
   xine_set_locale();

   xine_register_event_listener (priv->xine, event_listener, this);

   XUnlockDisplay (priv->display);

   /* now, create a xine thread */
   pthread_create (&priv->thread, NULL, xine_thread, this);

   return;
}


static void
gimv_xine_unrealize (GtkWidget *widget)
{
   GimvXine* this;
   GimvXinePrivate *priv;

   g_return_if_fail(widget);
   g_return_if_fail(GIMV_IS_XINE(widget));

   this = GIMV_XINE (widget);
   priv = this->private;

   gimv_xine_stop (this);

   /* stop event thread */
   priv->exit = TRUE;
   XFlush (priv->display);

   /* save configuration */
   priv->config->save (priv->config);

   xine_exit (priv->xine);
   priv->xine = NULL;

   /* Hide all windows */
   if (GTK_WIDGET_MAPPED (widget))
      gtk_widget_unmap (widget);

   GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);

   /* This destroys widget->window and unsets the realized flag */
   if (GTK_WIDGET_CLASS(parent_class)->unrealize)
      (* GTK_WIDGET_CLASS(parent_class)->unrealize) (widget);
}


GtkWidget *
gimv_xine_new (const gchar *video_driver_id, const gchar *audio_driver_id)
{
#if (GTK_MAJOR_VERSION >= 2)
   GtkWidget *this = GTK_WIDGET (g_object_new (gimv_xine_get_type (), NULL));
#else /* (GTK_MAJOR_VERSION >= 2) */
   GtkWidget *this = GTK_WIDGET (gtk_type_new (gimv_xine_get_type ()));
#endif /* (GTK_MAJOR_VERSION >= 2) */
   GimvXinePrivate *priv;

   g_return_val_if_fail (GIMV_IS_XINE (this), NULL);
   priv = GIMV_XINE (this)->private;

   if (video_driver_id)
      priv->video_driver_id = strdup (video_driver_id);
   else
      priv->video_driver_id = NULL;

   if (audio_driver_id)
      priv->audio_driver_id = strdup (audio_driver_id);
   else
      priv->audio_driver_id = NULL;

   return this;
}


static gint
gimv_xine_expose (GtkWidget *widget, 
                 GdkEventExpose *event)
{
   /*
   GimvXine *this = GIMV_XINE (widget);  
   */

   return TRUE;
}


static void
gimv_xine_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{  
   GimvXine *this;

   g_return_if_fail (widget);
   g_return_if_fail(GIMV_IS_XINE(widget));

   this = GIMV_XINE(widget);

   widget->allocation = *allocation;

   if (GTK_WIDGET_REALIZED (widget)) {
      gdk_window_move_resize (widget->window,
                              allocation->x,
                              allocation->y,
                              allocation->width - 2,
                              allocation->height - 2);
   }
}


gint
gimv_xine_set_mrl (GimvXine *gtx, const gchar *mrl)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, FALSE);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), FALSE);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, FALSE);

   g_print ("gtkxine: calling xine_open, mrl = '%s'\n", mrl);

   g_free (priv->mrl);
   if (mrl && *mrl)
      priv->mrl = g_strdup (mrl);

   return TRUE;
}


gint
gimv_xine_play (GimvXine *gtx, gint pos, gint start_time)
{
   GimvXinePrivate *priv;
   gint retval;

   g_return_val_if_fail (gtx, -1);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), -1);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, -1);

   printf ("gtkxine: calling xine_play start_pos = %d, start_time = %d\n",
           pos, start_time);

   start_time /= 1000;

   retval = xine_play (priv->xine, (char *) priv->mrl, pos, start_time);

   if (retval)
#ifdef USE_GTK2
      g_signal_emit (G_OBJECT(gtx),
                     gimv_xine_signals[PLAY_SIGNAL], 0);
#else /* USE_GTK2 */
      gtk_signal_emit (GTK_OBJECT(gtx),
                       gimv_xine_signals[PLAY_SIGNAL]);
#endif /* USE_GTK2 */

   return retval; 
}


void
gimv_xine_set_speed (GimvXine *gtx, gint speed)
{
   GimvXinePrivate *priv;

   g_return_if_fail (gtx);
   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;
   g_return_if_fail (priv->xine);

   xine_set_speed (priv->xine, speed);
}


gint
gimv_xine_get_speed (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, GIMV_XINE_SPEED_NORMAL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), GIMV_XINE_SPEED_NORMAL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, GIMV_XINE_SPEED_NORMAL);

   return xine_get_speed (priv->xine);
}


void
gimv_xine_stop (GimvXine *gtx)
{
   GimvXinePrivate *priv;
   gint i;

   g_return_if_fail (gtx);
   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;
   g_return_if_fail (priv->xine);

   xine_stop (priv->xine);

   for (i = 0; i < 1; i++) {
      usleep (1000);
      XFlush (priv->display);
   }

#ifdef USE_GTK2
   g_signal_emit (G_OBJECT(gtx),
                  gimv_xine_signals[STOP_SIGNAL], 0);
#else /* USE_GTK2 */
   gtk_signal_emit (GTK_OBJECT(gtx),
                    gimv_xine_signals[STOP_SIGNAL]);
#endif /* USE_GTK2 */
}


gint
gimv_xine_get_position (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   if (gimv_xine_is_playing (gtx))
      return xine_get_current_position (priv->xine);
   else
      return 0;
}


void
gimv_xine_set_audio_channel (GimvXine *gtx, gint audio_channel)
{
   GimvXinePrivate *priv;

   g_return_if_fail (gtx);
   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;
   g_return_if_fail (priv->xine);

   xine_select_audio_channel (priv->xine, audio_channel);
}


gint
gimv_xine_get_audio_channel (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return xine_get_audio_selection (priv->xine);
}


gchar **
gimv_xine_get_autoplay_plugins (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_autoplay_input_plugin_ids (priv->xine);
}


gint
gimv_xine_get_current_time (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return xine_get_current_time (priv->xine) * 1000;
}


gint
gimv_xine_get_stream_length (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return xine_get_stream_length (priv->xine) * 1000;
}


gint
gimv_xine_is_playing (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   if (xine_get_status (priv->xine) == XINE_PLAY)
      return TRUE;
   else
      return FALSE;
}


gint
gimv_xine_is_seekable (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return xine_is_stream_seekable (priv->xine);
}


void
gimv_xine_save_config (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_if_fail (gtx);
   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;
   g_return_if_fail (priv->xine);

   priv->config->save (priv->config);
}


void
gimv_xine_set_video_property (GimvXine *gtx, 
                              gint property, 
                              gint value)
{
   GimvXinePrivate *priv;

   g_return_if_fail (gtx);
   g_return_if_fail (GIMV_IS_XINE (gtx));

   priv = gtx->private;
   g_return_if_fail (priv->xine);

   priv->vo_driver->set_property (priv->vo_driver, property, value);
}


gint
gimv_xine_get_video_property (GimvXine *gtx, 
                              gint property)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return priv->vo_driver->get_property (priv->vo_driver, property);
}


#if 0
gint
gimv_xine_get_log_section_count (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, 0);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), 0);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, 0);

   return xine_get_log_section_count (priv->xine);
}


gchar **
gimv_xine_get_log_names (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_log_names (priv->xine);
}
#endif


gchar **
gimv_xine_get_log (GimvXine *gtx,
                   gint buf)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_log (priv->xine, buf);
}


gchar **
gimv_xine_get_browsable_input_plugin_ids (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_browsable_input_plugin_ids (priv->xine);
}


gchar **
gimv_xine_get_autoplay_input_plugin_ids (GimvXine *gtx)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_autoplay_input_plugin_ids (priv->xine);
}


gchar **
gimv_xine_get_autoplay_mrls (GimvXine *gtx,
                             const gchar *plugin_id,
                             gint    *num_mrls)
{
   GimvXinePrivate *priv;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);

   return xine_get_autoplay_mrls (priv->xine, (char *) plugin_id, num_mrls);
}


guchar *
gimv_xine_get_current_frame_rgb (GimvXine *gtx, gint *width_ret, gint *height_ret)
{
   GimvXinePrivate *priv;
   gint err = 0;
   GimvXinePrivImage *image;
   guchar *rgb = NULL;

   g_return_val_if_fail (gtx, NULL);
   g_return_val_if_fail (GIMV_IS_XINE (gtx), NULL);

   priv = gtx->private;
   g_return_val_if_fail (priv->xine, NULL);
   g_return_val_if_fail (width_ret && height_ret, NULL);

   image = gimv_xine_priv_image_new (0);

   err = xine_get_current_frame(priv->xine,
                                &image->width, &image->height,
                                &image->ratio_code,
                                &image->format,
                                &image->y, &image->u, &image->v);

   if (err == 0) goto ERROR;

   /* the dxr3 driver does not allocate yuv buffers */
   /* image->u and image->v are always 0 for YUY2 */
   if (!image->y) goto ERROR;

   rgb = gimv_xine_priv_yuv2rgb (image);
   *width_ret  = image->width;
   *height_ret = image->height;

 ERROR:
   gimv_xine_priv_image_delete (image);
   return rgb;
}

#endif /* (XINE_MAJOR_VERSION == 0) && (XINE_MINOR_VERSION == 9) && (XINE_SUB_VERSION <= 13) */
#endif /* ENABLE_XINE */
