/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_info.h,v 1.26.2.6 2003/06/11 06:41:19 makeinu Exp $
 */

#ifndef __IMAGE_INFO_H__
#define __IMAGE_INFO_H__

#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"
#include "gimv_io.h"


typedef enum
{
   IMAGE_INFO_SYNCED_FLAG               = 1 << 0,
   IMAGE_INFO_DIR_FLAG                  = 1 << 1,
   IMAGE_INFO_ARCHIVE_FLAG              = 1 << 2,
   IMAGE_INFO_ARCHIVE_MEMBER_FLAG       = 1 << 3,
   IMAGE_INFO_ANIMATION_FLAG            = 1 << 5,
   IMAGE_INFO_VECTOR_FLAG               = 1 << 6,
   IMAGE_INFO_MOVIE_FLAG                = 1 << 7,
   IMAGE_INFO_URL_FLAG                  = 1 << 8,
   IMAGE_INFO_MRL_FLAG                  = 1 << 9    /* used by xine */
} ImageInfoFlags;


struct ImageInfo_Tag
{
   gchar          *filename;   /* If the file is a member of a archive,
                                  this value sholud be set as relative path. */
   const gchar    *format;
   struct stat     st;

   gint            width;      /* if unknown, value is -1  */
   gint            height;     /* same with above */
   gint            depth;
   /* ColorSpace   color_space; */

   ImageInfoFlags  flags;

   /* reference count */
   guint           ref_count;
};


#define IMAGE_INFO_IS_SYNCED(info) (info ? (info->flags & IMAGE_INFO_SYNCED_FLAG) : FALSE)


/*
 *  Get ImageInfo object for specified file. If file is not exist,
 *  allocate new ImageInfo.
 *  To get ImageInfo for archive member, simply treat archive as directory.
 */
ImageInfo      *image_info_get                  (const gchar   *filename); 
ImageInfo      *image_info_get_url              (const gchar   *url);

/*
 *  Same with image_info_get, but never allocate new ImageInfo.
 */
ImageInfo      *image_info_lookup               (const gchar   *filename);

/* setter/getter */
void            image_info_set_size             (ImageInfo     *info,
                                                 gint           width,
                                                 gint           height);
void            image_info_set_comment          (ImageInfo     *info,
                                                 Comment       *comment);
Comment        *image_info_get_comment          (ImageInfo     *info);
void            image_info_set_link             (ImageInfo     *info,
                                                 const gchar   *link);
FRArchive      *image_info_get_archive          (ImageInfo     *info);
void            image_info_set_flags            (ImageInfo     *info,
                                                 ImageInfoFlags flags);
void            image_info_unset_flags          (ImageInfo     *info,
                                                 ImageInfoFlags flags);

/*  If the object is archive member, return relative path in archive. */
const gchar    *image_info_get_path             (ImageInfo     *info);

/*  If the object is archive member, add prefix (archive path). */
gchar          *image_info_get_path_with_archive(ImageInfo     *info);

/*  If the object is archive member, return the archive path. */
const gchar    *image_info_get_archive_path     (ImageInfo     *info);

gboolean        image_info_is_dir               (ImageInfo     *info);
gboolean        image_info_is_archive           (ImageInfo     *info);
gboolean        image_info_is_in_archive        (ImageInfo     *info);
gboolean        image_info_is_url               (ImageInfo     *info);

gboolean        image_info_is_animation         (ImageInfo     *info);
gboolean        image_info_is_movie             (ImageInfo     *info);
gboolean        image_info_is_audio             (ImageInfo     *info);
gboolean        image_info_is_same              (ImageInfo     *info1,
                                                 ImageInfo     *info2);

gboolean        image_info_extract_archive      (ImageInfo     *info);
const gchar    *image_info_get_format           (ImageInfo     *info);
void            image_info_get_image_size       (ImageInfo     *info,
                                                 gint          *width_ret,
                                                 gint          *height_ret);

gboolean        image_info_rename_image         (ImageInfo     *info,
                                                 const gchar   *filename);
gboolean        image_info_need_temp_file       (ImageInfo     *info);
gchar          *image_info_get_temp_file_path   (ImageInfo     *info);
gchar          *image_info_get_temp_file        (ImageInfo     *info);
GimvIO         *image_info_get_gio              (ImageInfo     *info);

/*
 *  memory management
 */
void            image_info_finalize             (ImageInfo     *info);
ImageInfo      *image_info_ref                  (ImageInfo     *info);
void            image_info_unref                (ImageInfo     *info);


/*
 *  used by fr-command only
 */
ImageInfo      *image_info_get_with_archive     (const gchar   *filename,
                                                 FRArchive     *archive,
                                                 struct stat   *st);
void            image_info_unref_with_archive   (ImageInfo     *info);

#endif /* __IMAGE_INFO_H__ */
