/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001-2002 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_loader.c,v 1.14.2.16 2003/05/17 17:43:27 makeinu Exp $
 */

#include "image_loader.h"
#include "gtk2-compat.h"
#include "fileutil.h"


typedef enum {
   LOAD_START_SIGNAL,
   PROGRESS_UPDATE_SIGNAL,
   LOAD_END_SIGNAL,
   LAST_SIGNAL
} ImageLoaderSignalType;


typedef enum
{
   IMAGE_LOADER_LOADING_FLAG = 1 << 0,
   IMAGE_LOADER_CANCEL_FLAG  = 1 << 1,
   IMAGE_LOADER_DEBUG_FLAG   = 1 << 2
} ImageLoaderFlags;


struct ImageLoaderPriv_Tag
{
   GimvIO      *gio;
   gboolean     use_specified_gio;
   GimvImage   *image;
   ImageLoaderLoadType load_type;
   gboolean     load_as_animation;
   gfloat       width_scale;
   gfloat       height_scale;
   gint         width;
   gint         height;
   const gchar *temp_file;
   gint         flags;

   /* que */
   ImageInfo *next_info;
};

static void      image_loader_class_init (ImageLoaderClass *klass);
static void      image_loader_init       (ImageLoader      *loader);
static void      image_loader_destroy    (GtkObject        *object);
static gboolean  idle_image_loader_load  (gpointer          data);

/* callback */
static void      image_loader_load_end   (ImageLoader      *loader);

static GtkObjectClass *parent_class = NULL;
static gint image_loader_signals[LAST_SIGNAL] = {0};


/****************************************************************************
 *
 *  Plugin Management
 *
 ****************************************************************************/
static GList *image_loader_list = NULL;


static gint
comp_func_priority (ImageLoaderPlugin *plugin1,
                    ImageLoaderPlugin *plugin2)
{
   g_return_val_if_fail (plugin1, 1);
   g_return_val_if_fail (plugin2, -1);

   return plugin1->priority_hint - plugin2->priority_hint;
}


gboolean
image_loader_plugin_regist (const gchar *plugin_name,
                            const gchar *module_name,
                            gpointer impl,
                            gint     size)
{
   ImageLoaderPlugin *image_loader = impl;

   g_return_val_if_fail (module_name, FALSE);
   g_return_val_if_fail (image_loader, FALSE);
   g_return_val_if_fail (size > 0, FALSE);
   g_return_val_if_fail (image_loader->if_version == GIMV_IMAGE_LOADER_IF_VERSION, FALSE);
   g_return_val_if_fail (image_loader->id, FALSE);

   image_loader_list = g_list_append (image_loader_list,
                                      image_loader);

   image_loader_list = g_list_sort (image_loader_list,
                                    (GCompareFunc) comp_func_priority);

   return TRUE;
}


/****************************************************************************
 *
 *
 *
 ****************************************************************************/
static void
image_loader_class_init (ImageLoaderClass *klass)
{
   GtkObjectClass *object_class;

   object_class = (GtkObjectClass *) klass;
   parent_class = gtk_type_class (gtk_object_get_type ());

   image_loader_signals[LOAD_START_SIGNAL]
      = gtk_signal_new ("load_start",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE (object_class),
                        GTK_SIGNAL_OFFSET (ImageLoaderClass, load_start),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   image_loader_signals[PROGRESS_UPDATE_SIGNAL]
      = gtk_signal_new ("progress_update",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE (object_class),
                        GTK_SIGNAL_OFFSET (ImageLoaderClass, load_end),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   image_loader_signals[LOAD_END_SIGNAL]
      = gtk_signal_new ("load_end",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE (object_class),
                        GTK_SIGNAL_OFFSET (ImageLoaderClass, load_end),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   gtk_object_class_add_signals (object_class,
                                 image_loader_signals, LAST_SIGNAL);

   object_class->destroy  = image_loader_destroy;

   klass->load_start      = NULL;
   klass->progress_update = NULL;
   klass->load_end        = image_loader_load_end;
}


static void
image_loader_init (ImageLoader *loader)
{
   loader->info                    = NULL;

   loader->priv                    = g_new0 (ImageLoaderPriv, 1);
   loader->priv->gio               = NULL;
   loader->priv->use_specified_gio = FALSE;
   loader->priv->image             = NULL;
   loader->priv->load_type         = IMAGE_LOADER_LOAD_NORMAL;
   loader->priv->load_as_animation = FALSE;
   loader->priv->width_scale       = -1.0;
   loader->priv->height_scale      = -1.0;
   loader->priv->width             = -1;
   loader->priv->height            = -1;
   loader->priv->temp_file         = NULL;
   loader->priv->flags             = 0;
   loader->priv->next_info         = NULL;

#ifdef USE_GTK2
   gtk_object_ref (GTK_OBJECT (loader));
   gtk_object_sink (GTK_OBJECT (loader));
#endif
}


guint
image_loader_get_type (void)
{
   static guint image_loader_type = 0;

   if (!image_loader_type) {
      static const GtkTypeInfo image_loader_info = {
         "ImageLoader",
         sizeof (ImageLoader),
         sizeof (ImageLoaderClass),
         (GtkClassInitFunc) image_loader_class_init,
         (GtkObjectInitFunc) image_loader_init,
         NULL,
         NULL,
         (GtkClassInitFunc) NULL,
      };

      image_loader_type = gtk_type_unique (gtk_object_get_type (), &image_loader_info);
   }

   return image_loader_type;
}


ImageLoader *
image_loader_new (void)
{
   ImageLoader *loader = IMAGE_LOADER (gtk_type_new (image_loader_get_type ()));

   return loader;
}


ImageLoader *
image_loader_new_with_image_info (ImageInfo *info)
{
   ImageLoader *loader;

   loader = image_loader_new ();
   image_loader_set_image_info (loader, info);

   return loader;
}


ImageLoader *
image_loader_new_with_file_name (const gchar *filename)
{
   ImageLoader *loader;
   ImageInfo *info;

   info = image_info_get (filename);
   if (!info) return NULL;
   loader = image_loader_new_with_image_info (info);

   return loader;
}


ImageLoader *
image_loader_ref (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), NULL);

   gtk_object_ref (GTK_OBJECT (loader));

   return loader;
}


void
image_loader_unref (ImageLoader *loader)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));

   gtk_object_unref (GTK_OBJECT (loader));
}


static void
image_loader_destroy (GtkObject *object)
{
   ImageLoader *loader = IMAGE_LOADER (object);

   if (image_loader_is_loading (loader))
      image_loader_load_stop (loader);

   if (loader->info)
      image_info_unref (loader->info);
   loader->info = NULL;

   if (loader->priv->next_info)
      image_info_unref (loader->priv->next_info);
   loader->priv->next_info = NULL;

   if (loader->priv->gio)
      gimv_io_unref (loader->priv->gio);
   loader->priv->gio = NULL;

   if (loader->priv->image)
      gimv_image_unref (loader->priv->image);
   loader->priv->image = NULL;

   if (loader->priv->temp_file)
      g_free ((gchar *) loader->priv->temp_file);
   loader->priv->temp_file = NULL;

   if (loader->priv)
      g_free (loader->priv);
   loader->priv = NULL;

   if (GTK_OBJECT_CLASS (parent_class)->destroy)
      (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


void
image_loader_set_image_info (ImageLoader *loader, ImageInfo *info)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));
   g_return_if_fail (info);

   if (image_loader_is_loading (loader)) {
      image_loader_load_stop (loader);
      if (loader->priv->next_info)
         image_info_unref (loader->priv->next_info);
      loader->priv->next_info = image_info_ref (info);
      return;
   }

   if (loader->priv->image)
      gimv_image_unref (loader->priv->image);
   loader->priv->image = NULL;

   if (loader->priv->temp_file)
      g_free ((gchar *)loader->priv->temp_file);
   loader->priv->temp_file = NULL;

   if (loader->info)
      image_info_unref (loader->info);

   if (info)
      loader->info = image_info_ref (info);
   else
      loader->info = NULL;
}


void
image_loader_set_gio (ImageLoader *loader, GimvIO *gio)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));
   g_return_if_fail (!image_loader_is_loading (loader));

   loader->priv->gio = gimv_io_ref (gio);
   if (gio)
      loader->priv->use_specified_gio = TRUE;
   else
      loader->priv->use_specified_gio = FALSE;
}


gboolean
image_loader_set_as_animation (ImageLoader *loader,
                               gboolean animation)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (!image_loader_is_loading (loader), FALSE);

   loader->priv->load_as_animation = animation;

   return TRUE;
}


gboolean
image_loader_set_load_type (ImageLoader *loader,
                            ImageLoaderLoadType type)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (!image_loader_is_loading (loader), FALSE);

   loader->priv->load_type = type;

   return TRUE;
}


gboolean
image_loader_set_scale (ImageLoader *loader,
                        gfloat       w_scale,
                        gfloat       h_scale)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (!image_loader_is_loading (loader), FALSE);

   loader->priv->width_scale  = w_scale;
   loader->priv->height_scale = h_scale;

   return TRUE;
}


gboolean
image_loader_set_size_request (ImageLoader *loader,
                               gint         width,
                               gint         height)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (!image_loader_is_loading (loader), FALSE);

   loader->priv->width   = width;
   loader->priv->height  = height;

   return TRUE;
}


void
image_loader_load_start (ImageLoader *loader)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));
   g_return_if_fail (loader->info);

   g_return_if_fail (!image_loader_is_loading(loader));

   loader->priv->flags &= ~IMAGE_LOADER_LOADING_FLAG;
   loader->priv->flags &= ~IMAGE_LOADER_CANCEL_FLAG;

   /* gtk_idle_add (idle_image_loader_load, loader); */
   image_loader_load (loader);
}


void
image_loader_load_stop (ImageLoader *loader)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));

   if (loader->priv->flags & IMAGE_LOADER_LOADING_FLAG) {
      loader->priv->flags |= IMAGE_LOADER_CANCEL_FLAG;
   }
}


gboolean
image_loader_is_loading (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);

   if (loader->priv->flags & IMAGE_LOADER_LOADING_FLAG)
      return TRUE;
   else
      return FALSE;
}


GimvImage *
image_loader_get_image (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), NULL);

   return loader->priv->image;
}


void
image_loader_unref_image (ImageLoader *loader)
{
   g_return_if_fail (IS_IMAGE_LOADER (loader));

   if (loader->priv->image)
      gimv_image_unref (loader->priv->image);
   loader->priv->image = NULL;
}


GimvIO *
image_loader_get_gio (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), NULL);

   return loader->priv->gio;
}


const gchar *
image_loader_get_path (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), NULL);

   if (!loader->info) return NULL;

   if (image_info_need_temp_file (loader->info)) {
      if (!loader->priv->temp_file)
         loader->priv->temp_file = image_info_get_temp_file_path (loader->info);
      return loader->priv->temp_file;
   }

   return image_info_get_path (loader->info);
}


ImageLoaderLoadType
image_loader_get_load_type (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader),
                         IMAGE_LOADER_LOAD_NORMAL);

   return loader->priv->load_type;
}


gboolean
image_loader_load_as_animation (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);

   return loader->priv->load_as_animation;
}


gboolean
image_loader_get_scale (ImageLoader *loader,
                        gfloat *width_scale_ret,
                        gfloat *height_scale_ret)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (width_scale_ret && height_scale_ret, FALSE);

   *width_scale_ret  = loader->priv->width_scale;
   *height_scale_ret = loader->priv->height_scale;

   if (*width_scale_ret < 0.0 || *height_scale_ret < 0.0)
      return FALSE;

   return TRUE;
}


gboolean
image_loader_get_size_request (ImageLoader *loader,
                               gint *width_ret,
                               gint *height_ret)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);
   g_return_val_if_fail (width_ret && height_ret, FALSE);

   *width_ret  = loader->priv->width;
   *height_ret = loader->priv->height;

   if (*width_ret < 0.0 || *height_ret < 0.0)
      return FALSE;

   return TRUE;
}


gboolean
image_loader_progress_update (ImageLoader *loader)
{
   g_return_val_if_fail (IS_IMAGE_LOADER (loader), FALSE);

   gtk_signal_emit (GTK_OBJECT(loader),
                    image_loader_signals[PROGRESS_UPDATE_SIGNAL]);

   if (loader->priv->flags & IMAGE_LOADER_CANCEL_FLAG)
      return FALSE;
   else
      return TRUE;
}


void
image_loader_load (ImageLoader *loader)
{
   GList *node;
   gboolean cancel = FALSE;

   g_return_if_fail (IS_IMAGE_LOADER (loader));

   if (image_loader_is_loading (loader)) {
      return;
   }

   /* set flags */
   loader->priv->flags &= ~IMAGE_LOADER_LOADING_FLAG;
   loader->priv->flags &= ~IMAGE_LOADER_CANCEL_FLAG;
   loader->priv->flags |= IMAGE_LOADER_LOADING_FLAG;

   gtk_signal_emit (GTK_OBJECT(loader),
                    image_loader_signals[LOAD_START_SIGNAL]);

   /* load GimvIO and name of temporary file */
   if (loader->info) {
      if (!loader->priv->use_specified_gio) {
         if (loader->priv->gio)
            gimv_io_unref (loader->priv->gio);
         loader->priv->gio = image_info_get_gio (loader->info);

         if (image_info_need_temp_file (loader->info)) {
            g_free ((gchar *) loader->priv->temp_file);
            loader->priv->temp_file = image_info_get_temp_file (loader->info);
            if (!loader->priv->temp_file) {
               if (loader->priv->gio) {
                  gimv_io_unref (loader->priv->gio);
                  loader->priv->gio = NULL;
               }
               return;
            }
         }
      }
   }

   /* free old image */
   if (loader->priv->image)
      gimv_image_unref (loader->priv->image);
   loader->priv->image = NULL;

   /* try all loader */
   for (node = image_loader_list; node; node = g_list_next (node)) {
      ImageLoaderPlugin *plugin = node->data;

      cancel = !image_loader_progress_update (loader);
      if (cancel) break;

      if (!plugin->loader) continue;

      if (loader->priv->gio)
         gimv_io_seek (loader->priv->gio, 0, SEEK_SET);

      loader->priv->image = plugin->loader (loader, NULL);

      if (loader->priv->image) {
         if (loader->info && !IMAGE_INFO_IS_SYNCED(loader->info)) {
            ImageInfoFlags flags = IMAGE_INFO_SYNCED_FLAG;

            /* set image info */
            if (gimv_image_is_anim (loader->priv->image)) {
               flags |= IMAGE_INFO_ANIMATION_FLAG;
            }
            image_info_set_size (loader->info,
                                 gimv_image_width (loader->priv->image),
                                 gimv_image_height (loader->priv->image));
            image_info_set_flags (loader->info, flags);
         }
         break;
      }

      if (loader->priv->flags & IMAGE_LOADER_CANCEL_FLAG) break;
   }

   /* free GimvIO if needed */
   if (!loader->priv->use_specified_gio) {
      if (loader->priv->gio)
         gimv_io_unref (loader->priv->gio);
      loader->priv->gio = NULL;
   }

   /* reset flags and emit signal */
   loader->priv->flags &= ~IMAGE_LOADER_LOADING_FLAG;
   if (loader->priv->flags & IMAGE_LOADER_CANCEL_FLAG) {
      loader->priv->flags &= ~IMAGE_LOADER_CANCEL_FLAG;
      if (loader->priv->flags & IMAGE_LOADER_DEBUG_FLAG)
         g_print ("----- loading canceled -----\n");
      /* emit canceled signal? */
      gtk_signal_emit (GTK_OBJECT(loader),
                       image_loader_signals[LOAD_END_SIGNAL]);
   } else {
      if (loader->priv->flags & IMAGE_LOADER_DEBUG_FLAG)
         g_print ("----- loading done -----\n");
      gtk_signal_emit (GTK_OBJECT(loader),
                       image_loader_signals[LOAD_END_SIGNAL]);
   }
}


static gboolean
idle_image_loader_load (gpointer data)
{
   ImageLoader *loader = data;

   image_loader_load (loader);

   return FALSE;
}


static void
image_loader_load_end (ImageLoader *loader)
{
   if (!loader->priv->next_info)
      return;

   loader->info = loader->priv->next_info;
   loader->priv->next_info = NULL;
   gtk_idle_add (idle_image_loader_load, loader);
}
