#include <stdlib.h>
#include <string.h>
#include "gmet/gmet.h"

int gmetDebug = 0;

static GmetData *
gmetNewData_( GmetData * parent, GmetStatement * org )
{
  GmetData *ret = ( GmetData * ) malloc( sizeof( GmetData ) );
  if ( !ret || !org )
    return NULL;
  ret->parent = parent;
  ret->org = org;
  if ( org->type == gmetTypeBlock )
  {
    ret->data.list = gmetNewVector( ( int ) org->size );
    if ( !ret->data.list )
      return NULL;
  }

  return ret;
}

static GmetData *
parseInt( GmetStatement * org, GmetData * parent, GmetStream * stream,
          int *ret )
{
  GmetData *data = gmetNewData_( parent, org );
  int size = gmetExpGet( parent, org->size );
  *ret = -1;

  GMET_CHKPTR( data );
  if ( !data )
    return NULL;

  if ( size > 32 )
  {
    gmetDeleteData( data );
    if ( gmetDebug )
      fprintf( stderr, "parseInt: stream error\n" );
    return NULL;
  }

  if ( *ret = gmetStreamIntN( stream, &data->data.num, size ) )
    return NULL;

  *ret = 0;
  return data;
}

static GmetData *
parseStr( GmetStatement * org, GmetData * parent, GmetStream * stream,
          int *ret )
{
  GmetData *data = gmetNewData_( parent, org );
  int size = gmetExpGet( parent, org->size );
  *ret = -1;

  if ( !data )
    return NULL;

  data->data.str = ( char * ) malloc( size + 1 );
  memset( data->data.str, 0, size + 1 );
  if ( *ret = gmetStreamBytes( stream, data->data.str, size ) )
  {
    fprintf( stderr, "parseStr: not bite alien\n" );
    gmetDeleteData( data );
    return NULL;
  }

  if ( gmetDebug )
    fprintf( stderr, "parseStr: get string \"%s\"\n", data->data.str );

  *ret = 0;
  return data;
}

static int
pushBlockDatas( GmetData * data, GmetVector * list, int size,
                GmetStream * stream )
{
  int i;
  int ret = -1;

  if ( !data || !list || !stream )
    return -1;

  for ( i = 0; i < size; i++ )
  {
    int j;
    GmetStatement *st = ( GmetStatement * ) gmetVectorGet( list, i );
    int max = gmetExpGet( data, st->loop );

    for ( j = 0; j < max; j++ )
    {
      GmetData *tmpData = st->readData( st, data, stream, &ret );
      if ( tmpData )
        gmetVectorPush( data->data.list, tmpData );
      if ( ret )
        return ret;
    }
  }

  return 0;
}

static GmetData *
parseIfElse( GmetStatement * this, GmetData * parent, GmetStream * stream,
             int *ret )
{
  GmetIfElse *org = ( GmetIfElse * ) this;
  int cond = gmetExpGet( parent, org->condition );
  GmetVector *list = ( cond ? org->blk.list : org->listElse );
  int size = ( cond ? ( int ) org->blk.st.size : org->sizeElse );
  int i;
  *ret = -1;

  if ( gmetDebug )
    fprintf( stderr, "parseIfElse: condition \"%s\" = %d\n",
             org->condition, cond );

  *ret = pushBlockDatas( parent, list, size, stream );

  return 0;
}

static GmetData *
parseBlock( GmetStatement * this, GmetData * parent, GmetStream * stream,
            int *ret )
{
  GmetBlock *org = ( GmetBlock * ) this;
  GmetData *data = gmetNewData_( parent, this );
  int i;
  *ret = -1;

  if ( !data || !org )
  {
    if ( gmetDebug )
      fprintf( stderr, "parseBlock: malloc GmetData ...failure\n" );
    return NULL;
  }

  *ret = pushBlockDatas( data, org->list, org->st.size, stream );

  return data;
}

GmetStatement *
gmetNewStatement( char *name, struct GmetBlock * parent, int type, char *loop,
                  char *size )
{
  GmetStatement *ret = NULL;

  ret = ( GmetStatement * ) malloc( sizeof( GmetStatement ) );
  if ( ret )
  {
    ret->name = name;
    ret->parent = parent;
    ret->type = type;
    ret->loop = loop;
    ret->size = size;
    switch ( ret->type )
    {
    case gmetTypeInt:
      ret->readData = parseInt;
      break;
    case gmetTypeStr:
      ret->readData = parseStr;
      break;
    default:
      ret->readData = NULL;
      break;
    }
  }
  return ret;
}

static void
deleteStatementList( GmetVector * v )
{
  int i, max = gmetVectorSize( v );
  GmetStatement *tmp;

  while ( tmp = gmetVectorPop( v ) )
    gmetDeleteStatement( tmp );

  gmetDeleteVector( v );
}

void
gmetDeleteStatement( GmetStatement * st )
{
  switch ( st->type )
  {
  case gmetTypeBlock:
    gmetDeleteBlock( ( GmetBlock * ) st );
    break;
  case gmetTypeIfElse:
    gmetDeleteIfElse( ( GmetIfElse * ) st );
    break;
  default:
    free( st );
    break;
  }
}

static GmetData *
gmetDataReserve( GmetStatement * this, GmetData * parent, GmetStream * stream,
                 int *ret )
{
  int size = gmetExpGet( parent, this->size );
  *ret = -1;

  gmetStreamReserve( stream, size );

  *ret = 0;
  return 0;
}

GmetStatement *
gmetNewFunc_reserve( GmetBlock * parent, char *arg )
{
  GmetStatement *ret =
    gmetNewStatement( "reserve", parent, gmetTypeFunc, "1", arg );
  if ( ret )
    ret->readData = gmetDataReserve;

  return ret;
}

static GmetData *
gmetDataAlign( GmetStatement * this, GmetData * parent, GmetStream * stream,
               int *ret )
{
  int size = gmetExpGet( parent, this->size );
  *ret = -1;

  gmetStreamAlign( stream, size );

  *ret = 0;
  return 0;
}

GmetStatement *
gmetNewFunc_align( GmetBlock * parent, char *arg )
{
  GmetStatement *ret =
    gmetNewStatement( "align", parent, gmetTypeFunc, "1", arg );

  if ( ret )
    ret->readData = gmetDataAlign;

  return ret;
}

GmetBlock *
gmetNewBlock( char *name, struct GmetBlock * parent, char *loop, int size )
{
  GmetBlock *ret = ( GmetBlock * ) malloc( sizeof( GmetBlock ) );
  GmetStatement *st = ( GmetStatement * ) ret;

  if ( ret )
  {
    st->name = name;
    st->parent = parent;
    st->type = gmetTypeBlock;
    st->loop = loop;
    st->size = ( char * ) size;
    st->readData = parseBlock;

    ret->list = gmetNewVector( size );
  }

  return ret;
}

void
gmetDeleteBlock( GmetBlock * blk )
{
  if ( !blk )
    return;
  deleteStatementList( blk->list );
  free( blk );
}

GmetIfElse *
gmetNewIfElse( GmetBlock * parent, int sizeIf, int sizeElse, char *condition )
{
  GmetIfElse *ret = ( GmetIfElse * ) malloc( sizeof( GmetIfElse ) );
  GmetStatement *st = ( GmetStatement * ) ret;

  if ( ret )
  {
    st->name = "if_else";
    st->parent = parent;
    st->type = gmetTypeIfElse;
    st->loop = "1";
    st->size = ( char * ) sizeIf;
    st->readData = parseIfElse;

    ret->blk.list = gmetNewVector( sizeIf );
    ret->sizeElse = sizeElse;
    ret->condition = condition;
    ret->listElse = gmetNewVector( sizeElse );
  }

  return ret;
}

void
gmetDeleteIfElse( GmetIfElse * ie )
{
  if ( !ie )
    return;
  deleteStatementList( ie->blk.list );
  deleteStatementList( ie->listElse );
  free( ie );
}

static int
gmetIfElseAdd( GmetIfElse * ie, GmetStatement * st, int if_ )
{
  if ( !st )
  {
    if ( gmetDebug )
      fprintf( stderr, "gmetIfElseAdd: don't add NULL\n" );
    return 1;
  }
  return gmetVectorPush( if_ ? ie->blk.list : ie->listElse, st );
}

int
gmetIfElseAddIf( GmetIfElse * ie, GmetStatement * st )
{
  return gmetIfElseAdd( ie, st, 1 );
}

int
gmetIfElseAddElse( GmetIfElse * ie, GmetStatement * st )
{
  return gmetIfElseAdd( ie, st, 0 );
}

int
gmetBlockAdd( struct GmetBlock *blk, struct GmetStatement *st )
{
  if ( !st )
  {
    if ( gmetDebug )
      fprintf( stderr, "gmetIfElseAdd: don't add NULL\n" );
    return 1;
  }
  return gmetVectorPush( blk->list, st );
}

GmetData *
gmetNewData( struct GmetData * parent, struct GmetStatement * org,
             struct GmetStream * stream )
{
  GmetData *data = NULL;
  int ret;

  if ( !org || !stream )
    return NULL;

  if ( gmetDebug )
    fprintf( stderr, "gmetNewData: start parse: \"%s\" - type %d\n",
             org->name, org->type );

  if ( org->readData )
    data = org->readData( org, parent, stream, &ret );

  // EOF if ret==-2
  if ( ret && ret != -2 )
    return NULL;

  GMET_CHKPTR( data );
  return data;
}

void
gmetDeleteData( GmetData * data )
{
  switch ( data->org->type )
  {
  case gmetTypeBlock:
  case gmetTypeIfElse:
    {
      GmetData *tmp;
      while ( tmp = gmetVectorPop( data->data.list ) )
        gmetDeleteData( tmp );
      break;
    }
  case gmetTypeStr:
    free( data->data.str );
    break;
  default:
    break;
  }

  free( data );
}

GmetData *
gmetArrayGet( GmetData * parent, GmetArray * ar )
{
  char *name;
  int idx;

  GMET_CHKPTR( parent );
  GMET_CHKPTR( ar );

  name = ar->name;
  idx = ar->idx;

  if ( parent->org->type != gmetTypeBlock
       && parent->org->type != gmetTypeIfElse )
    return NULL;

  {
    int i, max = gmetVectorSize( parent->data.list );

    for ( i = 0; i < max; i++ )
    {
      GmetData *tmp = ( GmetData * ) gmetVectorGet( parent->data.list, i );

      GMET_CHKPTR( tmp );

      if ( !strncmp( tmp->org->name, name, ar->size ) )
      {
        if ( !idx )
          return tmp;
        idx--;
      }
    }
  }

  if ( gmetDebug )
  {
    char tmp[32];
    memset( tmp, 0, 32 );
    memcpy( tmp, ar->name, MIN( ar->size, 31 ) );
    fprintf( stderr, "gmetArrayGet: \"%s\" doesn't have \"%s\"\n",
             parent->org->name, tmp );
  }

  return NULL;
}

int
gmetRefGet( GmetData * parent, GmetVector * array )
{
  GmetData *data = parent;

  if ( gmetDebug )
    fprintf( stderr, "gmetRefGet: start\n" );

  GMET_CHKPTR( array );

  {
    int i, max = gmetVectorSize( array );
    for ( i = 0; i < max; i++ )
    {
      GmetArray *tmp = ( GmetArray * ) gmetVectorGet( array, i );

      GMET_CHKPTR( tmp );

      data = gmetArrayGet( data, tmp );

      if ( !data )
      {
        if ( gmetDebug )
          fprintf( stderr, "gmetRefGet: go to parent\n" );
        GMET_CHKPTR( parent );
        if ( parent == NULL )
        {
          fprintf( stderr, "gmetRefGet: not found: code 1\n" );
          return -1;
        }
        return gmetRefGet( parent->parent, array );
      }
    }
  }

  if ( data->org->type == gmetTypeInt )
    return data->data.num;

  fprintf( stderr, "gmetRefGet: not found: code 2\n" );
  return -1;
}
