#include <stdlib.h>
#include <string.h>
#include "gmet/gmet_stream.h"

#define inline_ inline
#define BFSZ GmetStreamBuffSize
#define BIT2BYTE(x) ((x) >> 3)
#define BITOFFSET(x) (7 - ((x) & 7))

GmetStream *
gmetNewStream( FILE * fp )
{
  GmetStream *ret = ( GmetStream * ) malloc( sizeof( GmetStream ) );

  if ( ret )
  {
    ret->file = fp;
    ret->wp = 0;
    ret->rp = 0;
    ret->count = 0;
    memset( ret->buff, 0, GmetStreamBuffSize );
  }

  return ret;
}

void
gmetDeleteStream( GmetStream * stream )
{
  free( stream );
}

static int
checkAndUpdate( GmetStream * stream, int bit )
{
  if ( !stream )
    return -1;
  if ( stream->rp == 8 * BFSZ )
  {
    stream->rp = 0;
    stream->count++;
  }

  int lineRem = BFSZ - stream->wp;
  int recursiveflg = 0;
  if ( BIT2BYTE( stream->rp ) < stream->wp )
  {
    if ( BIT2BYTE( stream->rp + bit ) < stream->wp )
      return 0;
    recursiveflg++;
  }
  else if ( BIT2BYTE( stream->rp ) > stream->wp )
  {
    if ( BIT2BYTE( stream->rp + bit ) < stream->wp + BFSZ )
      return 0;
    lineRem = BIT2BYTE( stream->rp ) - stream->wp;
  }

  if ( gmetDebug )
    fprintf( stderr, "checkAndUpdate: update\n" );

  {
    int read = fread( stream->buff + stream->wp, 1, lineRem, stream->file );
    if ( !read )
    {
      return -2;
    }

    if ( gmetDebug )
      fprintf( stderr, " ...success\n" );

    stream->wp += read;
    if ( stream->wp == BFSZ )
      stream->wp = 0;
  }

  if ( recursiveflg )
    return checkAndUpdate( stream, bit );

  return 0;
}

int
gmetStreamByte( GmetStream * stream, char *ret )
{
  if ( !stream )
    return -1;
  if ( stream->rp & 7 )
  {
    fprintf( stderr, "gmetStreamByte: not byte align\n" );
    return 1;
  }
  {
    int eof;
    if ( eof = checkAndUpdate( stream, 8 ) )
      return eof;
    *ret = *( stream->buff + BIT2BYTE( stream->rp ) );
    stream->rp += 8;
  }
  return 0;
}

int
gmetStreamBytes( GmetStream * stream, char *ret, int size )
{
  if ( !stream )
    return -1;
  if ( stream->rp & 7 )
  {
    fprintf( stderr, "gmetStreamBytes: not byte align\n" );
    return -1;
  }
  {
    int i;

    for ( i = 0; i < size; i++ )
    {
      int eof;
      if ( eof = checkAndUpdate( stream, 8 ) )
        return eof;
      ret[i] = stream->buff[stream->rp >> 3];
      stream->rp += 8;
    }
  }

  return 0;
}

int
gmetStreamBit( GmetStream * stream, char *ret )
{
  int eof;
  if ( !ret )
    return -1;
  if ( eof = checkAndUpdate( stream, 1 ) )
    return eof;
  *ret =
    1 & ( stream->buff[BIT2BYTE( stream->rp )] >> BITOFFSET( stream->rp ) );
  stream->rp++;
  return 0;
}

int
gmetStreamBitCopyL( GmetStream * stream, char *ret, int size )
{
  int i = 0;
  char c = 0;

  for ( ; i < size; i++ )
  {
    char bit;
    int eof;
    if ( eof = gmetStreamBit( stream, &bit ) )
      return eof;
    if ( i + 1 == size )
      c = c << ( i & 7 );
    else if ( c )
      c = c << 1;
    else if ( i != 0 )
    {
      *( ret + i / 8 ) = c;
      c = 0;
    }
    c += bit;
  }

  return 0;
}

int
gmetStreamBitCopy( GmetStream * stream, char *ret, int size )
{
  return gmetStreamBitCopyL( stream, ret, size );
}

static int
isAlign( GmetStream * stream, int size )
{
  int ofs = BFSZ * 8 + stream->rp;
  if ( ofs % size )
    return 0;
  return 1;
}

/*
  "Big" which is tail of function name, mean binary data format.
  Now I have implement gmetStreamInt* for little endian machine.
  I must implement for big but yet.
*/

int
gmetStreamIntNBig( GmetStream * stream, int *ret, int size )
{
  switch ( size )
  {
  case 1:
    return gmetStreamBit( stream, ( char * ) ret );
  case 32:
    return gmetStreamInt32Big( stream, ret );
  default:
    {
      int i;
      *ret = 0;

      if ( gmetDebug )
        printf( "streamIntN start\n" );

      for ( i = 0; i < size; i++ )
      {
        int eof;
        char c = 0;
        if ( eof = gmetStreamBit( stream, &c ) )
        {
          if ( gmetDebug )
            fprintf( stderr, "streamInt done=%d/%d bit: error code=%d\n", i,
                     size, eof );
          return eof;
        }
        *ret = *ret << 1;
        *ret += c;
      }

      if ( gmetDebug )
        printf( "StreamIntN: size=%d ret=%d\n", size, *ret );
    }
  }

  return 0;
}

int
gmetStreamIntN( GmetStream * stream, int *ret, int size )
{
  return gmetStreamIntNBig( stream, ret, size );
}

int
gmetStreamInt32Big( GmetStream * stream, int *ret )
{
  if ( !isAlign( stream, 8 ) )
  {
    fprintf( stderr, "gmetStreamInt32Big: not 32bit align\n" );
    return 1;
  }
  {
    union
    {
      int a;
      char b[4];
    } tmp;
    int i = 3;
    for ( ; i >= 0; i-- )
    {
      int eof;
      if ( eof = gmetStreamByte( stream, tmp.b + i ) )
        return eof;
    }
    *ret = tmp.a;
  }

  return 0;
}

int
gmetStreamInt32( GmetStream * stream, int *ret )
{
  return gmetStreamInt32Big( stream, ret );
}

int
gmetStreamAlign( GmetStream * stream, int size )
{
  int ofs = BFSZ * 8 + stream->rp;
  int bitOfs = ofs % size;
  if ( !bitOfs )
    return 0;
  return gmetStreamReserve( stream, size - bitOfs );
}

int
gmetStreamReserve( GmetStream * stream, int size )
{
  int i;

  for ( i = 0; i < size; i++ )
  {
    int eof;
    if ( eof = checkAndUpdate( stream, 1 ) )
      return eof;
    stream->rp++;
  }

  return 0;
}
