/* $Id: background.c,v 1.2 2002/07/17 13:24:19 iiojun Exp $ */

/* Face Pointer Library
 * Copyright (C) 1999-2002 Kazuo HIYANE, Jun IIO, and Tomoyuki YATABE
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include "background.h"

/* private function prototypes **************************************/
static void background_write_frame_data (Background* filter,
					 MalibFrame* frame);

/* virtual function table *******************************************/
static BackgroundClass background_class = 
{
  (void (*)(MalibObject*)) 	     	 malib_filter_delete,
  (void (*)(MalibSource*, MalibFrame*))  background_write_frame_data
};

/* public functions *************************************************/
Background*
background_new ()
{
  MALIB_FILTER_GENERIC_NEW ( Background, &background_class,
			     MALIB_FRAME_COLORMODEL_RGB_A );
}

Background*
background_new_with_buf (MalibBuffer* buf)
{
  MALIB_FILTER_GENERIC_NEW_WITH_BUF ( Background, background_new,
				      background_set_buffer, buf );
}

Background*
background_new_with_image_and_buf (char* bg_image, MalibBuffer* buf)
{
  Background* retptr = background_new_with_buf (buf);
  background_set_bg_image (retptr, bg_image);
  return retptr;
}

void
background_set_buffer (Background* filter, MalibBuffer* buf)
{
  MalibFrame* frame;

  g_return_if_fail (filter && buf);
  
  frame = malib_buffer_get_current_frame (buf);

  if (filter->bg_image)
    {
      free (filter->bg_image);
    }

  /* alloc bg_image data area */
  filter->bg_image = (unsigned char*)
    malloc ( frame->width * frame->height
	     * MALIB_FRAME_DEFAULT_NUM_COLORS_RGB * sizeof (unsigned char) );

  /* create new frame, which has almost all the same information
     except data format (rgb_a -> rgb), and timestamp */
  malib_filter_set_buffer_with_colormodel ((MalibFilter*)filter, buf,
					   MALIB_FRAME_COLORMODEL_RGB);
}

void
background_set_bg_image (Background* filter, char* bg_image)
{
  MalibBuffer* input;
  MalibFrame* frame;
  GdkPixbuf* pix_file;
  GdkPixbuf* pixbuf;

  g_return_if_fail (filter && ((MalibFilter*)filter)->buf);

  input = ((MalibFilter*)filter)->buf;
  frame = malib_buffer_get_current_frame (input);
  
  /* create scaled pixbuf */
  g_return_if_fail (pix_file = gdk_pixbuf_new_from_file (bg_image));
  pixbuf = gdk_pixbuf_scale_simple (pix_file, frame->width, frame->height,
				    GDK_INTERP_HYPER);

  /* get bg_image data */
  memcpy (filter->bg_image, gdk_pixbuf_get_pixels (pixbuf), 
	  frame->width * frame->height 
	  * MALIB_FRAME_DEFAULT_NUM_COLORS_RGB * sizeof (unsigned char) );

  /* free pixbuf instances */
  gdk_pixbuf_unref (pixbuf);
  gdk_pixbuf_unref (pix_file);
}

/* private functions ************************************************/

static void
background_write_frame_data (Background* filter, MalibFrame* frame)
{
  MalibBuffer* input;
  unsigned int i, j;
  unsigned int image_size, maxval, alpha;

  /* the flag whether we need to propagate previous section */
  int need_increment = 0;
  
  /* pointers to rgb frame data and data area to store 
     the result of calculation */
  int* in_image;
  int* out_image;
  unsigned char* bg_image;

  g_return_if_fail (filter && frame);
  g_return_if_fail (((MalibFilter*)filter)->buf && frame->data);

  input = ((MalibFilter*)filter)->buf;
  out_image = frame->data;
  bg_image  = filter->bg_image;

  /* increment previous buffer data */
  MALIB_OBJECT_COUNT_REFERENCES (filter, need_increment);
  if (need_increment)
    {
      malib_holder_increment_frame ((MalibHolder*)input);
    }

  /* get pointers to rgb frame data */
  in_image = malib_buffer_get_current_frame (input) ->data;

  maxval = (1 << MALIB_FRAME_DEFAULT_DEPTH_RGB) - 1;

  image_size = malib_filter_calc_output_image_size ((MalibFilter*) filter);

  /* create mixed image */
  for (i = 0; i < image_size; i += MALIB_FRAME_DEFAULT_NUM_COLORS_RGB)
    {
      alpha = in_image [MALIB_FRAME_DEFAULT_NUM_COLORS_RGB];

      for (j = 0; j < MALIB_FRAME_DEFAULT_NUM_COLORS_RGB; j++)
	{
	  out_image [i + j] = ((maxval - alpha) * (*bg_image++) 
			       + alpha * (*in_image++)) / maxval;
	}
      in_image ++;
    }

}
