#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <gnome.h>

#include "range2d.h"

#define XSIZE  200
#define YSIZE  200

#define LIGHT_YELLOW 		"#ffffe0"
#define YELLOW 	     		"yellow"
#define DEFAULT_COLOR 		LIGHT_YELLOW
#define DEFAULT_BORDER_COLOR 	"forestgreen"
#define HIGHLIGHT_COLOR		YELLOW
#define HIGHLIGHT_BORDER_COLOR 	"red"

#define SCOPE_DEFAULT_X1 	0.45
#define SCOPE_DEFAULT_Y1 	0.45
#define SCOPE_DEFAULT_X2 	0.55
#define SCOPE_DEFAULT_Y2 	0.55

#define MINIMUM_SCOPE_SIZE	0.02

#define PICKER_SIZE 0.015

#define DEFAULT_A   0.5
#define DEFAULT_B   0.5
#define DEFAULT_RA  0.05
#define DEFAULT_RB  0.05

static void gtk_range2d_class_init(GtkRange2dClass* klass);
static void gtk_range2d_init(GtkRange2d* range2d);
static void gtk_range2d_destroy(GtkObject* object);
static void gtk_range2d_finalize(GtkObject* object);

static inline gboolean range_check_ok(GnomeCanvasItem *item, double dx, double dy);
static void save_parameter(GtkObject* obj, gpointer arg);
static void load_parameter(GtkObject* obj, gpointer arg);
static gint configured(GtkWidget* widget, GdkEvent* event, gpointer arg);
static gint exposed(GtkWidget* widget, GdkEvent* event, gpointer arg);
static gint scope_move_by_drag(GnomeCanvasItem *item, GdkEvent *event, gpointer arg);
static gint scope_resize_by_drag(GnomeCanvasItem *item, GdkEvent *event, gpointer arg);
static gint set_parameter(GtkWidget* widget, gpointer arg);
static void configure_gnome_canvas_items(GtkRange2d* range2d, GnomeCanvas* canvas);
static void entry_activate(GtkWidget* widget, gpointer arg);

static GtkWindowClass* parent_class = NULL;

/***** public functions *****/

GtkType gtk_range2d_get_type(void)
{
  static GtkType range2d_type = 0;
  
  if(!range2d_type) {
    GtkTypeInfo range2d_info = 
    {
      "GtkRange2d",
      sizeof(GtkRange2d),
      sizeof(GtkRange2dClass),
      (GtkClassInitFunc) gtk_range2d_class_init,
      (GtkObjectInitFunc) gtk_range2d_init,
      NULL,
      NULL,
      (GtkClassInitFunc) NULL,
    };
    range2d_type = gtk_type_unique(gtk_window_get_type(), &range2d_info);
  }

  return range2d_type;
}

GtkWidget* gtk_range2d_new()
{
  GtkRange2d* range2d;
  range2d = (GtkRange2d*)gtk_type_new(gtk_range2d_get_type());

  return GTK_WIDGET(range2d);
}

void
gtk_range2d_set_value(GtkRange2d* range2d,
		      double a, double b, double ra, double rb)
{
  range2d->parameter[0] = a;
  range2d->parameter[1] = b;
  range2d->parameter[2] = ra;
  range2d->parameter[3] = rb;

  set_parameter (NULL, range2d);
}

void
gtk_range2d_get_value(GtkRange2d* range2d,
		      double* a, double* b, double* ra, double* rb)
{
  *a = range2d->parameter[0];
  *b = range2d->parameter[1];
  *ra = range2d->parameter[2];
  *rb = range2d->parameter[3];
}

/***** private functions *****/

static void gtk_range2d_class_init(GtkRange2dClass* klass)
{
  GtkObjectClass* object_class;
  GtkWidgetClass* widget_class;

  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;

  parent_class = gtk_type_class(gtk_window_get_type());

  object_class->destroy = gtk_range2d_destroy;
  object_class->finalize = gtk_range2d_finalize;

  return;
}

static void gtk_range2d_init(GtkRange2d* range2d)
{
  gchar buf[32];

  range2d->parameter[0]=DEFAULT_A;
  range2d->parameter[1]=DEFAULT_B;
  range2d->parameter[2]=DEFAULT_RA;
  range2d->parameter[3]=DEFAULT_RB;

  range2d->move_dragged = FALSE;
  range2d->resize_dragged = FALSE;

  /* configuration of right side */
  range2d->frame = gtk_aspect_frame_new(NULL, 0.0, 0.0, 1.0, TRUE);
  gtk_frame_set_shadow_type(GTK_FRAME(range2d->frame), GTK_SHADOW_NONE);
  gtk_signal_connect (GTK_OBJECT(range2d->frame), "destroy",
		       GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
   gtk_container_border_width(GTK_CONTAINER(range2d->frame), 0);

   /* create a table which contains h/v rulers and canvas(GnomeCanvas). */
   range2d->table1 = gtk_table_new(2, 2, FALSE);
   gtk_container_add(GTK_CONTAINER(range2d->frame), range2d->table1);

   range2d->canvas = gnome_canvas_new();
   gnome_canvas_set_scroll_region(GNOME_CANVAS(range2d->canvas),
				  0, 0, 1.0, 1.0);
   gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(range2d->canvas),
				    (double) XSIZE);
   gtk_widget_set_usize(range2d->canvas, XSIZE, YSIZE);
   gtk_widget_set_events(range2d->canvas, GDK_BUTTON_PRESS_MASK);

   gtk_table_attach(GTK_TABLE(range2d->table1), range2d->canvas, 1, 2, 1, 2,
                     GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

   /* create a horizontal ruler */
   range2d->hrule = gtk_hruler_new();
   gtk_ruler_set_metric(GTK_RULER(range2d->hrule), GTK_PIXELS);

   /* set range and configure the signal link of auto-update */
   gtk_ruler_set_range(GTK_RULER(range2d->hrule), 0.0f, 1.0f, 0, 1.0f);
   gtk_signal_connect_object
     (GTK_OBJECT(range2d->canvas), "motion_notify_event",
       GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS
         (GTK_OBJECT (range2d->hrule) ->klass) ->motion_notify_event), 
	   GTK_OBJECT(range2d->hrule));

   /* put the horizontal ruler at upper-right position */
   gtk_table_attach(GTK_TABLE(range2d->table1), range2d->hrule, 1, 2, 0, 1,
                     GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);

   /* create a vertical ruler (and its configuration) */
   range2d->vrule = gtk_vruler_new();
   gtk_ruler_set_metric(GTK_RULER(range2d->vrule), GTK_PIXELS);
   gtk_ruler_set_range(GTK_RULER(range2d->vrule), 0.0f, 1.0f, 0.0f, 1.0f);
   gtk_signal_connect_object
    (GTK_OBJECT(range2d->canvas), "motion_notify_event",
      GTK_SIGNAL_FUNC(GTK_WIDGET_CLASS 
       (GTK_OBJECT (range2d->vrule) ->klass) ->motion_notify_event), 
         GTK_OBJECT(range2d->vrule));

   /* put the vertical ruler at lower-left position */
   gtk_table_attach(GTK_TABLE(range2d->table1), range2d->vrule, 0, 1, 1, 2,
                     GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);

   /* configure gnome canvas items */
   configure_gnome_canvas_items(range2d, GNOME_CANVAS(range2d->canvas));
  
   /* configuration of left side */
  range2d->entry_a = gtk_entry_new_with_max_length(5);
  gtk_widget_set_usize(GTK_WIDGET(range2d->entry_a), 40, 20);
  sprintf(buf, "%f", range2d->parameter[0]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_a), buf);
  range2d->entry_b = gtk_entry_new_with_max_length(5);
  gtk_widget_set_usize(GTK_WIDGET(range2d->entry_b), 40, 20);
  sprintf(buf, "%f", range2d->parameter[1]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_b), buf);
  range2d->entry_ra = gtk_entry_new_with_max_length(5);
  gtk_widget_set_usize(GTK_WIDGET(range2d->entry_ra), 40, 20);
  sprintf(buf, "%f", range2d->parameter[2]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_ra), buf);
  range2d->entry_rb = gtk_entry_new_with_max_length(5);
  gtk_widget_set_usize(GTK_WIDGET(range2d->entry_rb), 40, 20);
  sprintf(buf, "%f", range2d->parameter[3]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_rb), buf);

  range2d->label_a = gtk_label_new(" = ");
  range2d->label_b = gtk_label_new(" = ");
  range2d->label_ra = gtk_label_new("r = ");
  range2d->label_rb = gtk_label_new("r = ");

  range2d->button_save = gtk_button_new_with_label("save");
  gtk_widget_set_usize(GTK_WIDGET(range2d->button_save), 60, 20);
  range2d->button_load = gtk_button_new_with_label("load");
  gtk_widget_set_usize(GTK_WIDGET(range2d->button_load), 60, 20);

  range2d->hbox = gtk_hbox_new(FALSE, 0);
  range2d->table2 = gtk_table_new(2, 6, FALSE);

  gtk_container_add(GTK_CONTAINER(range2d), range2d->hbox);

  gtk_box_pack_start(GTK_BOX(range2d->hbox), range2d->table2,
		     FALSE, FALSE, 10);
  gtk_box_pack_start(GTK_BOX(range2d->hbox), range2d->frame, TRUE, TRUE, 0);

  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->label_a, 0, 1, 0, 1, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->entry_a, 1, 2, 0, 1, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2), 
		   range2d->label_b, 0, 1, 1, 2, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->entry_b, 1, 2, 1, 2, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->label_ra, 0, 1, 2, 3, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->entry_ra, 1, 2, 2, 3, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->label_rb, 0, 1, 3, 4, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->entry_rb, 1, 2, 3, 4, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2), 
		   range2d->button_save, 0, 2, 4, 5, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(range2d->table2),
		   range2d->button_load, 0, 2, 5, 6, 0, 0, 5, 5);

  /* connect signal */
  gtk_signal_connect(GTK_OBJECT(range2d->entry_a), "activate",
		     GTK_SIGNAL_FUNC(entry_activate), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_b), "activate",
		     GTK_SIGNAL_FUNC(entry_activate), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_ra), "activate",
		     GTK_SIGNAL_FUNC(entry_activate), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_rb), "activate",
		     GTK_SIGNAL_FUNC(entry_activate), range2d);

  gtk_signal_connect(GTK_OBJECT(range2d->entry_a), "activate",
		     GTK_SIGNAL_FUNC(set_parameter), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_b), "activate",
		     GTK_SIGNAL_FUNC(set_parameter), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_ra), "activate",
		     GTK_SIGNAL_FUNC(set_parameter), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->entry_rb), "activate",
		     GTK_SIGNAL_FUNC(set_parameter), range2d);

  gtk_signal_connect(GTK_OBJECT(range2d->canvas), "configure_event",
		     GTK_SIGNAL_FUNC(configured), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->canvas), "expose_event",
		     GTK_SIGNAL_FUNC(exposed), range2d);

  gtk_signal_connect(GTK_OBJECT(range2d->button_save), "clicked",
		     GTK_SIGNAL_FUNC(save_parameter), range2d);
  gtk_signal_connect(GTK_OBJECT(range2d->button_load), "clicked",
		     GTK_SIGNAL_FUNC(load_parameter), range2d);

  gtk_widget_show_all(range2d->hbox);
  return;
}
  
static void gtk_range2d_destroy(GtkObject* object)
{
  GtkRange2d* range2d;
  
  g_return_if_fail(object != NULL);
  g_return_if_fail(GTK_IS_RANGE2D(object));

  range2d = GTK_RANGE2D(object);

  if(range2d->hrule) gtk_widget_destroy(range2d->hrule);
  if(range2d->vrule) gtk_widget_destroy(range2d->vrule);
  if(range2d->entry_a) gtk_widget_destroy(range2d->entry_a);
  if(range2d->entry_b) gtk_widget_destroy(range2d->entry_b);
  if(range2d->entry_ra) gtk_widget_destroy(range2d->entry_ra);
  if(range2d->entry_rb) gtk_widget_destroy(range2d->entry_rb);
  if(range2d->label_a) gtk_widget_destroy(range2d->label_a);
  if(range2d->label_b) gtk_widget_destroy(range2d->label_b);
  if(range2d->label_ra) gtk_widget_destroy(range2d->label_ra);
  if(range2d->label_rb) gtk_widget_destroy(range2d->label_rb);
  if(range2d->button_save) gtk_widget_destroy(range2d->button_save);
  if(range2d->button_load) gtk_widget_destroy(range2d->button_load);  

  if(range2d->canvas) gtk_widget_destroy(range2d->canvas);
  if(range2d->table1) gtk_widget_destroy(range2d->table1);
  if(range2d->frame) gtk_widget_destroy(range2d->frame);
  if(range2d->table2) gtk_widget_destroy(range2d->table2);

  if(range2d->hbox) gtk_widget_destroy(range2d->hbox);


  if(GTK_OBJECT_CLASS(parent_class)->destroy)
    (* GTK_OBJECT_CLASS(parent_class)->destroy)(object);
}

static void gtk_range2d_finalize(GtkObject* object)
{
  g_return_if_fail(object != NULL);
  g_return_if_fail(GTK_IS_RANGE2D(object));

  (* GTK_OBJECT_CLASS(parent_class)->finalize)(object);
}

/* check whether item's new position (x+dx, y+d) is in the canvas */
static inline gboolean range_check_ok(GnomeCanvasItem *item, double dx, double dy)
{
  double x1, y1, x2, y2;
  gnome_canvas_item_get_bounds(item, &x1, &y1, &x2, &y2);

  /* return FALSE, if (x+dx, y+dy) is out of bounds */
  return !((x1+dx < 0) || (y1+dy < 0) || (x2+dx > 1.0) || (y2+dy > 1.0));
}

/* save parameters */
static
void
save_parameter(GtkObject* obj, gpointer arg)
{
  GtkRange2d* range2d = GTK_RANGE2D(arg);
  gnome_config_set_float("/gnodempa/color_params/a_mean", range2d->parameter[0]);
  gnome_config_set_float("/gnodempa/color_params/b_mean", range2d->parameter[1]);
  gnome_config_set_float("/gnodempa/color_params/a_coef", range2d->parameter[2]);
  gnome_config_set_float("/gnodempa/color_params/b_coef", range2d->parameter[3]);
  gnome_config_sync();
}

/* load parameters */
static
void
load_parameter(GtkObject* obj, gpointer arg)
{
  gchar buf[32];
  gboolean def;
  GtkRange2d* range2d = GTK_RANGE2D(arg);

  range2d->parameter[0] = gnome_config_get_float_with_default ("/gnodempa/color_params/a_mean=DEFAULT_A", &def);
  if(def) range2d->parameter[0] = DEFAULT_A;
  range2d->parameter[1] = gnome_config_get_float_with_default ("/gnodempa/color_params/b_mean=DEFAULT_B", &def);
  if(def) range2d->parameter[1] = DEFAULT_B;
  range2d->parameter[2] = gnome_config_get_float_with_default ("/gnodempa/color_params/a_coef=DEFAULT_RA", &def);
  if(def) range2d->parameter[2] = DEFAULT_RA;
  range2d->parameter[3] = gnome_config_get_float_with_default ("/gnodempa/color_params/b_coef=DEFAULT_RB", &def);
  if(def) range2d->parameter[3] = DEFAULT_RB;

  sprintf(buf, "%f", range2d->parameter[0]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_a), buf);
  sprintf(buf, "%f", range2d->parameter[1]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_b), buf);
  sprintf(buf, "%f", range2d->parameter[2]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_ra), buf);
  sprintf(buf, "%f", range2d->parameter[3]);
  gtk_entry_set_text(GTK_ENTRY(range2d->entry_rb), buf);
  set_parameter(NULL, range2d);
}

/* configure-event handler */
static
gint 
configured(GtkWidget* widget, GdkEvent* event, gpointer arg)
{
  GtkRange2d* range2d = GTK_RANGE2D(arg);

  gint w=range2d->hrule->allocation.width;
  gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(range2d->canvas), (double)w);

  return TRUE;
}

/* expose-event handler */
static
gint 
exposed(GtkWidget* widget, GdkEvent* event, gpointer arg)
{
  GtkRange2d* range2d = GTK_RANGE2D(arg);

  gint w=range2d->hrule->allocation.width;
  gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(range2d->canvas), (double)w);
  return TRUE;
}

/* move scope */
static gint scope_move_by_drag(GnomeCanvasItem *item, GdkEvent *event, gpointer arg)
{
  double x1, y1, x2, y2;
  
  GtkRange2d* range2d = GTK_RANGE2D(arg);

  switch(event->type)
    {
    case GDK_BUTTON_PRESS:
      range2d->move_dragged = TRUE;
      range2d->x0 = ((GdkEventMotion*) (event))->x;
      range2d->y0 = ((GdkEventMotion*) (event))->y;
      gnome_canvas_item_set(range2d->rect, "fill_color", HIGHLIGHT_COLOR, NULL);

      return TRUE;
      
    case GDK_MOTION_NOTIFY:
      if (!range2d->move_dragged) return FALSE;
      {
	double x = ((GdkEventMotion*) (event))->x;
	double y = ((GdkEventMotion*) (event))->y;
	double dx = x - range2d->x0;
	double dy = y - range2d->y0;
	gchar buf[32];

	if (! range_check_ok (range2d->rect, dx, dy)) return FALSE;
	/* gnome_canvas_item_move (item, x - x0, y - y0); */
	gnome_canvas_item_get_bounds(range2d->rect, &x1, &y1, &x2, &y2);
	gnome_canvas_item_set(range2d->rect, "x1", x1+dx, "y1", y1+dy,
			       "x2", x2+dx, "y2", y2+dy, NULL);
	gnome_canvas_item_set(range2d->ellipse, "x1", x1+dx, "y1", y1+dy,
			       "x2", x2+dx, "y2", y2+dy, NULL);
	gnome_canvas_item_set(range2d->pick11, 
	       "x1", x2+dx - PICKER_SIZE, "y1", y2+dy - PICKER_SIZE,
	       "x2", x2+dx, "y2", y2+dy, NULL);
	range2d->x0 = x;
	range2d->y0 = y;

	range2d->parameter[0] = (x1+x2)/2.0+dx;
	range2d->parameter[1] = (y1+y2)/2.0+dy;
	range2d->parameter[2] = (x2-x1)/2.0;
	range2d->parameter[3] = (y2-y1)/2.0;
	sprintf(buf, "%f", range2d->parameter[0]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_a), buf);
	sprintf(buf, "%f", range2d->parameter[1]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_b), buf);
	sprintf(buf, "%f", range2d->parameter[2]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_ra), buf);
	sprintf(buf, "%f", range2d->parameter[3]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_rb), buf);
      }
      return TRUE;

    case GDK_BUTTON_RELEASE:
      range2d->move_dragged = FALSE;
      gnome_canvas_item_set(range2d->rect, "fill_color", "", NULL);
      gnome_canvas_item_set(range2d->ellipse, "fill_color", DEFAULT_COLOR, NULL);

      return TRUE;
    }
  return FALSE;
}

/* resize scope */
static gint scope_resize_by_drag(GnomeCanvasItem *item, GdkEvent *event, gpointer arg)
{
  double x1, y1, x2, y2;

  GtkRange2d* range2d = GTK_RANGE2D(arg);

  gnome_canvas_item_get_bounds(range2d->rect, &x1, &y1, &x2, &y2);

  switch (event->type)
    {
    case GDK_BUTTON_PRESS:
      range2d->resize_dragged = TRUE;
      range2d->x0 = ((GdkEventMotion*) (event))->x;
      range2d->y0 = ((GdkEventMotion*) (event))->y;
      return TRUE;
      
    case GDK_MOTION_NOTIFY:
      if (!range2d->resize_dragged) return FALSE;
      {
	double x = ((GdkEventMotion*) (event))->x;
	double y = ((GdkEventMotion*) (event))->y;
	double dx = x - range2d->x0;
	double dy = y - range2d->y0;
	gchar buf[32];

	if (! range_check_ok (range2d->rect, dx, dy)) return FALSE;

	gnome_canvas_item_get_bounds(range2d->rect, &x1, &y1, &x2, &y2);
	if ((x2+x-range2d->x0 - x1) < MINIMUM_SCOPE_SIZE) return FALSE;
	if ((y2+y-range2d->y0 - y1) < MINIMUM_SCOPE_SIZE) return FALSE;

	gnome_canvas_item_set(range2d->rect, "x2", x2 + x - range2d->x0,
			             "y2", y2 + y - range2d->y0, NULL);
	gnome_canvas_item_set(range2d->ellipse, "x2", x2 + x - range2d->x0,
			             "y2", y2 + y - range2d->y0, NULL);
	gnome_canvas_item_get_bounds(range2d->rect, &x1, &y1, &x2, &y2);
	gnome_canvas_item_set(range2d->pick11, 
	       "x1", x2 - PICKER_SIZE, "y1", y2 - PICKER_SIZE,
	       "x2", x2, "y2", y2, NULL);
	range2d->x0 = x;
	range2d->y0 = y;

	range2d->parameter[0] = (x1+x2+dx)/2.0;
	range2d->parameter[1] = (y1+y2+dy)/2.0;
	range2d->parameter[2] = (x2+dx-x1)/2.0;
	range2d->parameter[3] = (y2+dy-y1)/2.0;
	sprintf(buf, "%f", range2d->parameter[0]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_a), buf);
	sprintf(buf, "%f", range2d->parameter[1]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_b), buf);
	sprintf(buf, "%f", range2d->parameter[2]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_ra), buf);
	sprintf(buf, "%f", range2d->parameter[3]);
	gtk_entry_set_text(GTK_ENTRY(range2d->entry_rb), buf);
      }
      return TRUE;

    case GDK_BUTTON_RELEASE:
      range2d->resize_dragged = FALSE;
      gnome_canvas_item_set(range2d->pick11, 
	     "x1", x2 - PICKER_SIZE, "y1", y2 - PICKER_SIZE,
	     "x2", x2, "y2", y2, NULL);
      return TRUE;
    }
  return FALSE;
}

/* set parameter */
static gint set_parameter(GtkWidget* widget, gpointer arg)
{
  double x1, x2, y1, y2;
  double a, b, ra, rb;
  GtkRange2d* range2d = GTK_RANGE2D(arg);

  a = range2d->parameter[0];
  b = range2d->parameter[1];
  ra = range2d->parameter[2];
  rb = range2d->parameter[3];
  x1 = a-ra; y1 = b-rb; x2 = a+ra; y2 = b+rb;

  gnome_canvas_item_set(range2d->rect, 
			 "x1", x1, "y1", y1,
			 "x2", x2, "y2", y2, NULL);
  gnome_canvas_item_set(range2d->ellipse, 
			 "x1", x1, "y1", y1,
			 "x2", x2, "y2", y2, NULL);
  gnome_canvas_item_set(range2d->pick11, 
			 "x1", x2 - PICKER_SIZE, "y1", y2 - PICKER_SIZE,
			 "x2", x2, "y2", y2, NULL);
  return TRUE;
}

/* configure canvas items */
static
void
configure_gnome_canvas_items(GtkRange2d* range2d, GnomeCanvas* canvas)
{
  range2d->root = gnome_canvas_root(GNOME_CANVAS(canvas));

  range2d->rect = gnome_canvas_item_new(range2d->root, 
				gnome_canvas_rect_get_type(),
				"x1", SCOPE_DEFAULT_X1, 
				"y1", SCOPE_DEFAULT_Y1, 
				"x2", SCOPE_DEFAULT_X2,
				"y2", SCOPE_DEFAULT_Y2,
				"outline_color", DEFAULT_BORDER_COLOR, 
				NULL);

  range2d->ellipse = gnome_canvas_item_new(range2d->root, 
				gnome_canvas_ellipse_get_type(),
				"x1", SCOPE_DEFAULT_X1,
				"y1", SCOPE_DEFAULT_Y1,
				"x2", SCOPE_DEFAULT_X2,
				"y2", SCOPE_DEFAULT_Y2,
				"fill_color", DEFAULT_COLOR, 
				"outline_color", DEFAULT_BORDER_COLOR, 
				NULL);

  range2d->pick11 = gnome_canvas_item_new(range2d->root, 
				gnome_canvas_rect_get_type(),
				"x1", SCOPE_DEFAULT_X2 - PICKER_SIZE,
				"y1", SCOPE_DEFAULT_Y2 - PICKER_SIZE,
				"x2", SCOPE_DEFAULT_X2, "y2", SCOPE_DEFAULT_Y2,
				"fill_color", DEFAULT_BORDER_COLOR,
				 NULL);

  gtk_signal_connect(GTK_OBJECT(range2d->root), "event",
		      GTK_SIGNAL_FUNC(scope_move_by_drag), range2d);

  gtk_signal_connect(GTK_OBJECT(range2d->pick11), "event",
		      GTK_SIGNAL_FUNC(scope_resize_by_drag), range2d);
}

/* entry_activate event handler */
static void entry_activate(GtkWidget* widget, gpointer arg)
{
  int i;
  double value[4];
  gchar buf[32];

  GtkRange2d* range2d = GTK_RANGE2D(arg);

  value[0] = atof(gtk_entry_get_text(GTK_ENTRY(range2d->entry_a)));
  value[1] = atof(gtk_entry_get_text(GTK_ENTRY(range2d->entry_b)));
  value[2] = atof(gtk_entry_get_text(GTK_ENTRY(range2d->entry_ra)));
  value[3] = atof(gtk_entry_get_text(GTK_ENTRY(range2d->entry_rb)));
  if (((value[0]-value[2])>=0.0) && ((value[1]-value[3])>=0.0)
     && ((value[0]+value[2])<=1.0) && ((value[1]+value[3])<=1.0)) {
    for (i=0; i<4; i++) range2d->parameter[i] = value[i];
  } else {
      sprintf(buf, "%f", range2d->parameter[0]);
      gtk_entry_set_text(GTK_ENTRY(range2d->entry_a), buf);
      sprintf(buf, "%f", range2d->parameter[1]);
      gtk_entry_set_text(GTK_ENTRY(range2d->entry_b), buf);
      sprintf(buf, "%f", range2d->parameter[2]);
      gtk_entry_set_text(GTK_ENTRY(range2d->entry_ra), buf);
      sprintf(buf, "%f", range2d->parameter[3]);
      gtk_entry_set_text(GTK_ENTRY(range2d->entry_rb), buf);
  }
}
